/*
 * Copyright (C) 1995, 1996 Systemics Ltd (http://www.systemics.com/)
 * All rights reserved.
 */
 
package cryptix.pgp;

import java.io.*;
import cryptix.mime.Base64;

public final class Armoury
{

	//
	// For reading
	//
	int linecount = 0;
	InputStream is;
	private final static int WAITING				= 0;
	private final static int PACKET					= 1;
	private final static int PACKET_START			= 2;
	private final static int SIGNED_DATA			= 3;
	private final static int SIGNED_DATA_FIRSTLINE	= 4;
	private final static int SIGNED_DATA_START		= 5;


	public Armoury(InputStream is)
	{ 
		this.is = is;
	} 

	// Perhaps be a bit more specific than 'Object'?
	public Object
	read()
		throws IOException, InvalidChecksumException
	{
		int state = WAITING;
		DataInputStream dis = (is instanceof DataInputStream)
							? (DataInputStream)is : new DataInputStream(is);

		String data_str = "";
		String packet_str = "";
		String s;
		boolean signed_data = false;

		while ((s = dis.readLine()) != null)
		{
			linecount++;

			switch(state)
			{
			case WAITING:
				if (s.startsWith("-----BEGIN"))
				{
					if (s.startsWith("-----BEGIN PGP SIGNED MESSAGE"))
					{
						state = SIGNED_DATA_START;
						break;
					}
					state = PACKET_START;
					break;
				}
				break;

			// Skip comments and the like
			case SIGNED_DATA_START:
				signed_data = true;
				if (s.length() == 0)
					state = SIGNED_DATA_FIRSTLINE;
				break;

			// Skip comments and the like
			case PACKET_START:
				if (s.length() == 0)
					state = PACKET;
				break;

			case SIGNED_DATA_FIRSTLINE:
			case SIGNED_DATA:
				if (s.startsWith("-----BEGIN PGP SIGNATURE"))
				{
					state = PACKET_START;
					break;
				}

				if ((s.length() > 2) && (s.charAt(0) == '-' ))
					s = s.substring(2);

				if (state == SIGNED_DATA_FIRSTLINE)
					state = SIGNED_DATA;
				else
					data_str += "\r\n";
				data_str += s;

				break;

			case PACKET:
				if (s.startsWith("="))
				{
					int csum = decodeChecksum(s);

					s = dis.readLine();
					if (!s.startsWith("-----END"))
						// Change this to a real exception
						throw new RuntimeException("-----END ... expected after checksum");

					// Get the data and packet as byte arrays
					byte[] packet = Base64.decode(packet_str);

					// Verify checksum
					if (csum != CRC.checksum(packet))
						throw new RuntimeException("invalid checksum");

					// Create appropriate object
					Packet pkt_obj = PacketFactory.load(packet);

					if (signed_data)
					{
						int len = data_str.length();
						byte[] data = new byte[len];
						data_str.getBytes(0, len, data, 0);

						if (pkt_obj instanceof Signature)
							return new SignedData(data, (Signature)pkt_obj);
						// Otherwise this isn;t a signature
						throw new RuntimeException("Invalid signature packet");
					}
					return pkt_obj;
				}

				packet_str += s;

				break;

			default:
				throw new RuntimeException("state machine error");
			}
		}
		// Perhaps we should throw an exception if the state is not WAITING?
		return null;	// EOF
	}
 
	public static String
	encodeChecksum(byte[] buf)
	{
		byte checksum[] = new byte[3];
		int csum = CRC.checksum(buf);
		checksum[0] = (byte)(csum >>> 16);
		checksum[1] = (byte)(csum >>> 8);
		checksum[2] = (byte)csum;
		return Base64.encode(checksum); 
	}

	public static int
	decodeChecksum(String s)
		throws InvalidChecksumException
	{
		if (s.length() != 5)
		{
		}

		byte[] asc_sum = new byte[4];
		s.getBytes(1, 5, asc_sum, 0);
		byte[] binsum = Base64.decode(asc_sum);
		int sum = ((((int)binsum[0] & 0xFF) << 16)
			| (((int)binsum[1] & 0xFF) << 8)
			| ((int)binsum[2]) & 0xFF);
		return sum;
	}


	public static String
	encodeWithChecksum(byte[] buf)
	{
		return Base64.encode(buf)+"="+encodeChecksum(buf); 
	}

	public static void
	message(OutputStream os, String name, byte buf[])
	{
		PrintStream out = new PrintStream (os);
		out.println("-----BEGIN PGP "+name+"-----");
		out.println("Version: Cryptix: 0.02");
		out.println();
		out.print(encodeWithChecksum(buf));
		out.println("-----END PGP "+name+"-----");
	}

	public static void
	message(OutputStream os, Packet pkt)
		throws IOException
	{
		PrintStream out = new PrintStream (os);
		message(out, name(pkt), pkt.save());
	}

	public static String
	name(Packet pkt)
	{
		if ( pkt instanceof Signature )
			return "SIGNATURE";
		// else if ( pkt instanceof SecretKeyRingEntry )
		// 	return "SECRET KEY BLOCK";
		// else if ( pkt instanceof PublicKeyRingEntry )
		// 	return "PUBLIC KEY BLOCK";

		throw new IllegalArgumentException("PGP Packet type "+pkt.getClass().getName()+" not valid for this method");
	}

//	readUntilBegin(InputStream dis)
//	{
//		String line = dis.readLine();
//		if (line.startsWith( "-----BEGIN PGP" ))
//		{
//			line = dis.readLine();
//			line.trim();
//			// Length should now be zero
//		}
//	}


// readUntilBegin
// readPacket


	public static void
	main(String[] argv)
		throws IOException, InvalidChecksumException
	{
		FileInputStream fis = new FileInputStream(argv[0]);
		Armoury armoury = new Armoury(fis);
		Object pkt = armoury.read();
		cryptix.crypt.rsa.PublicKey key = ((cryptix.pgp.PublicKeyCertificate)pkt).publicKey();
		System.out.println(pkt.toString());

		fis = new FileInputStream(argv[1]);
		armoury = new Armoury(fis);
		pkt = armoury.read();
		if (pkt instanceof SignedData)
		{
			SignedData sd = (SignedData)pkt;
			System.out.println("Got some signed data:");
			System.out.print(sd.dataAsString());
			System.out.println("Key id = "+sd.keyId());
			System.out.println("Signature "+(sd.verify(key) ? "OK" : "Not OK"));

// FileOutputStream fos = new FileOutputStream(argv[2]);
// fos.write(sd.data);
// fos.write(sd.signature.extra);
// fos.close();
		}
		else
		{
			System.out.println(pkt.toString());
			System.out.println("Got an unknown packet");
		}
	}

}
