/*
** CONNECT.C - This is the ODBC sample driver code for
** allocation and connection.
**
**	This code is furnished on an as-is basis as part of the ODBC SDK and is
**	intended for example purposes only.
**
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

//	-	-	-	-	-	-	-	-	-

#include "gorta.h"
#include "states.h"
#include "ctl3d.h"
#include "odbcinst.h"					// ODBC installer prototypes

static char far * ODBC_INI = "odbc.ini";

//	-	-	-	-	-	-	-	-	-
// Winmsql include
#include <msql.h>

//	-	-	-	-	-	-	-	-	-

//	CONNECT.C
//
//	SQLC connection functions.

//	-	-	-	-	-	-	-	-	-

//	Allocate an environment (ENV) block.

RETCODE SQL_API SQLAllocEnv(
	HENV FAR *phenv)
{
	*phenv = (HENV FAR *)NULL;
	return SQL_SUCCESS;
}

//	-	-	-	-	-	-	-	-	-

//	Allocate a DBC block.

RETCODE SQL_API SQLAllocConnect(
	HENV	 henv,
	HDBC FAR *phdbc)
{
	HGLOBAL	hdbc;
	LPDBC lpdbc;

    hdbc = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT, sizeof (DBC));
	if (!hdbc || (*phdbc = (HDBC)GlobalLock (hdbc)) == SQL_NULL_HDBC)
	{
		GlobalFree (hdbc);	//	Free it if lock fails
		return SQL_ERROR;
	}
	lpdbc = *phdbc;
	lpdbc->bConnected = FALSE;
	lpdbc->iSocketFd = 0;
	lpdbc->lpei = NULL;
	lpdbc->pcDSN = NULL;
	return SQL_SUCCESS;
}

//	-	-	-	-	-	-	-	-	-
/*
 * This is the function which initiates the connection to the 
 * MSQL database.
 *
 * Pull the name of the server from the odbc.ini file entry (szDSN).
 * Pass this value to the msqlConnect() function which in turn uses
 * this string to look up msql.ini.
 */
RETCODE SQL_API SQLConnect(
	HDBC	  hdbc,
	UCHAR FAR *szDSN,
	SWORD	  cbDSN,
	UCHAR FAR *szUID,
	SWORD	  cbUID,
	UCHAR FAR *szAuthStr,
	SWORD	  cbAuthStr)
{
	LPDBC lpdbc;
	RETCODE retcode = SQL_SUCCESS;
	char szErrorMsg[160];
	LPEI lpeiTmp;
	
	lpdbc = hdbc;
	
	while (lpdbc->lpei)
	{
		lpeiTmp = lpdbc->lpei;
		lpdbc->lpei = lpeiTmp->pNext;
		free(lpeiTmp);
	}

	/*
	 * Make the server connection.
	 */    
	SQLGetPrivateProfileString(szDSN, "Server", "Server", 
		lpdbc->caServerName, sizeof(lpdbc->caServerName), ODBC_INI);

	lpdbc->iSocketFd = msqlConnect(lpdbc->caServerName);
	if (lpdbc->iSocketFd < 0)
	{
		lpdbc->bConnected = FALSE;
		retcode = SQL_ERROR;
		lpeiTmp = malloc(sizeof(EI));
		if (!lpeiTmp)
		{
			retcode = SQL_ERROR;
			goto Exit;
		}
		lpeiTmp->pNext = lpdbc->lpei;
		lpeiTmp->lErrorCode=-1;
		strncpy(lpeiTmp->szSqlState, MSQL_UNABLE_TO_CONNECT, 5);
		msqlErrorText(szErrorMsg);						/* MSQL Extension */
		sprintf(lpeiTmp->szErrorMessage, "[MSQLODBC][MSQL]%.*s", 160, szErrorMsg);
		lpdbc->lpei=lpeiTmp;
		goto Exit;
	}
	
	SQLGetPrivateProfileString(szDSN, "Database", "", 
		lpdbc->caDBName, sizeof(lpdbc->caDBName), ODBC_INI);
	if (!lpdbc->caDBName[0])
	{
		lpdbc->bConnected = FALSE;
		retcode = SQL_ERROR;
		lpeiTmp = malloc(sizeof(EI));
		if (!lpeiTmp)
		{
			retcode = SQL_ERROR;
			goto Exit;
		}
		lpeiTmp->pNext = lpdbc->lpei;
		lpeiTmp->lErrorCode=-1;
		strncpy(lpeiTmp->szSqlState, MSQL_UNABLE_TO_CONNECT, 5);
		sprintf(lpeiTmp->szErrorMessage, "[MSQLODBC]No database name in the ODBC.INI file");
		lpdbc->lpei=lpeiTmp;
		msqlClose(lpdbc->iSocketFd);
		goto Exit;
	}
	
	if (msqlSelectDB(lpdbc->iSocketFd,lpdbc->caDBName) != 0)
	{
		lpdbc->bConnected = FALSE;
		retcode = SQL_ERROR;
		lpeiTmp = malloc(sizeof(EI));
		if (!lpeiTmp)
		{
			retcode = SQL_ERROR;
			goto Exit;
		}
		lpeiTmp->pNext = lpdbc->lpei;
		lpeiTmp->lErrorCode=-1;
		strncpy(lpeiTmp->szSqlState, MSQL_UNABLE_TO_CONNECT, 5);
		sprintf(lpeiTmp->szErrorMessage, 
			"[MSQLODBC]Unable to connect to database <%s>", lpdbc->caDBName);
		lpdbc->lpei=lpeiTmp;
		msqlClose(lpdbc->iSocketFd);
		goto Exit;
	}
    
    
    lpdbc->pcDSN = strdup(szDSN);
	lpdbc->bConnected = TRUE;

Exit:
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

//	Sample Dialog Proc for SQLDriverConnect.
//	This has to be exported or subtle, yet very rude, things will happen.

BOOL FAR PASCAL FDriverConnectProc(
	HWND	hdlg,
	WORD	wMsg,
	WPARAM  wParam,
	LPARAM  lParam)
{
	switch (wMsg) {
	case WM_INITDIALOG:
		Ctl3dRegister (s_hModule);
#ifdef WIN32
		Ctl3dSubclassDlg(hdlg, CTL3D_ALL);
#else
		Ctl3dSubclassDlgEx(hdlg, CTL3D_ALL);
#endif
		return TRUE;

#ifdef WIN32
	case WM_CTLCOLORBTN:
	case WM_CTLCOLORDLG:
	case WM_CTLCOLOREDIT:
	case WM_CTLCOLORLISTBOX:
	case WM_CTLCOLORMSGBOX:
	case WM_CTLCOLORSCROLLBAR:	
	case WM_CTLCOLORSTATIC:
		return (BOOL)Ctl3dCtlColorEx(wMsg, wParam, lParam);

	case WM_SETTEXT:
	case WM_NCPAINT:
	case WM_NCACTIVATE:
		SetWindowLong(hdlg, DWL_MSGRESULT,
			Ctl3dDlgFramePaint(hdlg, wMsg, wParam, lParam));
		return TRUE;
#endif

	case WM_SYSCOLORCHANGE:
		return Ctl3dColorChange();

	case WM_COMMAND:
		switch (GET_WM_COMMAND_ID(wParam, lParam)) {
		case IDOK:

		case IDCANCEL:
			Ctl3dUnregister (s_hModule);
			EndDialog(hdlg, GET_WM_COMMAND_ID(wParam, lParam) == IDOK);
			return TRUE;
		}
	}
	return FALSE;
}

//	-	-	-	-	-	-	-	-	-

//	This function as its "normal" behavior is supposed to bring up a
//	dialog box if it isn't given enough information via "szConnStrIn".  If
//	it is given enough information, it's supposed to use "szConnStrIn" to
//	establish a database connection.  In either case, it returns a
//	string to the user that is the string that was eventually used to
//	establish the connection.

RETCODE SQL_API SQLDriverConnect(
	HDBC	  hdbc,
	HWND	  hwnd,
	UCHAR FAR *szConnStrIn,
	SWORD 	  cbConnStrIn,
	UCHAR FAR *szConnStrOut,
	SWORD	  cbConnStrOutMax,
	SWORD FAR *pcbConnStrOut,
	UWORD	  fDriverCompletion)
{
	short	iRet;
	BOOL	fPrompt = FALSE;
	char    *pcAlloc, *pcParse;
	RETCODE retcode = SQL_SUCCESS;
	LPDBC   lpdbc;

	lpdbc = hdbc;

	if ((szConnStrIn == NULL) || (!cbConnStrIn) ||
		((cbConnStrIn == SQL_NTS) && (!szConnStrIn[0])))
		fPrompt = TRUE;
	else {
		//	Check connection string for completeness
		//	Create a null terminated buffer
		if (cbConnStrIn == SQL_NTS)
		{
			pcAlloc = strdup(szConnStrIn);
		}
		else
		{
			assert(cbConnStrIn > 0);
			pcAlloc = malloc(cbConnStrIn + 1);
			assert(pcAlloc);
			memcpy(pcAlloc, szConnStrIn, cbConnStrIn);
			pcAlloc[cbConnStrIn] = '\0';
		}			
		//
		//	The only item MSQL requires is the DNS String. Parse
		//	through the connection string and see if it has been
		//	passed.
		if (pcParse = strstr(pcAlloc, "DSN="))
		{
			pcParse = strtok(pcParse + 4, ";");
			if (pcParse)
			{
				retcode = SQLConnect(hdbc, pcParse, SQL_NTS, NULL, 0, NULL, 0);
				if (retcode == SQL_SUCCESS)
				{
					if (cbConnStrOutMax > 0)
					{
						*pcbConnStrOut = sprintf(szConnStrOut, "DSN=%.*s", 
							cbConnStrOutMax - 4, pcParse);
					}
				}
				free(pcAlloc);
				goto Exit;
			}
		}
		free(pcParse);
		if (fDriverCompletion == SQL_DRIVER_COMPLETE ||
			fDriverCompletion == SQL_DRIVER_PROMPT)
			fPrompt = TRUE;
	}
	if (fPrompt) {
		//	This section is a RELIC of the sample program. It has
		// 	not been implemented for MSQL
		//
		//	It is not necessary to call "MakeProcInstance" if you
		//	generate a dialog box from a DLL.
		//
		iRet = DialogBox(s_hModule, MAKEINTRESOURCE(EDRIVERCONNECT), hwnd,
			FDriverConnectProc);
		if ((!iRet) || (iRet == -1))
			return SQL_NO_DATA_FOUND;
	}
Exit:
	return(retcode);
}

//	-	-	-	-	-	-	-	-	-

RETCODE SQL_API SQLBrowseConnect(
	HDBC	  hdbc,
	UCHAR FAR *szConnStrIn,
	SWORD	  cbConnStrIn,
	UCHAR FAR *szConnStrOut,
	SWORD	  cbConnStrOutMax,
	SWORD FAR *pcbConnStrOut)
{
	return SQL_SUCCESS;
}

//	-	-	-	-	-	-	-	-	-

RETCODE SQL_API SQLDisconnect(
	HDBC	  hdbc)
{
	LPDBC lpdbc;
	RETCODE retcode = SQL_SUCCESS;
	
	lpdbc = hdbc;
    
    /* TBD _ Ensure that the connection is valid before passing
	 * to msql API.
	 */
	msqlClose(lpdbc->iSocketFd);
	lpdbc->iSocketFd = -1;
	lpdbc->bConnected = FALSE;
	if (lpdbc->pcDSN)
	{
		free(lpdbc->pcDSN);
		lpdbc->pcDSN = NULL;
	}

Exit:
	return(retcode);
}
//	-	-	-	-	-	-	-	-	-

RETCODE SQL_API SQLFreeConnect(
	HDBC	  hdbc)
{
	GlobalUnlock (GlobalPtrHandle(hdbc));
	GlobalFree (GlobalPtrHandle(hdbc));
	return SQL_SUCCESS;
}

//	-	-	-	-	-	-	-	-	-

RETCODE SQL_API SQLFreeEnv(
	HENV	  henv)
{

	return SQL_SUCCESS;
}
