//
//  XTFontManagerTests.m
//  TadsTerp
//
//  Created by Rune Berg on 06/09/14.
//  Copyright (c) 2014 Rune Berg. All rights reserved.
//

#import <XCTest/XCTest.h>
#import "XTFontManager.h"


@interface XTFontManagerTests : XCTestCase

@end


@implementation XTFontManagerTests

static XTFontManager *fontManager;
static NSArray *fontNames;
static NSNumber *pointSize;
static NSNumber *htmlSize;
static NSFont *font;

- (void)setUp
{
    [super setUp];
    fontManager = [XTFontManager new];
	fontNames = nil;
	pointSize = nil;
	htmlSize = nil;
	font = nil;
}

- (void)tearDown
{
    [super tearDown];
}

//------  simplest case  --------

- (void)testSimplestCase
{
	fontNames = [NSArray arrayWithObject:@"Arial"];
	pointSize = [NSNumber numberWithFloat:12.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Arial", [font familyName]);
	XCTAssertEqual(12.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

//------  bold and italics  --------

- (void)testBoldConcreteFont
{
	fontNames = [NSArray arrayWithObject:@"Arial"];
	pointSize = [NSNumber numberWithFloat:13.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:YES italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Arial", [font familyName]);
	XCTAssertEqual(13.0, [font pointSize]);
	XCTAssertTrue([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testItalicConcreteFont
{
	fontNames = [NSArray arrayWithObject:@"Arial"];
	pointSize = [NSNumber numberWithFloat:14.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:YES small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Arial", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertTrue([self fontIsItalic]);
}

- (void)testBoldItalicConcreteFont
{
	fontNames = [NSArray arrayWithObject:@"Arial"];
	pointSize = [NSNumber numberWithFloat:16.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:YES italics:YES small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Arial", [font familyName]);
	XCTAssertEqual(16.0, [font pointSize]);
	XCTAssertTrue([self fontIsBold]);
	XCTAssertTrue([self fontIsItalic]);
}

//------  concrete font names  --------

- (void)testConcreteFontNames
{
	fontNames = [NSArray arrayWithObject:@"Courier"];
	pointSize = [NSNumber numberWithFloat:12.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Courier", [font familyName]);
	XCTAssertEqual(12.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

//------  parameterized font names  --------

- (void)testParameterizedFont_XTADS_Default
{
	fontNames = [NSArray arrayWithObject:@"XTADS-Default"];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);

	fontNames = [NSArray arrayWithObject:@"XTADS-Default"];
	pointSize = [NSNumber numberWithFloat:18.0f];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:YES italics:YES small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(18.0f, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testParameterizedFont_XTADS_FixedWidth
{
	fontNames = [NSArray arrayWithObject:@"XTADS-FixedWidth"];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Consolas", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testParameterizedFont_TADS_Serif
{
	fontNames = [NSArray arrayWithObject:@"TADS-Serif"];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Georgia", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testParameterizedFont_TADS_Sans
{
	fontNames = [NSArray arrayWithObject:@"TADS-Sans"];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testParameterizedFont_TADS_Script
{
	fontNames = [NSArray arrayWithObject:@"TADS-Script"];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Apple Chancery", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertTrue([self fontIsItalic]);
}

- (void)testParameterizedFont_TADS_Typewriter
{
	fontNames = [NSArray arrayWithObject:@"TADS-Typewriter"];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Consolas", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testParameterizedFont_TADS_Input
{
	fontNames = [NSArray arrayWithObject:@"TADS-Input"];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}
//------  leading / trailing spaces in font names  --------

- (void)testLeadingOrTrailingSpacesInFontName
{
	fontNames = [NSArray arrayWithObject:@" Courier"];
	pointSize = [NSNumber numberWithFloat:12.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Courier", [font familyName]);
	XCTAssertEqual(12.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);

	fontNames = [NSArray arrayWithObject:@"Courier  "];
	pointSize = [NSNumber numberWithFloat:12.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Courier", [font familyName]);
	XCTAssertEqual(12.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
	
	fontNames = [NSArray arrayWithObject:@"  Courier "];
	pointSize = [NSNumber numberWithFloat:14.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Courier", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

//------  case insensitive font names  --------

- (void)testCaseInsensitiveConcreteFontName
{
	fontNames = [NSArray arrayWithObject:@"COURIER"];
	pointSize = [NSNumber numberWithFloat:12.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Courier", [font familyName]);
	XCTAssertEqual(12.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);

	fontNames = [NSArray arrayWithObject:@"coURier"];
	pointSize = [NSNumber numberWithFloat:12.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Courier", [font familyName]);
	XCTAssertEqual(12.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testCaseInsensitiveParameterizedFontName
{
	fontNames = [NSArray arrayWithObject:@"tads-SERIF"];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Georgia", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);

	fontNames = [NSArray arrayWithObject:@"Tads-SerIF"];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Georgia", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

//------  list of font names  --------

- (void)testMoreThanOneGoodFontName
{
	fontNames = [NSArray arrayWithObjects:@"Arial", @"Courier", nil];
	pointSize = [NSNumber numberWithFloat:12.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Arial", [font familyName]);
	XCTAssertEqual(12.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);

	fontNames = [NSArray arrayWithObjects:@"Courier", @"Monaco", nil];
	pointSize = [NSNumber numberWithFloat:12.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Courier", [font familyName]);
	XCTAssertEqual(12.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testUnknownThenKnownFontName
{
	fontNames = [NSArray arrayWithObjects:@"nogoodatall", @"Courier", nil];
	pointSize = [NSNumber numberWithFloat:12.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Courier", [font familyName]);
	XCTAssertEqual(12.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);

	fontNames = [NSArray arrayWithObjects:@"nogoodatall", @"TADS-Typewriter", nil];
	pointSize = [NSNumber numberWithFloat:12.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Consolas", [font familyName]);
	XCTAssertEqual(12.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
	
	fontNames = [NSArray arrayWithObjects:@"nogoodatall", @"XTADS-Default", nil];
	pointSize = [NSNumber numberWithFloat:16.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:YES italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(16.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

//------  empty / nil / with-leading-or-trailing-spaces font names  --------

- (void)testUnknownFontName
{
	fontNames = [NSArray arrayWithObjects:@"nogoodatall", nil];
	pointSize = nil;
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);

	fontNames = [NSArray arrayWithObjects:@"nogoodatall", nil];
	pointSize = [NSNumber numberWithFloat:18.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(18.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testMissingFontName
{
	fontNames = [NSArray array];
	pointSize = [NSNumber numberWithFloat:18.0];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testEmptyFontName
{
	fontNames = [NSArray arrayWithObjects:@"", nil];
	pointSize = [NSNumber numberWithFloat:18.0];
	font = [fontManager getFontWithName:fontNames pointSize:pointSize htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(18.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);

	fontNames = [NSArray arrayWithObjects:@"  ", nil];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

- (void)testNilFontName
{
	fontNames = [NSArray arrayWithObjects:nil];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);

	fontNames = [NSArray arrayWithObjects:nil, nil];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:nil bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Helvetica", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

//------  pointSize vs htmlSize  --------

- (void)testHtmlSizeSimpleCase
{
	fontNames = [NSArray arrayWithObject:@"Arial"];
	htmlSize = [NSNumber numberWithUnsignedInteger:3];
	font = [fontManager getFontWithName:fontNames pointSize:nil htmlSize:htmlSize bold:NO italics:NO small:NO big:NO];
	XCTAssertNotNil(font);
	XCTAssertEqualObjects(@"Arial", [font familyName]);
	XCTAssertEqual(14.0, [font pointSize]);
	XCTAssertFalse([self fontIsBold]);
	XCTAssertFalse([self fontIsItalic]);
}

//TODO more tests...
// 1..7 for both
//		param'd f.n.
//		concrete f.n.
// both pt size and html size

//------  too small / large font sizes  --------

//TODO

//-----------

- (BOOL)fontIsBold
{
	NSFontSymbolicTraits symTraits = [[font fontDescriptor] symbolicTraits];
	BOOL res = (symTraits & NSFontBoldTrait);
	return res;
}

- (BOOL)fontIsItalic
{
	NSFontSymbolicTraits symTraits = [[font fontDescriptor] symbolicTraits];
	BOOL res = (symTraits & NSFontItalicTrait);
	return res;
}

@end
