/* FIRST THINGS FIRST - functions.t
/*  Copyright (c) 1996, 1997, 1999, 2000 by J. Robinson Wheeler. All Rights Reserved. */


/*					 STANDARD GAME FUNCTIONS				*/


/*	The scoreRank() function displays how well the player is doing.
 *	In addition to displaying the numerical score and number of turns
 *	played so far, we'll classify the score with a rank.
 *
 *	Note that "global.maxscore" defines the maximum number of points
 *	possible in the game.
 */
replace scoreRank: function
{
	local s;
	
	s := global.score;
	
	"In a total of "; say( global.turnsofar );
	" turns, you have achieved a score of ";
	say( s ); " points out of a possible "; say( global.maxscore );
	", which means you are ";
	
	if ( s < 20 ) "barely getting started. ";
	else if ( s < 40 ) "starting to get somewhere. ";
	else if ( s < 65 ) "a budding adventurer. ";
	else if ( s < 90 ) "an adventurer. ";
	else if ( s < 110 ) "an experienced adventurer. ";
	else if ( s < 130 ) "a first class adventurer. ";
	else if ( s < 145 ) "a master adventurer. ";
	else if ( s < global.maxscore ) "almost there -- keep going! ";
	else "at last a raving success!";
}

/*	 The terminate() function is called just before the game ends. We
 *	just display a good-bye message.
 */
replace terminate: function
{
	"\bThank you for playing \(First Things First\).\n";
}

/*
 *	 The goToSleep() function is called when the player types the command
 *	 "sleep. "  Here it's just a dummy function because this game doesn't
 *	 require sleeping, but ADV.T requires its declaration.
 */
replace goToSleep: function
{
	"You'd rather wait until you can get inside your house";
	
	if ( global.timeLoc != 3 )
		", in your proper time,";
	
	" and into your own bed. ";
}

/*	darkTravel() is called whenever the player attempts to move from a dark
 *	location into another dark location. This game has no dark locations, so
 *	we don't have it do much of anything.
 */
replace darkTravel: function
{
	"You can't get anywhere in the dark. ";
}

/*	The die() function is called when the player dies. It tells the
 *	player how well he has done (with his score), and asks if he'd
 *	like to start over (the alternative being quitting the game).
 *
 *	Also, fullscore (v1_091).
 */
replace die: function
{
	"\b*** You have won ***\b";
	scoreRank();
	"\bYou may restore a saved game, start over, quit, or undo
	the current command. ";
	while (true)
	{
		local resp;

		"\bPlease enter RESTORE, RESTART, QUIT, FULLSCORE, or UNDO: \b>>";
		resp := upper(input());
		if (resp = 'RESTORE')
		{
			resp := askfile('File to restore',
							ASKFILE_PROMPT_OPEN, FILE_TYPE_SAVE);
			if (resp = nil)
				"Restore failed. ";
			else if (restore(resp))
				"Restore failed. ";
			else
			{
				parserGetMe().location.lookAround(true);
				scoreStatus(global.score, global.turnsofar);
				abort;
			}
		}
		else if (resp = 'RESTART')
		{
			scoreStatus(0, 0);
			restart();
		}
		else if (resp = 'QUIT' or resp = 'Q' or resp = 'QUITY')
		{
			terminate();
			quit();
			abort;
		}
		else if (resp = 'UNDO')
		{
			if (undo())
			{
				"(Undoing one command)\b";
				parserGetMe().location.lookAround( true );
				scoreStatus( global.score, global.turnsofar );
				abort;
			}
			else
				"Sorry, no undo information is available. ";
		}
		else if ( ( resp = 'FULL') or ( resp = 'SCORE' ) or ( resp = 'FULLSCORE' ) )
		{
			fullscoreVerb.action( Me );
		}
	}
}

/********************* specialized functions & daemons *********************/


/*	timemachineDaemon()
 *
 *	The guts of the sleepDaemon function are used to pull a trick -- the appearance,
 *	slowly, over time, of the time machine in the clearing. It's a daemon started
 *	by the character's second appearance in the clearing.
 *
 *	The time machine's appearance warnings are controlled also by the room the 
 *	player is standing in. If it's the clearing itself, the player sees the full
 *	messages. If the player is in the adjacent room (woods), a different message
 *	is displayed for most of the time, then a final message to announce the machine
 *	arrived.
 *
 *	The function uses moveTimeMachine() rather than the timeMachine.moveInto command
 *	in order to make sure that all of the time machine parts get moved properly.
*/
timemachineDaemon: function( parm )
{
	local f, m;

	global.machineTime := global.machineTime + 1;
	f := global.firstTime;
	m := global.machineTime;
	
	if ( m = f+6 )
		remdaemon( timemachineDaemon, nil);		 // deactivate after the sixth turn

	if ( m = f+5 )								// after 5 turns, move the time machine
		moveTimeMachine( clearing );		   // into the clearing, no matter what

	if ( Me.location = clearing )					// print the messages for each step
	{
		if ( m < f+2 )
			"\bYou feel as if something is about to happen, but don't know what. ";
		else if ( m < f+3 )
		{
			"\bThere is a strange vibration to the air, and the smell of ozone. ";
			clearing.ozone := true;
		}
		else if ( m < f+4 )
		{
			"\bSomething is definitely happening!  The center of the clearing is starting 
			to glow bright yellow. ";
			prettyLight.isWeird := 1;				// in case player wants
		}											// to examine the light
		else if ( m < f+5 )
			"\bA sound like the crack of thunder erupts, sucking in the air from around 
			you into a vacuum in the center of the clearing. The light becomes blindingly 
			bright. ";
		else if ( m = f+5 )
		{
			"\bWhen the light fades, you see that a time machine has appeared! ";
			prettyLight.isWeird := 2;		// light just after appearance
			notify( prettyLight, &isWeirdNoLonger, 5 );
			timeMachine.isSeen := true;
		}
	}
	else if ( Me.location = woods )				// if you're in the adjacent room
	{										// print a slightly different message
		if ( m > f+2 )
		{
			if ( m < f+5)
				"\bSomething is definitely going on in that clearing!  It's glowing bright 
				yellow. ";
			else if ( m = f+5 )
				"\bWhatever was going on in the clearing just stopped, because it's not 
				glowing any more. ";
		}
	}
}

/*	The newTime() function computes the new time location (1 - 5, representing
 *	FAR PAST through FAR FUTURE, respectively) based on some arithmetic involving
 *	where the player is currently ( global.timeLoc ) and the timeKnob.setting 
 *	value, which says to jump in increments of 1 or 2 units, positive or negative
 * 	(in the game, functionally equal to -20 years, -10 years, +10 years, and +20 
 *	years).
 *
 *	The alertLight.flash setting is set to true if an illegal value is reached
 *	after the arithmetic is done. This value helps the printTimeMessage function
 *	to determine the proper message to display.
 */
newTime: function ( setting )
{
	local oldTime, newTime;
	
	oldTime := global.timeLoc;
	
	switch ( setting )
	{
		case 1:
			newTime := ( oldTime - 2 );
			break;
		case 2:
			newTime := ( oldTime - 1 );
			break;
		case 3:
			newTime := oldTime;
			break;
		case 4:
			newTime := ( oldTime + 1 );
			break;
		case 5:
			newTime := ( oldTime + 2 );
			break;
	}
	
	if ( ( newTime < 1 ) or ( newTime > 5 ) )	// Restrict ourselves to legal values
	{
		newTime := oldTime;
		alertLight.flash := true;
	}
	
	return ( newTime );
}

/*	The moveTimeMachine function does the business of moving the time machine (which is
 *	actually several different objects) from place to place when you push the button.
 *	Otherwise, you end up with a machine but no platform (which the actor is always 
 *	required to be standing on) or knob.
 */
moveTimeMachine: function ( newLocation )
{
	timeMachine.moveInto( newLocation );
	timeKnob.moveInto( newLocation );
	platform.moveInto( newLocation );
	
	timeMachine.heat := 8;								// heat up the time machine
	setdaemon( coolOffDaemon, timeMachine );			// start it cooling off
	addword( timeMachine, &adjective, 'heat' );
	
	compartment.heat := 15;								// the compartment gets extremely hot
	setdaemon( coolOffDaemon, compartment );
	addword( compartment, &adjective, 'heat' );
	
	if ( length( compartment.contents ) > 0 )					// if there's something inside
	{
		local i, l;
		l := compartment.contents;
		
		for( i:=1; i <= length(l); i++ )
		{
			if ( l[i].heat = nil || l[i].heat = 0 )
				setdaemon( coolOffDaemon, l[i] );
			l[i].heat := 12;
			addword( l[i], &adjective, 'heat' );
		}
	}
}

/*	The printTimeMessage function determines the appropriate declaration based on 
 *	where the time machine is going, or whether the alert bulb has flashed because 
 *	of error (alertLight.flash = true). If all goes well, a message prints, with
 *	slight variations of color based on how far you are travelling (-20, -10, etc.),
 *	using the setting of the time machine knob as a value check.
 */
printTimeMessage: function ( newTimeLoc, setting )
{
	if ( setting = 3 )
	{
		"Nothing seems to happen. ";
		return ( nil );
	}
	else
	{
		if ( alertLight.flash )
		{
			"A little alert bulb on the time machine flashes briefly. Perhaps the 
			machine has a limited range, because nothing seems to happen. ";

			alertLight.flash := nil;
			return ( nil );
		}
		else
		{
			"The time machine vibrates and heats up like a toaster oven. A ";
	
			if ( setting = 1 )	"deep red ";
			else if ( setting = 2 ) "dull orange ";
			else if ( setting = 4 ) "bright blue ";
			else if ( setting = 5 ) "blinding purple ";
	
			"light envelops you. ";
			
			"The world around you spins into a wild vortex 
			that suddenly vanishes, leaving you to find yourself in a ... \b";
			
			return ( true );
		}
	}
}

/*	The timeJump() function determines whether a valid jump is in progress, and if
 *	so, moves the time machine (using the moveTimeMachine() function), moves the 
 *	player into the same new location, and then finally moves the player onto the
 *	time machine platform again. It does the double-move so that the player is
 *	counted as having entered the new location, activating firstseen functions and
 *	the like.
 *
 *  This function calls printTimeMessage to determine the validity of the jump, which
 *	is a strange way to do it, perhaps, because printTimeMessage should be in the
 *	business only of printing messages, not making value judgments. However, this
 *	seemed like the best way to get the job done without a major rewrite of all of
 *	this time machine jumping code.
 */
timeJump: function ( actor, newLoc )
{
	local goodJump;
	
	goodJump := printTimeMessage( global.timeLoc, timeKnob.setting );
	
	if ( goodJump )									// valid jump?
	{
		moveTimeMachine( newLoc );
		actor.travelTo( newLoc );
		actor.moveInto( platform );
	}
}

/*	garageState() is called whenever the player pushes a garage door opener
 *	button. The only garage doors in this game have otherside properties, but
 *	the function checks to make sure just in case.
 *
 *	playerInLoc is true if the player is standing in the same room as the 
 *	garage door, and therefore would see it open or shut.
*/
garageState: function ( whichdoor, playerInLoc )
{
	if ( whichdoor.isopen )
	{
		if ( playerInLoc )
			"\bThe garage door smoothly glides shut. ";
		whichdoor.setIsopen( nil );
	}
	else
	{
		if ( whichdoor.islocked )
		{
			if ( playerInLoc )
				"\bThe garage door mechanism strains and grinds against
				the locked door, and then gives up. ";
		}
		else
		{
			if ( playerInLoc )
				"\bThe garage door smoothly glides open. ";
			whichdoor.setIsopen( true );
		}
	}
}


/*	The coolOff daemon simply decrements the heat property of an item once per turn.
 *	This is used for the time machine and any sandwiches put inside it.
 */
coolOffDaemon: function ( item )
{
	item.heat--;
	
	if ( item.heat < 0 )
	{
		item.heat := 0;
		remdaemon( coolOffDaemon, item );
		delword( item, &adjective, 'heat' );
	}
}

/* cartMove: A function for moving a cartItem and the player (or blocking the 
 * attempted travel) when pushing a cartItem around. Specifically, this refers
 * to the wheelbarrow(s) in the game, which is too big to fit through most 
 * doorways. 
*/
cartMove: function ( newCart, actor, newLoc, direction )
{
	if ( newCart.location = newLoc ) {
		"Pushing the cart that direction doesn't seem to get you anywhere. ";
		return( nil );
	}
	if ( newLoc.doordest )
	{
		if ( isclass( newLoc, garageDoorItem ) )
		{
			if ( newLoc.isopen ) {
				"You push <<newCart.thedesc>> ";
				say( newCart.moveStr[direction] );
				". ";
				newCart.moveInto( newLoc.doordest );
			}
			else
				"The wheelbarrow is blocked by the closed <<newLoc.sdesc>>. ";
		}
		else
		{
			if ( newLoc.isopen )
				"The wheelbarrow won't fit through the doorway. ";
			else
				"\^<<newLoc.thedesc>> is closed, but the wheelbarrow wouldn't
				fit through it anyway. ";
			return( nil );
		}
	}
	else
	{
		if ( isclass( newCart.location, platformItem ) ) 
			"You maneuver <<newCart.thedesc>> off of <<newCart.location.thedesc>> and 
			push it ";
		else
			"You push <<newCart.thedesc>> ";
		
		say( newCart.moveStr[direction] );
		". ";
		newCart.moveInto( newLoc );
	}
	actor.travelTo( newLoc );
	return( true );
}

/*	interest() computes the interest on an account balance over the course of ten 
 *	years. It is actually an unrealistic 80% rate, but somehow the numbers it
 *	generated seemed "right" for the game's purposes, so left it. Also, when I
 *	changed the 8/10 to 8/100 it caused the function to fail and generate only
 *	a year's worth of interest for ten years' time. Oh, well, it works well 
 *	enough for the plot.
 *
 *    (Beta 4) Leaving the interest function the same, I halved the results to
 *    make the interest dividends somewhat less fantastic.
 */
interest: function ( oldBalance, years )
{
	local newBalance;

	newBalance := 0;

	if ( years = 0 )
	{
		newBalance := oldBalance;
	}
	else
	{
		newBalance := oldBalance + ( interest( oldBalance, ( years - 1 ) ) * 71 / 100 );
	}

	return newBalance;
}

updateAccount: function ( account )
{
	local fp, p, pt, f, ff;
	
	fp := account.value;
	p := interest( fp, 10 );					// interest over ten years	
	pt := interest( p, 10 );					// etc.	
	f := interest( pt, 10 );
	ff := interest( f, 10 );
	
	account.FP := fp;
	account.P := p;
	account.PT := pt;
	account.F := f;
	account.FF := ff;
	
	if ( account.F > 10000000 )
		ATM.outOfOrder := true;
}

setupAccount: function ( account )
{
	local yesno;
	
	if ( account.isopen ) 
	{
		"Laura taps on her computer and says, \"You seem to already
		have an account. You need to see the teller for transactions.\"
		She smiles at you. \"Sorry, one per customer!\" ";
		
		return( nil );
	}
	 
	"Laura says, \"Okay. We require a minimum deposit of $100 in order to 
	open your account. Is that all right?\" \b>>";
					
	yesno := yorn();
	
	if ( yesno = 1 )
	{
		if ( global.money >= 100 )
		{
			global.money -= 100;
			playerCash.value -= 100;
			
			if ( playerCash.value = 0 )
				playerCash.moveInto( nil );
			
			account.isopen := true;
			account.value := 100;
			updateAccount( account );
			ATMkeypad.setting := account.PINnumber;
			
			incscore( 7 );
			global.scoreCard[19] := 1;
			
			"Laura smiles and says, \"Okay, you're all set. Your PIN number
			will be 1234. Thank you!\" ";
			
			return( true );
		}
		else
		{
			"Laura frowns. \"I'm sorry, but can't help you if you don't 
			have the money to open a new account,\" she says. ";
			
			account.failedToOpen := true;
		}
	}
	else
		"\"Well, have a nice day, then.\" ";
	
	account.askedAboutOnce := true;
	
	return( nil );
}

timeOfDay: function( timeLoc )
{
	switch( timeLoc )
	{
		case 1:
			"early morning";
			break;
		case 2:
			"mid-day";
			break;
		case 3:
			"late evening";
			break;
		case 4:
			"late afternoon";
			break;
		case 5:
			"in the twilight before full sunset";
			break;
	}
}

/*  ConsultWords takes three arguments and ultimately returns true
	(NPC had something unique to say on the subject) or nil (NPC 
	had nothing to say, so eventually a default message from 
	disavow() will be printed).

	The three arguments: word (actually the object) being asked/spoken 
	about, who is asking/telling, and whether this is an ASK or a TELL.
 */
consultWords: function( word, who, askOrTell ) 
{
	if ( askOrTell = 1 )		/* ask */
	{
		if ( word = who )   // asking NPC about NPC
		{
			who.talkAboutMyself;
			return( true );
		}
		
		if ( word = Me )	// asking NPC about myself
		{
			who.doWhoAmI( Me );
			return( true );
		}
		
		return( word.askingAboutMe( who ) );	// pass it on to the word.
	}
	else				/* tell */
	{
		if ( word = who )   // telling NPC about NPC
		{
			"There's not much you can tell <<who.thedesc>> about 
			<<who.fmtYoum>>self";
			
			if ( ( ( who = Fred || who = FFred ) && FFred.met ) || 
				( ( who = Laura || who = PLaura || who = FFLaura ) && 
				( FLaura.met || FFLaura.met ) ) )
				", unless maybe you meant to tell <<who.fmtYoum>> about a 
				different-era version of <<who.thedesc>>";
			". ";
			
			return( true );
		}
		
		if ( word = Me )	// telling NPC about myself
		{
			switch( who )
			{
				case Fred:
				case Laura:
					"You fill <<who.thedesc>> in a little bit about yourself: your
					interest in science and time travel, how you've lived in the area
					for about ten years now, how you prefer to walk rather than drive
					a car. \^<<who.fmtYou>> listens with a polite but only passing 
					interest. ";
					return( true );
				case Architect:
					"You start to tell <<who.thedesc>> a little bit about yourself
					and your interest in science and time travel, but he cuts you
					off with a plea to let him get back to work. ";
					return( true );
				case PLaura:
					"You tell Laura a little bit about yourself and your interests,
					but she seems distracted and obviously isn't paying any
					attention. ";
					return( true );
				case FFLaura:
					"You get the feeling that she knows all about you without
					your having to say anything. ";
					return( true );
				case youngMan:
					"You get the feeling that he knows all about you without
					your having to say anything. ";
					return( true );
				case securityGuard:
					"You start to tell the guard about yourself and how you got
					here, but she cuts you off.\b
					\"I'm not the least bit interested,\" she says. ";
					return( true );
				case workers:
					"They probably aren't interested in your life story. ";
					return( true );
				case teller:
					"He listens with rigid politeness, then interrupts. \"May I
					help you with a withdrawal or a deposit today, sir?\" ";
					return( true );
				case antiqueMerchant:
					"She listens, nodding with a friendly smile. \"That's 
					nice,\" she says. \"Let me know if you would like to
					sell or buy any antiques today.\" ";
				default:
					return( nil );
			}
		}
		
		return( word.tellAboutMe( who ) );
	}
}

doSquirrelAction: function( num )		 // animates the squirrel.
{
	"\b";
	switch ( num )
	{
		 case 0:
			"The squirrel is rather squat and feral, but it regards you with
			an almost intelligent interest. ";
			break;
		 case 1:
		 	"The squirrel gazes into the distance, daydreaming about something. ";
			break;
		 case 2:
		 	"The squirrel dashes from one spot to another, then twitches
		 	its jaws. ";
			break;
		 case 3:
		 	"The squirrel nibbles at his tail. ";
			break;
		 case 4:
		 	"The squirrel lazes about in a state of squirrel contentment. ";
			break;
		 case 5:
		 	"The squirrel shakes his little head. It doesn't mean anything
		 	significant. ";
			break;
		 case 6:
		 	"The squirrel gazes into the distance, daydreaming about something. ";
			break;
		 case 7:
		 	"The squirrel flicks his tail and blinks. ";
		 	break;
		 case 8:
		 	"The squirrel paces back and forth. ";
		 	break;
		 case 9:
		 	"The squirrel squats down. ";
			break;
		 case 10:
		 	"A mood seems to come over the squirrel, indistinguishable from
		 	other squirrel moods. ";
			break;
		 case 11:
		 	"The squirrel pouts, looks excited, then pouts again. Just
		 	your usual squirrel business. ";
			break;
		 case 12:
		 	"The squirrel waddles a bit. ";
			break;
		 case 13:
		 	"The squirrel scampers back and forth on the rooftop. ";
			break;
		 case 14:
		 	"The squirrel looks around, then stares at you. ";
			break;
		 case 15:
		 	"The squirrel nibbles and fidgets. ";
			break;
		 case 16:
		 	"The squirrel doesn't react. ";
			break;
		 case 17:
		 	"The squirrel walks in a slow circle. ";
			break;
		 case 18:
		 	"The squirrel makes a funny 'tk-tk-tk-tk' noise. ";
			break;
		 case 19:
		 	"The squirrel paces around, turns, and paces some more. Then he
		 	just kind of sits there. ";
			break;
		 case 20:
		 	"The squirrel nibbles at one of his little paws. ";
			break;
		 case 21:
		 	"The squirrel looks shocked and clutches at his little neck.
		 	Or not. He was probably just doing some squirrel thing that
		 	you read too much into. ";
			break;
		 case 22:
		 	"The squirrel swishes its tail in a curling motion. ";
			break;
		 case 23:
		 	"The squirrel flops out flat and takes a short squirrel nap. ";
			break;
		 case 24:
		 	"The squirrel fluffs his tail. ";
			break;
		 case 25:
		 	"The squirrel perks up, then looks bored again. No wait,
		 	he just looked bored the whole time. ";
			break;
		 case 26:
		 	"The squirrel does nothing at all. ";
			break;
		 case 27:
		 	"The squirrel retreats a few paces from you, probably just
		 	because it's a squirrel and it likes to move around constantly. ";
			break;
		 case 28:
		 	"The squirrel tilts his head. ";
			break;
		 case 29:
		 	"The squirrel hunches on his back legs and stares blankly
		 	at you. ";
			break;
		 case 30:
		 	"The squirrel twitters and sniffits. ";
			break;
		 case 31:
			"The squirrel goes a bit nutty, but he's always like that. ";
			break;
		 case 32:
		 	"The squirrel flicks his tail and blinks. ";
			break;
		 case 33:
		 	"The squirrel paces back and forth. ";
			break;
		default: 
			"The squirrel scratches its belly. "; 
			break;
	}
}

doFredAction: function( num )		 // animates fred
{
	switch ( num )
	{
		case 0:
		case 1:
		case 2:
		case 3:
			"\bFred is cleaning some utensils. ";
			break;
		case 5:
			"\bFred dries his hands with a paper towel. ";
			break;
		case 4:
		case 14:
		case 24:
		case 34:
			"\bFred whistles a half-remembered melody. ";
			break;
		case 6:
		case 7:
			"\bFred dices some vegetables. ";
			break;
		case 9:
			"\bFred puts some bread into the oven. ";
			break;
		case 8:
		case 10:
			"\bFred cleans the countertop. ";
			break;
		case 11:
		case 12:
		case 13:
			"\bFred stirs the hot meatball sauce. ";
			break;
		case 15:
			"\bFred refills the mustard containers. ";
			break;
		case 16:
		case 17:
		case 19:
			"\bFred refills the napkin dispenser. ";
			break;
		case 18:
		case 22:
		case 28:
			"\bFred puts some fresh cold cuts in the fridge. ";
			break;
		case 20:
		case 21:
			"\bFred swats away a pesky fly. ";
			break;
		case 23:
			"\bFred scratches his nose. ";
			break;
		case 25:
			"\bFred fidgets. ";
			break;
		case 26:
		{
			local topic;
			topic := chooseTopic( Fred );
			
			"\bFred looks at you and clears his throat. ";
			"\^<<Fred.thedesc>> brings up the subject of <<topic.topicDesc>>. \b";
			
			Fred.doAskAbout( Me, topic );
			break;
		}
		case 27:
			"\bFred dices some vegetables. ";
			break;
		case 29:
			"\bFred puts some bread into the oven. ";
			break;
		case 31:
			"\bFred cleans the countertop. ";
			break;
		case 33:
			"\bFred stirs the hot meatball sauce. ";
			break;
		case 37:
		{
			local topic;
			topic := chooseTopic( Fred );
			
			"\bFred raps his knuckles on the counter. ";
			"\^<<Fred.thedesc>> brings up the subject of <<topic.topicDesc>>. \b";
			
			Fred.doAskAbout( Me, topic );
			break;
		}
		case 38:
			"\bFred puts some fresh cold cuts in the fridge. ";
			break;
		case 40:
			"\bFred swats away a pesky fly. ";
			break;
		case 41:
			"\bFred scratches his nose. ";
			break;
		case 42:
			"\bFred fidgets. ";
			break;
		case 43:
		{
			local topic;
			topic := chooseTopic( Fred );
			
			"\^<<Fred.thedesc>> brings up the subject of <<topic.topicDesc>>. \b";
			
			Fred.doAskAbout( Me, topic );
			break;
		}
		case 44:
			"\bFred picks up a handful of utensils. ";
			break;
		default: 
			break;
	}
}

doLauraAction: function( num )		 // animates laura
{
	"\b";
	switch ( num )
	{
		case 0:
		case 2:
		case 12:
			"Laura smiles at you. ";
			break;
		case 19:
			"Laura smiles at you. ";
			// local topic;
			// chooseTopic( topic );
			"She brings up the subject of ";
			"[BUG]" ;// self.topic.topicDesc;
			". ";
			// self.doAskAbout( Me, topic );
			break;
		case 9:
		case 11:
			// local topic;
			// chooseTopic( topic );
			"Laura brings up the subject of ";
			"[BUG]" ;// self.topic.topicDesc;
			". ";
			// self.doAskAbout( Me, topic );
			break;
		case 1:
		case 3:
		case 4:
		case 7:
		case 8:
		case 13:
		case 14:
		case 19:
		case 21:
			"Laura types data into her computer. ";
			break;
		case 5:
			"Laura's phone rings. She answers it. ";
			break;
		case 6:
			"Laura hangs up the phone again. ";
			break;
		case 15:
			"Laura opens her desk drawer and looks for something. ";
		case 16:
			"Laura looks through her desk drawer a second time. ";
			break;
		case 17:
			"Laura finds a large binder clip and pins together a stack
			of paperwork with it. ";
			break;
		case 18:
			"Laura hands the stack of paperwork to another bank employee. ";
		case 10:
		case 20:
		case 30:
		case 40:
			"Picks up a handful of notes and shuts them in a drawer. ";
		case 22:
			"Laura prints out a document. ";
		default: 
			"Laura types numbers into a spreadsheet. "; 
			break;
	}
}

doBlackieAction: function( num )		 // animates Blackie the dog.
{
	switch ( num )
	{
		 case 0:
		 case 1:
		 case 2:
		 case 3:
		 case 4: break;
		 case 5:
		 	"\b\^<<Blackie.thedesc>> scratches his ear. ";
			break;
		 case 6:
		 case 7:
		 case 8:
		 case 9:
		 case 10: break;
		 case 11:
		 	"\b\^<<Blackie.thedesc>> chews at a flea. ";
			break;
		 case 12:
		 case 13:
		 case 14:
		 case 15:
		 case 16:
		 case 17:
		 case 18:
		 case 19: break;
		 case 20:
		 	"\b\^<<Blackie.thedesc>> chews at a flea. ";
			break;
		 case 21:
		 case 22:
		 case 23:
		 case 24:
		 case 25:
		 case 26:
		 case 27: break;
		 case 28:
		 	"\b\^<<Blackie.thedesc>> tilts his head. ";
			break;
		 case 29:
		 case 30:
		 case 31: 
		 case 32: break;
		 case 33:
		 	"\b\^<<Blackie.thedesc>> wags his tail. ";
			break;
		default: 
			break;
	}
}

newListenDesc: function( loc, str )
{
	swapListenDesc( loc );
	loc.savedListenDesc := loc.listenDesc;
	loc.listenDesc := str;
}
;
swapListenDesc: function( loc )
{
	local temp;
	temp := loc.savedListenDesc;
	loc.savedListenDesc := loc.listenDesc;
	loc.listenDesc := temp;
}
;
restoreListenDesc: function( loc )
{
	loc.listenDesc := loc.savedListenDesc;
}
;

/* chooseTopic:  NPCs randomly bring up conversation topics that might give the
 *	player a nudge in the right direction.
 */
chooseTopic: function( who )
{
	local i, topic;
	
	i := 1;
	
	while( global.scoreCard[i] != 0 )	// find what the player hasn't done yet
	{
		i++;
	}
	
	switch( i )
	{
		case 1:
			if ( carKey.location )
			{
				if ( littleCar.hasOpened )
					topic := armrest;
				else
					topic := littleCar;
			}
			else
			{
				who.askedAboutKey := true;
				topic := stones;
			}
			break;
		case 2: topic := stoneWall1; break;
		case 3: 
			if ( junk.isSeen )
				topic := junk;
			else
			{
				if ( garageOpener.isSeen )
					topic := garageOpener;
				else
					topic := armrest;
			}
			break;
		case 4: topic := dummyClearing; break;
		case 5: 
		case 6: 
		case 7: topic := timeMachine; break;
		case 8: topic := shingles2; break;
		case 9: topic := drainpipe; break;
		case 10: 
			if ( diviningRod.isIn( Me ) )
				topic := diviningRod; 
			else if ( machete.isIn( Me ) )
				topic := machete;
			else
				topic := diviningRod;
			break;
		case 11: topic := acorn; break;
		case 12: topic := frontWheel; break;
		case 13: topic := tinyTree; break;
		case 14: topic := jug5; break;
		case 15: topic := window3; break;
		case 16: topic := tireSwing; break;
		case 17: topic := keepingSandwichesHot; break;
		case 18:
			if ( blueprint.secretpassage )
				topic := secretDoor; 
			else
				topic := blueprint; 
			break;
		case 19: topic := raisingMoney; break;
		case 20: 
			if ( lightningRod.location )
				topic := lightningRod;
			else
				topic := antiques;
			break;
		case 21: 
			if ( FFred.met )
				topic := alcoholism;
			else if ( holeInWall.isSeen )
				topic := holeInWall;
			else
				topic := gettingIntoFutureHouse;
			break;
		case 22: topic := dibbleBitsStock; break;
		case 23: topic := newFuture; break;
		case 24: topic := duplicateKey; break;
		case 25: 
			who.askedAboutKey := true;
			topic := stones; break;
	}
	
	return( topic );
}

/* provideHint() takes a topic chosen by chooseTopic() and gives the player a 
 * 	hint about what maybe to do next.
 */
provideHint: function( topic )
{
	switch( topic )
	{
		case stones:
			if ( frontDoor.triedOnce )
				"\n\ Is it possible that I've hidden a key somewhere 
				outside the house<<doormat.lookedUnderOnce ? 
				", and just haven't found it yet?" : "?">> ";
			else
				"\n\ I'm really tired. I need to go in and go to sleep. ";
			return( true );
		case stoneWall1:
		{
			local i;
			i := 1;
			{
				if ( northOfHouse.isseen )
					 i++;
				if ( behindHouse.isseen )
					i++;
				if ( southOfHouse.isseen )
					i++;
			}
			"\n\ <<i = 4 ? "I've seen all four sides, but i" : "I">>s there any part 
			of the house I haven't explored yet? ";
			return( true );
		}
		case littleCar:
			"\n\ <<frontDoor.spareKeyTried ? "I've tried it on the
			front door, but I still" : "I">> haven't figured out what the
			spare key might open. ";
			return( true );
		case armrest:
			if ( gloveCompartment.isSeen )
				"\n\ I've been in the little car, but have I searched 
				it thoroughly enough? ";
			else
				"\n\ I've unlocked the little car, but I think I should
				climb in and see what's in there. ";
			return( true );
		case garageOpener:
			if ( garageDoor1.isopen )
				"\n\ I think I should probably go into the garage now that
				I've managed to open it. ";
			else
				"\n\ I wonder if the garage door opener can open the garage. I 
				bet I should try it. ";
			return( true );
		case junk:
			if ( crowbar.location )
				"\n\ There was a lot of stuff there. I wonder if there's 
				anything else still hidden in the junk in the garage. ";
			else
				"\n\ I wonder if there's anything hidden in the junk
				in the garage. ";
			return( true );
		case dummyClearing: 
			if ( not clearing.isseen )
				"\n\ I wonder if I've explored everywhere I can go yet. ";
			else
			{
				if ( timeMachine.isSeen )
					"\n I wonder if I've examined the time machine closely
					enough. ";
				else
					"\n\ There was that funny warmth to the air in the 
					clearing. I wonder if I should go back to investigate 
					what was happening there. ";
			}
			return( true );
		case timeMachine: 
			if ( timeButton.pushedOnce )
			{
				if ( timeMachine.location )
				{
					if ( timeManual.readOnce || timeManual.readAboutTimeButton )
					{
						"\n\ I think I need to let the time machine 
						cool off, like the manual said. ";
					}
					else
					{
						"\n\ I wonder why the button isn't working. ";
						
						if ( timeManual.isSeen )
							"Maybe I should read about it in the manual. ";
						else
							"Maybe I should examine the time machine more
							closely. ";
					}
				}
				else
				{
					"\n\ D'oh! I wonder if I should have examined the time 
					machine more closely before pushing the button. ";
				}
			}
			else
			{
				local i;
				i := 0;
				{
					if ( FPclearing.isseen )
						 i++;
					if ( Pclearing.isseen )
						i++;
					if ( Fclearing.isseen )
						i++;
				}
				
				if ( i > 0 )
					"\n\ I wonder where else the time machine can take me. ";
				else
					"\n\ I haven't used the time machine yet. I probably need to 
					see where it can take me. ";
			}
			return( true );
		case shingles2: 
			"\n\ Those bad shingles on the roof really bother me. I wonder if
			I can do something to change that. ";
			return( true );
		case drainpipe: 
		{
			"\n\ The drainpipe is really clogged";
			
			if ( length( stuckLimbo.contents ) > 0 )
				", and I've lost some things down there";
			
			". I wonder if there's some way I can force everything out
			of it. ";
			return( true );
		}
		case diviningRod: 
		case machete:
			"\n\ I haven't been able to find my way through the dense 
			foliage in the far past yet. I wonder if there's something I
			can use to get me through it? ";
			return( true );
		case acorn: 
			"\n\ I haven't planted the acorn yet. Maybe I can find a good 
			spot to plant it. ";
			return( true );
		case frontWheel: 
			if ( acornMound.groCount > 0 )
			{
				if ( acornMound.groCount = 1 )
				{
					if ( wheelbarrow.broken )
						"\n\ I wonder if there's some way I could have fixed 
						the busted wheel on the wheelbarrow before it broke off
						for good. ";
					else
						"\n\ The busted front wheel on the wheelbarrow bothers me.
						I wonder if I can fix it before it breaks for good. ";
				}
				else
					"\n\ I wonder if there's some way to get all of the 
					fertilizer to the acorn in one trip. ";
				return( true );
			}
		case tinyTree: 
			if ( tinyTree.isSeen )
				"\n\ I planted the tree, but it didn't grow up as big as I 
				was expecting. I wonder if I need to nourish it with some
				kind of fertilizer so that it grows big and strong. ";
			else
				"\n\ I planted the acorn. I wonder how big it'll grow up
				to be. ";
			return( true );
		case jug5: 
			"\n\ The tree is really huge now, but I wonder if it could
			grow even just a little taller. ";
			
			if ( jug5.isSeen = nil )
				"Maybe I should see if I can find any more miracle-gro. ";
			
			return( true );
		case window3: 
			"\n\ I wonder if I can get into the attic of the house 
			through the tiny window on the south side now. ";
			return( true );
		case tireSwing: 
			"\n\ There's really only one way out of the attic. I wonder 
			if <<southOfHouse.ankle ? "I should have used" :
			"should use">> something to break my fall when I slide down
			the rope, because it's a long way down. ";
			return( true );
		case keepingSandwichesHot: 
			"\n\ I need to get the architect to leave his desk for a 
			couple of minutes, and he seems to want a hot sandwich. ";
			if ( hotSandwich.heat > 0 )
				"Perhaps I can distract him by giving it to him. ";
			else
				"If only I had some way to reheat it before 
				giving it to him. ";
			return( true );
		case blueprint: 
			"\n\ I wonder if <<bluePencil.isSeen ? "I should use the
			blue pencil to draw something new on the blueprint" : 
			"there's a blue pencil somewhere, so that I could draw
			something new on the blueprint">>. ";
			return( true );
		case secretDoor: 
			"\n\ I can't seem to find the secret door. I wonder if the
			architect could tell me how to open it. ";
			return( true );
		case raisingMoney:
			"\n\ I really need some money so that I can open a bank 
			account. I wonder if I could sell any of the junk from my
			garage to raise some funds. ";
			return( true ); 
		case antiques:
			"\n\ I wonder if there's anything hidden in the antique
			store that I haven't discovered. ";
			return( true );
		case lightningRod:
			"\n\ I can't help thinking about the storm damage to my house
			in the future. I wonder if installing the lightning rod would
			prevent that. ";
			return( true );
		case gettingIntoFutureHouse:
			"\n\ With all the damage to the house in the future, I wonder
			if maybe I can be a bit more cavalier about getting in. There's
			probably a couple of ways in, now that I think about it. ";
			return( true );
		case holeInWall:
			"\n\ There was that hole punched in the wall in the upstairs
			hallway of my house in the future. I wonder if I should have
			looked in there. ";
			return( true );
		case alcoholism: 
			"\n\ It makes me sad to see the way Fred turns out. I wonder 
			if I can tell him about this, and somehow make him see what
			happens to him. Maybe it'll set him straight. ";
			return( true );
		case dibbleBitsStock: 
			"\n\ If I want to do something useful with the money I've 
			made, maybe I should invest it in a stock I know will 
			be a big success in the future. That might give me some
			leverage against the Lothario Corporation trying to buy 
			my home. ";
			return( true );
		case newFuture: 
			"\n\ I wonder if what I've just done has changed the future
			for the better. Maybe it's time to travel there and see 
			what's come of things. ";
			return( true );
		case duplicateKey: 
			"\n\ I think I should put the duplicate key under the doormat,
			where it should have been in the first place. First things
			first, after all. ";
			return( true );
	}
	return( nil );
}


/* listIssues() will display a prose list of unsolved puzzles facing 
 * the player, if the player sits on the rock to meditate.
 */
listIssues: function( scoreCard )
{
	local temp;
	for ( temp:=1; temp<=20; temp++ )
	{
		if ( scoreCard[ temp ] = 0 )
		{
			switch( temp )
			{
				case 1: 
					"\n\ Have you explored every possible way into the house yet? ";
					abort;
				case 2: 
					"\n\ Have you been back to the clearing to investigate what was
					happening there? ";
					abort;
				case 3: 
					"\n\ Have you been everywhere the time machine can take you? ";
					abort;
				case 4: 
					"\n\ Have you been everywhere the time machine can take you? ";
					abort;
				case 5:
					"\n\ Have you been everywhere the time machine can take you? ";
					abort;
/*				case 6: 
					"\n\ 3 points for cleaning out the drainpipe ";
					break;
				case 7: 
					"\n\ 5 points for finding the armrest compartment ";
					break;
				case 8: "\n12 points for getting to the end of the junk in the garage ";
					break;
				case 9: "\n\ 2 points for fixing the wheel with the screwdriver ";
					break;
				case 10: "\n\ 4 points for giving the bad shingle to the architect ";
					break;
				case 11: "\n\ 4 points for making it through the maze to the stream ";
					break;
				case 12: "\n\ 6 points for planting the acorn in the hole ";
					break;
				case 13: "\n\ 3 points for for pouring a second jug of miracle-gro ";
					break;
				case 14: "\n\ 5 points for for pouring a fifth jug of miracle-gro ";
					break;
				case 15: "\n10 points for leaping to the attic ";
					break;
				case 16: "\n\ 1 point for using the tire to break your fall from the attic ";
					break;
				case 17: "\n10 points for giving a piping hot sandwich to the architect ";
					break;
				case 18: "\n\ 4 points for entering the secret passage the first time ";
					break;
				case 19: "\n\ 7 points for opening a new bank account ";
					break;
				case 20: "\n\ [BUG] ";
					break;
				case 21: "\n";
					if ( global.scoreCard[ temp ] < 10 )
						"\ ";
					"<<global.scoreCard[ temp ]>> points for finding
					various items ";
					break;
*/
			}
		}
	}
//	"\b[ BUG ] - list_issues();\n ";
	"\n";
	return( true );
}

/* topLocation: function
 * This function returns the "topmost" room an object is in, even if 
 * obj.location is in a nested container, platform, or vehicle within that room.
*/
topLocation: function( obj )
{
	if ( obj = nil ) {
		"Error: can't call topLocation() with nil object. ";
		return( nil );
	}
	if ( obj.location = nil ) return( obj );
	return( topLocation( obj.location ) );
}

/* handleMoneyTrade: function
 *
 * handleMoneyTrade( newMoneyItem, newValue ) does the business of sorting out if the 
 * player is already carrying a moneyItem, and if so, adding newMoneyItem's value to 
 * the player's cash, and removing the old moneyItem in favor of a new one with the 
 * complete total.
 */
handleMoneyTrade: function( newMoneyItem, newValue )
{
	if ( playerCash.location = nil ) {
		playerCash.moveInto( Me );
	}
	playerCash.value += newValue;
	global.money += newValue;
	if ( newMoneyItem != playerCash )
		newMoneyItem.moveInto( nil );
	
	if ( playerCash.location != Me ) {
		if ( playerCash.deepIsIn( topLocation( Me ) ) ) {
			"You put the money into <<playerCash.location.thedesc>> along
			with the rest of it. ";
		}
		else {
			playerCash.moveInto( Me );
			"You somehow gather all of your money together
			along with the new cash.\b
			[It is now in your hands, and not in <<playerCash.location.thedesc>> 
			where you last left it. Presto!]";
		}
	}
	return( true );
}

sayMessage: function( num )
{
	switch( num )
	{
		case 0: 
			unnotify( workers, &actorDaemon );
			break;
		case 1:
			"Someone shouts, \"Get that paint over to number eight!\" ";
			break;
		case 2:
			"A teenaged kid dashes by with hot coffee for the foreman. ";
			break;
		case 3:
			"Two guys wheel past with a load of cinder blocks. ";
			break;
		case 4:
			"A man with a hard hat barks commands to some dust-covered
			strongarms. ";
			break;
		case 5:
			"Someone shouts, \"Where's the langstrom 7-inch gangley wrench?\" ";
			break;
		case 6:
			"A young man scurries by, holding three bundles of copper piping. ";
			break;
		case 7:
			"Two women walk by, discussing their favorite winch pulley
			manufacturers. ";
			break;
		case 8:
			"Someone up in the roof rafters shouts orders that are ignored. ";
			break;
		case 9:
			"Someone cries, \"Yeah! We've been waitin' for those!\" ";
			break;
		case 10:
			"A huge man with a mustache to match lopes by, carrying a trowel
			and a bag of plaster mix. ";
			break;
		case 11:
			"A short man in a blue shirt says, \"We ain't got enough water
			in the mix. Where's the foreman? Get the foreman.\" ";
			break;
		case 12:
			"A crashing noise is heard, and then a burst of laughter. You 
			can't see what caused it. ";
			break;
		case 13:
			"The sharp noise of nails being hammered into thick planks causes 
			you to blink your eyes several times. ";
			break;
		case 14:
			"A young woman with a large box of wood screws rushes past, crying, 
			\"Coming! Coming!\" ";
			break;
		case 15:
			"A sweaty, red-faced man yells, \"Where are my ducts? Where are
			my flim-flanged ducts?\" ";
			break;
		case 16:
			"Someone yells, \"That's a riot! Where's Ed? He's gotta hear this!\" ";
			break;
		case 17:
			"A nervous kid slouches by, looking like it's his first day on 
			the job. ";
			break;
		case 18:
			"A trio of men with puffy faces and scruffy chest hair stride past,
			telling dirty jokes. ";
			break;
		case 19:
			"A beefy blond man waddles past, blowing his nose ferociously
			into the sleeve of his shirt. ";
			break;
		case 20:
			"Someone yells, \"Fifteen! Fifteen, everybody!\"
			\nSomeone yells back, \"You got that right!\" ";
			break;
		case 21:
			"A young woman with a pile of floor tiles rushes past, crying, 
			\"Who ordered these?\" ";
			break;
		case 22:
			"A guy clambers down from above and yells, \"Let's eat!\". ";
			break;
		case 23:
			"Someone rushes up to you, looks you up and down, and then
			shakes his head and runs off. ";
			break;
		case 24:
			"A man with round spectacles stares at the sky and predicts
			rain. ";
			break;
	}
}

sayMessage2: function( num )
{
	switch( num )
	{
		case 25:
			"A wiry lad from the electrical crew says, \"We need to wait for
			the plumbers.\" ";
			break;
		case 26:
			"There is a crashing noise and some amount of furor and muttered
			oaths from the construction crew. ";
			break;
		case 27:
			"Someone shouts, \"Nobody move! Wait, everyone get out of the
			way! I said, get out of the way!\" ";
			break;
		case 28:
			"A fat man jumps up and down and makes 'bibble-brrrrrr' 
			noises with his mouth. Oddly, nobody seems to care. ";
			break;
		case 29:
			"A handsome, tanned woman walks past, sipping a cup of tea. ";
			break;
		case 30:
			"Someone shouts, \"Budget! We're over budget! I can feel it.\" ";
			break;
		case 31:
			"There is a whoop from a nearby card game being played by a 
			team of idle roof shinglers. ";
			break;
		case 32:
			"A somber man winds past, unfurling a length of thick cabling
			behind him. ";
			break;
		case 33:
			"A set of floor tiles is laid out on the ground and inspected. ";
			break;
		case 34:
			"A slouching plumber trundles by, snapping his chewing gum. ";
			break;
		case 35:
			"A tall man with a hard hat checks the crossbeams with a level. 
			\n\"Just about perfect,\" he intones. ";
			break;
		case 36:
			"A huge truck departs, diffusing the sunlight with its smoky
			exhaust. ";
			hugeTruck.moveInto( nil );
			break;
		case 37:
			"Two young boys scamper by, tossing a ball back and forth. Someone
			shouts, \"Hey! Get your kids off the site!\" ";
			break;
		case 38:
			"A voice cries, \"Higher! Higher! Keep going! Stop!\" ";
			break;
		case 39:
			"An old gent with nails clenched between his lips totters by, 
			carrying a selection of hammers. ";
			break;
		case 40:
			"A beautiful woman walks by, slips on a cable, and breaks one of
			her high heels.
			\n\"Ooohh!\" she yells, enraged. ";
			break;
		case 41:
			"Someone cries, \"Hammers! Hammers!\"
			\nAnother voice replies, \"Ol' Meretzsky just went by!\" ";
			break;
		case 42:
			"Four electricians push their way through, carrying cartons of
			wiring and sockets. ";
			break;
		case 43:
			"A voice yells, \"We need more of that one, and less of the
			other!\"
			\nSomeone responds, \"You mean like this?\"
			\nThe first voice shouts, \"No, the other way around, Artie!\" ";
			break;
		case 44:
			"A woman shouts, \"Has anyone seen Brian?\" ";
			break;
		case 45:
			"A noisy blast comes from the cement mixer. Several workers,
			clutching their heads, emit vain cries of protest. ";
			workers.messageNum := 0;
			break;
		default: workers.messageNum := 0;
			break;
	}
}

/*
 *	housekeeping():  Miscellaneous routines to run at end of turn, also 
 *	known as ugly hacks.
 */
housekeeping: function( parm )
{
	local temp;
	
	if ( global.money < 0 )
		global.money := 0;
	
	if ( Fred.state = 100 ) 	// to make score print at the right time
	{
		local FredLoc;
		
		"Fred says, \"Okay, I can find my way from here.\" He shakes 
		your hand. \"This is really heavy-duty, you know? But thanks.
		Thanks.\" ";
		
		if ( Fred.location = platform || Fred.location = platform2 )
			FredLoc := Fred.location.location;
		else
			FredLoc := Fred.location;
		
		if ( FredLoc.indoors )
		{
			"He starts to walk away, up the stairs and out the secret door. 
			You hear Fred mumble to himself, \"That's not what I want. That's 
			not the way I want my life to go.\" \b";
		}
		else
		{
			"He starts to walk away through the woods. You hear Fred
			mumble to himself, \"That's not what I want. That's not the way I
			want my life to go.\" \b
			
			Just before he disappears into the trees, he points up at the sky
			and calls out, \"Beautiful night, huh?\" ";
		}
		
		"With that, Fred is gone. ";
		
		Fred.state := 0;
		Fred.moveInto( nil );
		FgarageDoor1.setIsopen( nil );
		FgarageDoor1.setIslocked( nil );
		FFred.moveInto( nil );
	}
	
	if ( global.notifie != nil )			// from notify.t by G. Kevin Wilson
	{
		if ( global.lastScore = 0 ) 
		{
			return( nil );
		}
		
		if ( global.lastScore > 0 )
		{
			switch( global.lastScore )
			{
				case 1:
					"\b[ Your score just went up by <<global.lastScore>> point.\ ";
					break;
				default:
					"\b[ Your score just went up by <<global.lastScore>> points.\ ";
			}
		}
		
		if ( global.lastScore < 0 )
		{
			switch( global.lastScore ) 
			{
				case -1:
					"\b[ Your score just went down by <<global.lastScore*(-1)>> point.\ ";
					break;
				default:
					"\b[ Your score just went down by <<global.lastScore*(-1)>> points.\ ";
			}
		}
		
		if ( global.warned )
		{
			"] ";
		} 
	  	else 
		{
			" You can toggle these notifications at any time by typing 'notify.'\ ] ";
			global.warned := true;
		}
	}
	global.lastScore := 0;			// reset so we don't get a message next turn
}

/* deepGrab: function
	Takes an item recursively tucked away somewhere within the player's 
	possessions and moves it into the main inventory, opening all of the
	containers as it goes.
*/
deepGrab: function( obj )
{
	if ( obj.location = Me )
		return;

	"(first taking <<obj.thedesc>> out of <<obj.location.thedesc>>)\n";
	recursiveOpen( obj.location );
	"\b";
	obj.moveInto( Me );
}

recursiveOpen: function( box )
{
	if ( box = Me )
		return;
	
	if ( box.iscontainer && !box.isopen )
	{
		"(first opening <<box.thedesc>>)\n";
		box.isopen := true;
	}
	recursiveOpen( box.location );
}



/************************ END OF FUNCTION DEFINITIONS **********************/

/* eof */
