/*
 * chan_server.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.14 $
 * $Date: 1992/02/05 20:13:42 $
 */

#include "xkernel.h"
#include "chan_internal.h"

#ifdef __STDC__

static int	doCallDemux( XObj, CHAN_HDR *, Msg * );
static void	serverTimeout( void * );
static void 	getProcServer( XObj );

#else

static int	doCallDemux();
static void	serverTimeout();
static void 	getProcServer();

#endif __STDC__


/*
 * chanSvcOpen
 */
XObj 
chanSvcOpen(self, e, down_s, chan_num, prot_id)
    XObj 	self, down_s;
    Enable	*e;
    u_short 	chan_num;
    int 	prot_id;
{
    XObj   	s;
    CHAN_STATE 	*state;
    ActiveID 	ext_id;
    CHAN_HDR 	*hdr;
    PSTATE 	*pstate = (PSTATE *)self->state;
    
    xTrace0(chanp, 3, "CHAN svc open ..........................");
    
    ext_id.down_s    = down_s;
    ext_id.chan      = chan_num;
    ext_id.prot_id   = prot_id;
    ext_id.direction = SERVER;
    
    xTrace1(chanp, 3, "chanSvcOpen: down_s = %x", ext_id.down_s);
    xTrace1(chanp, 3, "chanSvcOpen: chan = %d", ext_id.chan);
    xTrace1(chanp, 3, "chanSvcOpen: prot_id = %d", ext_id.prot_id);
    /*
     * Is there an existing session?
     */
    s = (XObj) mapResolve(pstate->active_server_map, (char *) &ext_id);
    if ( s != ERR_XOBJ ) {
	xTrace0(chanp,3,"chanSvcOpen: channel exists ");
	return ERR_XOBJ;
    }
    /*
     * Need new session
     */
    state = X_NEW(CHAN_STATE);
    bzero((char *)state, sizeof(CHAN_STATE));
    /*
     * Fill in  state
     */
    state->cur_state 	   = SVC_IDLE; 
    state->direction 	   = SERVER;
    msg_clear(state->saved_msg);
    /*
     * Fill in header
     */
    hdr 	= &state->hdr;
    hdr->chan 	= ext_id.chan;
    hdr->flags 	= 0;
    hdr->prot_id= ext_id.prot_id;
    hdr->seq 	= 0;
    hdr->boot_id = 0;
    /*
     * Create session and bind to address
     */
    xDuplicate(down_s);
    s 		= xCreateSessn(getProcServer, e->hlpRcv, self, 1, &down_s);
    s->state 	= (char *) state;
    s->binding 	= (Bind) mapBind(pstate->active_server_map, (char *) &ext_id,
				 (int)s);
    /*
     * Just to be paranoid
     */
    if (s->binding == ERR_BIND) {
	xTrace0(chanp, 1, "chanSvcOpen: could not bind session"); 
	xTrace3(chanp, 1, "chanSvcOpen: down_s = %x, chan = %d, prot_id = %d",
		ext_id.down_s, (int)ext_id.chan, ext_id.prot_id);
	xClose(s);
	return ERR_XOBJ;
    }
    xOpenDone(s, e->hlpType);
    
    xTrace1(chanp, 3, "chanSvcOpen returns %x", s);
    return s;
}


/*
 * chan_openenable
 */
xkern_return_t
chanOpenEnable( self, hlpRcv, hlpType, p )
    XObj	self, hlpRcv, hlpType;
    Part 	*p;
{
    PassiveID 	key;
    PSTATE 	*pstate;
    Enable	*e;
    
    xTrace0(chanp, 3, "CHAN open enable .......................");
    pstate = (PSTATE *)self->state;
    if ( (key = chanGetProtNum(self, hlpType)) == -1 ) {
	return XK_FAILURE;
    }
    xTrace1(chanp, 3, "chan_openenable: prot_id = %d", key);
    e = (Enable *) mapResolve(pstate->passive_map, (char *) &key);
    if ( e != ERR_ENABLE ) {
	if ( e->hlpRcv == hlpRcv && e->hlpType == hlpType ) {
	    e->rcnt++;
	} else {
	    return XK_FAILURE;
	}
    } else {
	/*
	 * Bind server to high level protocol
	 */
	e = X_NEW(Enable);
	e->hlpType = hlpType;
	e->hlpRcv = hlpRcv;
	e->rcnt	= 1;
	e->binding = mapBind(pstate->passive_map, (char *) &key, (int) e);
	xAssert( e->binding != ERR_BIND );
    }
    return XK_SUCCESS;
}


xkern_return_t
chanOpenDisable( self, hlpRcv, hlpType, p )
    XObj	self, hlpRcv, hlpType;
    Part 	*p;
{
    PassiveID 	key;
    PSTATE 	*pstate;
    Enable	*e;
    
    xTrace0(chanp, TR_MAJOR_EVENTS,
	    "CHAN openDisable .......................");
    pstate = (PSTATE *)self->state;
    if ( (key = chanGetProtNum(self, hlpType)) == -1 ) {
	return XK_FAILURE;
    }
    xTrace1(chanp, TR_MAJOR_EVENTS, "chanOpenDisable: prot_id = %d", key);
    e = (Enable *) mapResolve(pstate->passive_map, (char *) &key);
    if ( e == ERR_ENABLE ) {
	xTrace0(chanp, TR_SOFT_ERRORS,
		"chanOpenDisable could not find enable object");
	return XK_FAILURE;
    }
    if ( e->hlpRcv != hlpRcv || e->hlpType != hlpType ) {
	xTrace0(chanp, TR_SOFT_ERRORS, "chanOpenDisable mismatch");
	return XK_FAILURE;
    }
    if ( --e->rcnt == 0 ) {
	mapRemoveBinding(pstate->passive_map, e->binding);
	xFree((char *)e);
    }
    return XK_SUCCESS;
}


/*
 * chanServerPop
 */
static xkern_return_t
chanServerPop(self, lls, msg)
    XObj self;
    XObj lls;
    Msg *msg;
{
    CHAN_STATE 	*state;
    CHAN_HDR	*hdr;
    SEQ_STAT 	status;
    
    xTrace0(chanp, TR_EVENTS, "CHAN pop ...............................");
    
    state = (CHAN_STATE *)self->state;
    hdr = (CHAN_HDR *)msgGetAttr(msg);
    if ( chanCheckMsgLen(hdr->len, msg) ) {
	/* 
	 * Message was too short -- probably mangled
	 */
	return XK_SUCCESS;
    }
    /* 
     * Check boot_id field
     */
    if ( hdr->boot_id != state->hdr.boot_id ) {
	if ( hdr->boot_id < state->hdr.boot_id ) {
	    xTrace0(chanp, 4, "chanServerPop -- received old boot ID");
	    return XK_SUCCESS;
	} else {
	    xTrace0(chanp, 4, "chanServerPop -- received new boot ID");
	    chanFreeResources(self);
	    state->cur_state = SVC_IDLE;
	    state->hdr.boot_id = hdr->boot_id;
	    state->hdr.seq = 0;
	}
    }
    /* 
     * Check sequence number
     */
    status = chanCheckSeq(state->hdr.seq, hdr->seq);
    if ( status == old ) {
	xTrace0(chanp, 4, "chanSrvrPop: old sequence number");
	return XK_SUCCESS;
    }
    if ( status == new ) {
	xTrace0(chanp, 4, "chanSrvrPop: new sequence number");
	state->hdr.seq = hdr->seq;
    }
    xTrace4(chanp, 4,
	    "state: %s  boot = %d, seq = %d  status: %s",
	    chanStateStr(state->cur_state), state->hdr.boot_id,
	    state->hdr.seq, chanStatusStr(status));
    xAssert(hdr->chan == state->hdr.chan);
    switch(state->cur_state) {
	
      case SVC_IDLE: 
	if ( hdr->flags & REQUEST ) {
	    if ( status == new ) {
		/*
		 * Fire up the server
		 */
		doCallDemux(self, hdr, msg);
	    } else {
		xTrace0(chanp, 3, "CHAN srvrPop: SVC_IDLE: old seq number"); 
	    }
	} else {
	    xTrace1(chanp, 1, "chan_pop: SVC_IDLE: wrong message type %x",
		    hdr->flags);
	}
	break;
	
      case SVC_WAIT:
	if ( status == new || hdr->flags & CLNT_EXPLICIT_ACK ) {
	    /*
	     * We can free resources
	     */
	    xTrace0(chanp, 5, "Server received ACK.  Freeing message");
	    chanFreeResources(self);
	    state->cur_state = SVC_IDLE;
	}
	if ( hdr->flags & REQUEST ) {
	    if ( status == new ) {
		/*
		 * New call -- Fire up the server
		 */
		doCallDemux(self, hdr, msg);
		return XK_SUCCESS;
	    } else {
		xAssert( status == current );
		/*
		 * Client apparently lost reply message -- retransmit
		 */
		xTrace0(chanp, 5,
			"chanPop: client lost reply -- retransmitting");
		if ( ! msg_isnull(state->saved_msg) ) {
		    Msg	packet;
		    XObj	lls = xGetDown(self, 0);
		    
		    state->hdr.flags = REPLY;
		    msgConstructCopy(&packet, &state->saved_msg.m);
		    msgPush(&packet, chanHdrStore, &(state->hdr), CHANHLEN, 0);
		    xAssert(xIsSession(lls));
		    state->ticket = xPush(lls, &packet);
		    msgDestroy(&packet);
		} else {
		    xTrace0(chanp, 5, "chan_pop: saved message is null");
		}
		/* We won't send ack back to client. If we are retransmiting
		 * the server message, then it is not needed. If we lost the
		 * server message, what is the point of the ack.
		 */
	    }
	}
	break;

      case SVC_EXECUTE:
	if ( hdr->flags & REQUEST ) {
	    if ( status == new ) {
		xTrace0(chanp, 4, "chan_pop: SVC_EXECUTE: new seqence number");
		/* 
		 * We're going to give up on the old message (sequence
		 * number has changed in our state and the reply to the old
		 * message will not return)
		 */
		doCallDemux(self, hdr, msg);
	    } else {
		xTrace0(chanp, 4, "chan_pop: SVC_EXECUTE: duplicate request");
		/*
		 * Send ack if requested
		 */
		if (hdr->flags & ACK_REQUESTED) {
		    chanReply(xGetDown(self, 0), hdr, SVC_EXPLICIT_ACK);
		}
	    }
	} else {
	    xTrace1(chanp, 4, "chan_pop: SVC_EXECUTE wrong message type %x",
		    hdr->flags);
	}
	break;
	
      default:
	xTrace1(chanp,1,"chan_pop: invalid state  %d",  state->cur_state);
	break;
    } 
    return XK_SUCCESS;
}


/* 
 * serverTimeout: server timout detects inactive channels and asks for ack 
 */
static void
serverTimeout(arg)
    VOID *arg;
{
    TimeoutState	*tos = (TimeoutState *)arg;
    CHAN_STATE	*state;
    Msg 	msg;
    CHAN_HDR 	hdr;
    XObj	lls;
    
    state = (CHAN_STATE *)tos->ses->state;
    xTrace0(chanp, 2, "CHAN: serverTimeout: timeout!");
    
    if ( state->cur_state == SVC_WAIT && state->hdr.seq == tos->seq ) {
	/* 
	 * This event is still relevant
	 */
	if ( --state->tries == 0 ) {
	    /* 
	     * Give up and free server resources
	     */
	    xTrace0(chanp, 2, "CHAN server timeout gives up");
	    msg_flush(state->saved_msg);
	    state->cur_state = SVC_IDLE;
	} else {
	    /* 
	     * Retry
	     */
	    hdr 	= state->hdr;
	    hdr.flags 	= PROBE | ACK_REQUESTED; 
	    hdr.len 	= 0;
	    state->wait = MIN(2*state->wait, MAX_SERVER_WAIT);
	    
	    msgConstructEmpty(&msg);
	    msgPush(&msg, chanHdrStore, &hdr, CHANHLEN, NULL);
	    lls = xGetDown(tos->ses, 0);
	    xAssert(xIsSession(lls));
	    xPush(lls, &msg);
	    msgDestroy(&msg);
	    state->event = evSchedule(serverTimeout, tos, state->wait);
	    return;
	}
    }
    xFree((char *)tos);
    /* 
     * Remove reference count corresponding to this event
     */
    xClose(tos->ses);
    return;
}


/*
 * doCallDemux
 */
static int
doCallDemux(self, inHdr, inMsg)
    XObj self;
    CHAN_HDR *inHdr;
    Msg *inMsg;
{
    CHAN_STATE	*state;
    CHAN_HDR	*hdr;
    Msg		rmsg;
    int		packet_len;
    XObj	lls;
    SeqNum	oldSeq;
    u_int	oldBootId;
    
    state = (CHAN_STATE *)self->state;
    state->cur_state = SVC_EXECUTE;
    hdr = &state->hdr;
    lls = xGetDown(self, 0);
    xAssert(xIsSession(lls));
    /*
     * Send ack if requested
     */
    if ( inHdr->flags & ACK_REQUESTED ) {
	chanReply( lls, inHdr, SVC_EXPLICIT_ACK );
    }
    xTrace1(chanp, 4, "chan_pop: incoming length (no chan hdr): %d",
	    msgLen(inMsg));
    /* 
     * Save current values of boot_id and the sequence number.  We
     * will not send the reply if these values have changed by the
     * time the reply is ready.
     */
    oldBootId = state->hdr.boot_id;
    oldSeq = state->hdr.seq;
    msgConstructEmpty(&rmsg);
    xCallDemux(self, inMsg, &rmsg);
    xTrace0(chanp, 4, "chan_pop: xCallDemux returns");
    xTrace1(chanp, 4, "chan_pop: (SVC) outgoing length: %d", msgLen(&rmsg));
    if ( oldBootId != state->hdr.boot_id || oldSeq != state->hdr.seq ) {
	xTrace0(chanp, 3, "CHAN -- reply no longer relevant.");
	msgDestroy(&rmsg);
	return 0;
    }
    msgConstructCopy(&state->saved_msg.m, &rmsg);
    msg_valid(state->saved_msg);
    /*
     * Fill in reply header
     */
    hdr->flags 	= REPLY;
    hdr->len 	= msgLen(&rmsg); 
    
    xIfTrace(chanp, 4) { 
	pChanHdr(hdr);
    } 
    msgPush(&rmsg, chanHdrStore, hdr, CHANHLEN, NULL);
    
    xTrace1(chanp, 4, "chan_pop: packet len %d", msgLen(&rmsg)); 
    xTrace1(chanp, 4, "chan_pop: length field: %d", hdr->len);
    xTrace2(chanp, 4, "chan_pop: down_s %x packet = %x", xGetDown(self, 0),
	    &rmsg); 
    /*
     * Send reply
     */
    packet_len    = msgLen(&rmsg); 
    state->ticket = xPush(lls, &rmsg);
    msgDestroy(&rmsg);
    if (state->ticket < 0) {
	xTrace0(chanp, 0, "chan_pop: (SVC) can't send message");
	msg_flush(state->saved_msg);
	return -1;
    }
    state->cur_state = SVC_WAIT;
    state->wait      = CHAN_SVC_DELAY(packet_len);
    state->tries = SERVER_TRIES;
    xTrace1(chanp, 9, "chan_pop: (SVC) server_wait: %d", state->wait);
    state->evState = X_NEW(TimeoutState);
    state->evState->ses = self;
    state->evState->seq = hdr->seq;
    /* 
     * Add reference count for the timeout event
     */
    xDuplicate(self);
    state->event     = evSchedule(serverTimeout, state->evState, state->wait);
    
    return 0;
}


static void 
getProcServer(s)
    XObj s;
{
    xAssert(xIsSession(s));
    s->close 	= chanCloseSessn;
    s->pop 	= chanServerPop; 
    s->control 	= chanControlSessn;
}
