/*
 * chan_client.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.13 $
 * $Date: 1992/02/05 20:13:41 $
 */

#include "xkernel.h"
#include "chan_internal.h"

#ifdef __STDC__

static xkern_return_t	chanCall( XObj, Msg *, Msg * );
static xkern_return_t	chanClientPop( XObj, XObj, Msg * );
static void	clientTimeout( void * );
static int	getBootId( void );
static void 	getProcClient( XObj );

#else

static xkern_return_t	chanCall();
static xkern_return_t	chanClientPop();
static void	clientTimeout();
static int	getBootId();
static void 	getProcClient();

#endif  __STDC__


/* 
 * chanOpen
 */
XObj 
chanOpen( self, hlpRcv, hlpType, p )
    XObj self, hlpRcv, hlpType;
    Part *p;
{
    XObj   	s;
    CHAN_STATE 	*state;
    ActiveID 	ext_id;
    CHAN_HDR 	*hdr;
    PSTATE 	*pstate = (PSTATE *)self->state;
    
    xTrace0(chanp, TR_MAJOR_EVENTS,
	    "CHAN open ..............................");
    if ( (ext_id.prot_id = chanGetProtNum(self, hlpType)) == -1 ) {
	return ERR_XOBJ;
    }
    /*
     * Open lower session
     */
    ext_id.down_s = xOpen(self, self, xGetDown(self, 0), p);
    if ( ext_id.down_s == ERR_XOBJ) {
	xTrace0(chanp, TR_SOFT_ERRORS,
		"chan_open: can't open lower session");
	return ERR_XOBJ;
    }
    ext_id.chan = pstate->channel_number++;
    ext_id.direction = CLIENT;
    
    xTrace1(chanp, TR_MAJOR_EVENTS, "chan_open: chan = %d", ext_id.chan);
    xTrace1(chanp, TR_MAJOR_EVENTS, "chan_open: prot_id = %d", ext_id.prot_id);
    xTrace1(chanp, TR_MAJOR_EVENTS, "chan_open: down_s = %x", ext_id.down_s);
    /* 
     * We always use a new channel number, so this will be a unique
     * new session  
     */
    state = X_NEW(CHAN_STATE);
    bzero((char *)state, sizeof(CHAN_STATE));
    /*
     * Fill in  state
     */
    state->cur_state 	= CLNT_FREE;
    state->direction 	= CLIENT;
    msg_clear(state->saved_msg);
    semInit(&state->reply_sem, 0);
    /*
     * Fill in header
     */
    hdr	= &state->hdr;
    hdr->chan = ext_id.chan;
    hdr->flags = 0;
    hdr->seq = START_SEQ;
    hdr->prot_id = ext_id.prot_id;
    hdr->boot_id = getBootId();
    /*
     * Create session and bind to address
     */
    s = xCreateSessn(getProcClient, hlpRcv, self, 1, &ext_id.down_s);
    s->state 	= (char *) state;
    s->binding 	= (Bind) mapBind(pstate->active_client_map, (char *) &ext_id, 
				 s);
    /*
     * Just to be paranoid
     */
    if (s->binding == ERR_BIND) {
	xTrace0(chanp, TR_ERRORS, "chan_open: could not bind session");
	xClose(s);
	return ERR_XOBJ;
    }
    
    xTrace1(chanp, TR_MAJOR_EVENTS, "chan_open returns %x", s);
    return(s);
}


/* 
 * chanCall
 */
static xkern_return_t
chanCall(self, msg, rmsg)
    XObj self;
    Msg *msg;
    Msg *rmsg;
{
    CHAN_STATE   	*state;
    CHAN_HDR 	*hdr;
    int 		packet_len;
    TimeoutState	*tos;
    XObj		lls;
    
    xTrace0(chanp, TR_EVENTS, "CHAN call ..............................");
    
    state = (CHAN_STATE *)self->state;
    xTrace1(chanp, TR_MORE_EVENTS, "chan_call: state = %x", state);
    xTrace1(chanp, TR_MORE_EVENTS,
	    "chan_call: outgoing length (no chan hdr): %d", msgLen(msg));
    
    if ((state->cur_state != CLNT_FREE)) {
	xTrace0(chanp, TR_SOFT_ERRORS, "chan_call: incorrect initial state");
	return XK_FAILURE;
    }
    
    msg_flush(state->saved_msg);
    /*
     * Save data (without header) for retransmit
     */
    msgConstructCopy(&state->saved_msg.m, msg);
    msg_valid(state->saved_msg);
    
    /*
     * Fill in header
     */
    hdr 		= &state->hdr;
    hdr->flags 	= REQUEST;
    hdr->len 	= msgLen(msg);
    
    xIfTrace(chanp, TR_MORE_EVENTS) { 
	pChanHdr(hdr);
    } 
    
    msgPush(msg, chanHdrStore, hdr, CHANHLEN, NULL);
    
    lls = xGetDown(self, 0);
    xAssert(xIsSession(lls));
    xTrace1(chanp, TR_MORE_EVENTS, "chan_call: s->push  = %x", lls->push); 
    xTrace1(chanp, TR_MORE_EVENTS, "chan_call: packet len %d", msgLen(msg)); 
    xTrace1(chanp, TR_MORE_EVENTS, "chan_call: length field: %d", hdr->len);
    xTrace2(chanp, TR_MORE_EVENTS, "chan_call: down_s %x, packet = %x",
	    lls, msg); 
    state->answer = rmsg;
    state->replyValue = SUCCESS;
    /*
     * Send message
     */
    packet_len    = msgLen(msg);
    state->ticket = xPush(lls, msg);
    if (state->ticket < 0) {
	xTrace0(chanp, TR_ERRORS, "chan_call: can't send message");
	msg_flush(state->saved_msg);
	return XK_FAILURE;
    }
    state->cur_state 	= CLNT_WAIT;
    state->wait 		=  CHAN_CLNT_DELAY(packet_len);
    state->tries 		=  CLIENT_TRIES;
    xTrace2(chanp, TR_DETAILED,
	    "chan_call: client_wait = %d, client_tries =%d", 
	    state->wait, state->tries);
    state->evState = tos = X_NEW(TimeoutState);
    tos->ses = self;
    tos->seq = state->hdr.seq;
    /* 
     * Add reference count for the timeout event
     */
    xDuplicate(self);
    state->event = evSchedule(clientTimeout, tos, state->wait);
    /*
     * Wait for reply
     */
    semWait(&state->reply_sem);
    state->hdr.seq++;
    msg_flush(state->saved_msg);
    return state->replyValue == SUCCESS ? XK_SUCCESS : XK_FAILURE;
}


/*
 * chanClientPop
 */
static xkern_return_t
chanClientPop(self, lls, msg)
    XObj self;
    XObj lls;
    Msg *msg;
{
    CHAN_STATE 	*state;
    CHAN_HDR	*hdr;
    u_int 	seq;
    SEQ_STAT 	status;
    
    xTrace0(chanp, TR_EVENTS, "CHAN Client Pop");
    state = (CHAN_STATE *)self->state;
    hdr = (CHAN_HDR *)msgGetAttr(msg);
    seq = hdr->seq;
    status = chanCheckSeq(state->hdr.seq, seq);
    xTrace4(chanp, TR_MORE_EVENTS,
	    "state: %s  cur_seq = %d, hdr->seq = %d  status: %s",
	    chanStateStr(state->cur_state), state->hdr.seq, hdr->seq,
	    chanStatusStr(status));
    if ( chanCheckMsgLen(hdr->len, msg) ) {
	return XK_SUCCESS;
    }
    
    xAssert(hdr->chan == state->hdr.chan);
    switch(state->cur_state) {
	
      case CLNT_WAIT:
	if (hdr->boot_id != state->hdr.boot_id) {
	    xTrace0(chanp, TR_SOFT_ERRORS,
		    "chan_pop: CLNT_WAIT: wrong boot_id");
	    return XK_SUCCESS;
	}
	if (status != current) {
	    xTrace0(chanp, TR_SOFT_ERRORS,
		    "chan_pop: CLNT_WAIT: wrong seqence number");
	    return XK_SUCCESS;
	}
	if (hdr->flags & REPLY) {
	    xTrace0(chanp,4,"chan_pop: CLNT_WAIT: Received reply"); 
	    chanFreeResources(self);
	    /*
	     * Return results
	     */
	    msgAssign(state->answer, msg);
	    semSignal(&state->reply_sem);
	    state->cur_state = CLNT_FREE;
	    xTrace0(chanp, TR_MORE_EVENTS,
		    "chan_pop: CLNT_WAIT: returning OK");
	    return XK_SUCCESS;
	}
	if (hdr->flags & SVC_EXPLICIT_ACK) {
	    xTrace0(chanp, TR_MORE_EVENTS,
		    "chan_pop: CLNT_WAIT: received SVC_EXPLICIT_ACK");
	    return XK_SUCCESS;
	}
	
	xTrace1(chanp, TR_ERRORS, "chan_pop: CLNT_WAIT wrong message type %x",
		hdr->flags);
	return XK_SUCCESS;
	break;
	
      case CLNT_FREE:
	if (hdr->boot_id != state->hdr.boot_id) {
	    xTrace0(chanp, TR_SOFT_ERRORS,
		    "chan_pop: CLNT_FREE: wrong boot_id");
	    return XK_SUCCESS;
	}
	
	if (hdr->flags & PROBE) {
	    if (hdr->flags & ACK_REQUESTED) {
		/* 
		 * Send ACK's only for sequence numbers we have seen
		 */
		if ( status != new ) {
		    xTrace0(chanp, TR_MORE_EVENTS,
			    "Client sending explicit ACK");
		    chanReply(xGetDown(self, 0), hdr, CLNT_EXPLICIT_ACK);
		} else {
		    xTrace0(chanp, TR_MORE_EVENTS,
			    "chanClientPop ACK requested for new seq number!");
		}
	    }
	    return XK_SUCCESS;
	}
	if (hdr->flags & REPLY) {
	    xTrace0(chanp, TR_MORE_EVENTS,
		    "chan_pop: CLNT_FREE ignoring old reply");
	    return XK_SUCCESS;
	}
	xTrace1(chanp, TR_ERRORS, "chan_pop: CLNT_FREE wrong message type %x",
		hdr->flags);
	return XK_SUCCESS;
	break;
	
     default:
	xTrace1(chanp, TR_ERRORS,
		"chan_pop: invalid state  %d",  state->cur_state);
	return XK_SUCCESS;
	break;
    } 
    return XK_SUCCESS;
}


/*
 * clientTimeout: exponential backoff retry 
 */
static void
clientTimeout(arg)
    VOID *arg;
{
    TimeoutState	*tos = (TimeoutState *)arg;
    Msg 	packet;
    CHAN_STATE	*state;
    XObj	lls;
    
    xTrace0(chanp, TR_EVENTS, "chan: clientTimeout called!");
    state = (CHAN_STATE *)tos->ses->state;
    if ( state->hdr.seq != tos->seq || state->cur_state != CLNT_WAIT ) {
	/* 
	 * This timeout event is no longer relevant
	 */
	xTrace0(chanp, TR_EVENTS, "clientTimeout: spurious timeout");
    } else { 
	if ( --state->tries == 0 ) {
	    /*
	     * Call fails
	     */
	    xTrace0(chanp, TR_SOFT_ERRORS, "clientTimeout: call fails!");
	    /*
	     * Notify client of bad news
	     */
	    state->replyValue = FAILURE;
	    semSignal(&state->reply_sem);
	} else {
	    /*
	     * Retry
	     */
	    state->hdr.flags = REQUEST | ACK_REQUESTED; 
	    xAssert( ! msg_isnull(state->saved_msg));
	    msgConstructCopy(&packet, &state->saved_msg.m);
	    msgPush(&packet, chanHdrStore, &state->hdr, CHANHLEN, NULL);
	    /*
	     * Send message
	     */
	    lls = xGetDown(tos->ses, 0);
	    xAssert(xIsSession(lls));
	    state->ticket = xPush(lls, &packet);
	    msgDestroy(&packet);
	    /*
	     * Exponential backoff
	     */
	    state->wait  = state->wait*2; 
	    state->event = evSchedule(clientTimeout, tos, state->wait);
	    return;
	}
    }
    xFree((char *)tos);
    /* 
     * Remove the reference count corresponding to this event
     */
    xClose(tos->ses);
    return;
}
    

/*
 * getBootId
 */
static int
getBootId()
{
    XTime t;
    
    xGetTime(&t);
    return (int) t.sec;
}


static void 
getProcClient(s)
    XObj s;
{
    xAssert(xIsSession(s));
    s->close 	= chanCloseSessn;
    s->call 	= chanCall;
    s->pop 	= chanClientPop; 
    s->control 	= chanControlSessn;
}


