/* 
 * ms_memory.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "process.h"
#include "memory.h"
#include "assert.h"

/* Globals */
int  freePageList;	/* list head */
Pmeg freePmegList;
char *StackBottom;

Pmeg PmegMap[NUM_PMEGS];
struct pageInfo {
  Bit32 	use	:2;
  Bit32 		:11;
  Bit32		next	:19;
} PageMap[MAXPAGES];

int FreePages = 0, FreePmegs = 0;
TimeRecord Time;

char *FindKernelMapAddress();
char *FindUserMapAddress();

/*********************************************************
* Machine specific (Ms_) routines; called from memory.c
*********************************************************/

Ms_InitMemory()
/*
 * Setup the memory mapping as required.
 *
 * We assume the EPROM monitor has initialized the memory map for
 * context 0 in its standard way.  The kernel is
 * assumed to have been put at its normal address (0x4000) by boot.
 * and the pages and segments it is loaded into are
 * assumed to be mapped 1-to-1, that is, the virtual and physical
 * addresses are the same, and the pmeg and segment numbers are the same.
 *
 * We assign the kernel virtual addresses between 0xf004000 and whatever in
 * all contexts.  Only pages that are actually used are mapped to
 * physical memory.
 *
 * The area between MONITOR_START and MONITOR_END, and the single page at
 * MONITOR_SHORTPAGE is mapped in a complex way
 * by the ROM monitor, as described in /usr/include/mon/sunromvec.h.  We are
 * not to change these if we expect any support from the monitor, except
 * that we are allowed to change the PMEG used to map MONITOR_SHORTPAGE.
 */
{
  int pages;
  int ctx, i;
  PageMapEntry pme;
  Pmeg sme;
  VirtualAddress a, ktop;
  extern int end;

  StackBottom = (char *)FetchSP() - 1024;

  ktop.u = (unsigned) &end;
  SetContext(0);	/* just to be sure */
  pages = GetMemorySize();

  TRACE1(memoryinit, 1, "Memory size: 0x%x pages.", pages);

  /* Initialize page and pmeg stuff */
  initPageMap();
  initPmegMap();

  TRACE1(memoryinit, 1, "Kernel size in bytes: %d", ktop.u - KERNEL_START);
  if (ktop.u >= KERNEL_LIMIT) Kabort("Kernel too big");

  /* Set page protection on kernel memory appropriately */
  TRACE2(memoryinit, 1, "Setting protection on kernel memory 0x%x - 0x%x",
    KERNEL_START - PAGE_SIZE, ktop.u);
  for (a.u = KERNEL_START-PAGE_SIZE; a.u < ktop.u; a.u += PAGE_SIZE) {
    pme.u = GetPageMap(a.u);
    pme.f.protection = SUPER_ONLY|PAGE_WRITE;
    SetPageMap(a.u, pme.u);
    claimPage(pme.f.pagenum, USE_KERNEL);
    if (a.u == KERNEL_START || a.u % SEG_SIZE == 0) claimPmeg(GetSegMap(a.u));
  }

  /* The ROM monitor sets the stack to 0x2000, I change it to 0xf002000

  /* Invalidate the pages at the beginning and end of the kernel */
  TRACE2(memoryinit, 1, "Invalidating beginning of kernel text 0x%x - 0x%x",
    downtoseg(KERNEL_START), KERNEL_START - PAGE_SIZE);
  for (i = downtoseg(KERNEL_START); i<KERNEL_START-PAGE_SIZE; i += PAGE_SIZE) {
    SetPageMap(i, INVALID_PTE);
  }
  TRACE2(memoryinit, 1, "Invalidating end of kernel text/data 0x%x - 0x%x",
    uptopage(ktop.u), uptoseg(ktop.u));
  for (i = uptopage(ktop.u); i < uptoseg(ktop.u); i += PAGE_SIZE) {
    SetPageMap(i, INVALID_PTE);
  }
  TRACE2(memoryinit, 1, "Invalidating kernel end segments 0x%x - 0x%x",
    uptoseg(ktop.u), KERNEL_LIMIT);
  for (i = uptoseg(ktop.u); i < KERNEL_LIMIT; i += SEG_SIZE) {
    SetSegMap(i, NULL_PMEG);
  }

  /* Initialize the segment maps for non-zero contexts, and the segment map
   *   for user portion of context 0.  We copy in the
   *   context 0 mapping for the kernel and the area above ASPACE_LIMIT,
   *   and set the rest to NULL_PMEG.
   */
  TRACE0(memoryinit, 1, "Initializing segment maps");
  TRACE0(memoryinit, 1, "  Kernel area and above");
  for (a.u = downtoseg(KERNEL_START); a.u < CONTEXT_SIZE; a.u += SEG_SIZE) {
    sme = GetSegMap(a.u);
    for (ctx = 1; ctx < NUM_CONTEXTS; ctx++) {
      SetContext(ctx);
      SetSegMap(a.u, sme);
    }
    SetContext(0);
  }
  TRACE0(memoryinit, 1, "  Address space area");
  for (a.u = downtoseg(ASPACE_START); a.u < ASPACE_LIMIT; a.u += SEG_SIZE) {
    for (ctx = 0; ctx < NUM_CONTEXTS; ctx++) {
      SetContext(ctx);
      SetSegMap(a.u, NULL_PMEG);
    }
  }
  SetContext(0);

  /* Claim the Prom pages and pmegs. */
  TRACE2(memoryinit, 1, "Claiming prom pmegs and pages %X - %X",
    MONITOR_START, MONITOR_END);
  for (a.u = MONITOR_START; a.u < MONITOR_END; a.u += PAGE_SIZE) {
    if (a.u % SEG_SIZE == 0) claimPmeg(GetSegMap(a.u));
    pme.u = GetPageMap(a.u);
    if (pme.f.valid && pme.f.type == ONBOARD_MEM && pme.f.pagenum < MAXPAGES) 
      claimPage(pme.f.pagenum, USE_PROM);
  }
  TRACE1(memoryinit, 1, "Claiming prom short page %X", MONITOR_SHORTPAGE);
  claimPmeg(GetSegMap(MONITOR_SHORTPAGE));
  pme.u = GetPageMap(MONITOR_SHORTPAGE);
  if (pme.f.valid && pme.f.type == ONBOARD_MEM) {
    claimPage(pme.f.pagenum, USE_PROM);
  }

  /* Set page protection on high memory */
  TRACE2(memoryinit, 1, "Setting protection on high memory, 0x%x - 0x%x",
    KERNEL_LIMIT, CONTEXT_SIZE);
  for (a.u = KERNEL_LIMIT; a.u < CONTEXT_SIZE; a.u += PAGE_SIZE) {
    pme.u = GetPageMap(a.u);
    if (pme.u == INVALID_PTE) continue;
    pme.f.protection |= SUPER_ONLY;
    SetPageMap(a.u, pme.u);
  }

  /* Unprotect the frame buffer */
  for (a.u = FBUFFER; a.u < FBUFFER + FBUFFERSIZE; a.u += PAGE_SIZE) {
    pme.u = GetPageMap(a.u);
    pme.f.protection &= ~SUPER_ONLY;
    SetPageMap(a.u, pme.u);
  }

  /* Claim the color display pmegs (if present) */
  if (GetSegMap(CFBUFFER) != NULL_PMEG) {
    TRACE0(memoryinit, 1, "Claiming pmegs for color buffer");
    for (a.u = CFBUFFER; a.u < CFBUFFER + CFBUFFERSIZE; a.u += SEG_SIZE) {
      sme = GetSegMap(a.u);
      if (sme != NULL_PMEG) claimPmeg(sme);
    }
  }

  initFreePmegList();
  initFreePageList(pages);

  TRACE1(memoryinit, 1, "Total free pages = %d", FreePages);
  TRACE1(memoryinit, 1, "Total free pmegs = %d", FreePmegs);
  CheckOutMemory();
  TRACE0(memoryinit, 1, "Memory initialization done");
}

Ms_AllocateADs()
{
  unsigned adMemory;
  
  adMemory = min( ( (4*1024*1024) >> 6 ), 0x10000 );

  AspaceDescriptors = (Aspace *) malloc(adMemory);
  MaxAspaces = adMemory/sizeof( Aspace );
  TRACE2(memoryinit, 1, "  Allocating %d aspace descriptors (%d bytes).",
    MaxAspaces, adMemory);
}

Ms_CreateAspace( as ) 
register Aspace *as;
/*
 * Perform the machine-specific portion of aspace creation.
 */
{
  as->aspace_space.size = (char *)ASPACE_START;
  Ms_SetAspaceSize(as, (char *) 0);	/* Be sure aspace is null */
}

Ms_InitAs( as, i ) 
register Aspace *as; 
register unsigned i;
/*
 * Perform machine-specific initialization of aspace descriptors
 */
{
  if (i < NUM_CONTEXTS) {
    as->aspace_space.context = i;
  } else {
    SetContext(1);
    SaveSegMap(as->aspace_space.segmap);
    SetContext(0);
    as->aspace_space.context = MAP_NOT_RESIDENT;
  }
  as->aspace_space.size = (char *) ASPACE_START; /* Set the initial size */
}

Ms_FreePage(addr)
register unsigned addr;
{
  PageMapEntry freepg;
  TRACE1(user, 2, "Free page at %x", addr);
  if (addr > CONTEXT_SIZE) {
    printf("Free page %X\n", addr);
    Kabort("Invalid virtual address");
  }
  freepg.u = GetPageMap(addr);
  if (freepg.f.valid) {
    freePage(freepg.f.pagenum);
    SetPageMap(addr, INVALID_PTE);
  }
}

/* Very rudimentary. All of the memory BETTER BE ALLOCATED. */
Ms_FreeMemory( thestart, length )
unsigned thestart;
unsigned length;
{
  register unsigned ptr, theend;

  theend = thestart + length;

  for ( ptr = thestart; ptr < theend; ptr += PAGE_SIZE ) freePage( ptr );
}

Ms_SetAspaceSize(ad, size)
register Aspace *ad;
char *size;
{
  register VirtualAddress firstUnusedPage, newFirstUnusedPage;
  register int fail;
  Aspace *oldaspace;

  /* Range check on size */
  if (size < (char *) ASPACE_START) size = (char *) ASPACE_START;
  if (size >= (char *) ASPACE_LIMIT) size = (char *) ASPACE_LIMIT;

  /* Temporarily switch contexts */
  oldaspace = Ms_GetAddressableAspace();
  Ms_SetAddressableAspace(ad);


  firstUnusedPage.u = uptopage(ad->aspace_space.size);
  newFirstUnusedPage.u = uptopage(size);

  fail = 0;
  if (firstUnusedPage.u <= newFirstUnusedPage.u) {
    /* More memory needed */
    for ( ; firstUnusedPage.u < newFirstUnusedPage.u;
	    firstUnusedPage.u += PAGE_SIZE ) {
      if (fail = AllocatePage(firstUnusedPage.u, FALSE)) break;
    }
  } else {
    /* Less memory needed */
    for (firstUnusedPage.u -= PAGE_SIZE;
	 firstUnusedPage.u >= newFirstUnusedPage.u;
	 firstUnusedPage.u -= PAGE_SIZE) {
      Ms_FreePage(firstUnusedPage.u);
      /*
       * We are assuming the simple memory model in which
       * address spaces do not have holes, so if we free the
       * first page in a pmeg here, we can free the pmeg.
       */
      if (firstUnusedPage.u == uptoseg(firstUnusedPage.u)) {
	FreePmegAt(firstUnusedPage.u);
	SetSegMap(firstUnusedPage.u, NULL_PMEG);
      }
    }
  }

  Ms_SetAddressableAspace(oldaspace);

  if (fail) {
    ad->aspace_space.size = firstUnusedPage.p;
  } else {
    ad->aspace_space.size = size;
  }
  TRACE2(memoryinit, 3, "aspace %x size set to %x",ad,ad->aspace_space.size);
}

/*
 * Routines to set/get the currently addressable aspace space.
 * The Sun-3 only has 8 sets of segment mapping registers, so if
 * we have more than 8 aspaces, we have to do LRU swapping of the
 * segment maps into and out of registers.
 */

/* Ms_GetAddressableAspace() is now a macro */

Ms_SetAddressableAspace(newaspace)	/* mostly machine independent */
register Aspace *newaspace;
{
  register Aspace *ad, *oldest;

  /* see if already in that aspace */
  if (newaspace == AddressableAspace)
    return;
  /* Bring in segment map if not resident */
  if (newaspace->aspace_space.context == MAP_NOT_RESIDENT) {
    /* Find the LRU context... */
    oldest = NULL;
    for (ad = AspaceDescriptors; ad < (AspaceDescriptors + MaxAspaces); ad++) {
      if ( ad->aspace_space.context != MAP_NOT_RESIDENT &&
	    (oldest == NULL ||
	     ad->aspace_space.timestamp < oldest->aspace_space.timestamp) )
	oldest = ad;
    }

    /* ...and swap out its segment map */
    newaspace->aspace_space.context = oldest->aspace_space.context;
    oldest->aspace_space.context = MAP_NOT_RESIDENT;
    SetContext(newaspace->aspace_space.context);
    SaveSegMap(oldest->aspace_space.segmap);
    RestoreSegMap(newaspace->aspace_space.segmap);
  } else {
    SetContext(newaspace->aspace_space.context);
  }

  AddressableAspace->aspace_space.timestamp = Time.seconds;
  AddressableAspace = newaspace;
#ifdef MC68020
  flushCache();
#endif MC68020
}

char *Ms_MapPage(oldaddr, newaddr, prot)
char *oldaddr, *newaddr;
{
  PageMapEntry oldpg;
  int kernel = prot & SUPER_ONLY;
  int x = Ms_Spl7();

  oldpg.u = GetPageMap(oldaddr);
  oldpg.f.protection = prot;
  if (newaddr == 0) {
    newaddr = kernel ? 
      FindKernelMapAddress(oldpg.f.pagenum) :
      FindUserMapAddress(oldpg.f.pagenum);
  }

  /* Make sure there is a pmeg mapped to this segment */
  if (GetSegMap(newaddr) == NULL_PMEG) {
    if (AllocatePmeg(newaddr, kernel)) {
      Kabort("Ms_MapPage: Can't get a pmeg\n");
    }
  }
  
  SetPageMap((unsigned)newaddr, oldpg.u);
  SetPageMap((unsigned)oldaddr, INVALID_PTE);
  Ms_Splx(x);
  return newaddr;
}

char *Ms_AllocateKernelPage()	/* really machine independent */
{
  int x = Ms_Spl7();
  int thepagenumber = freePageList;
  unsigned theaddress;

  if (thepagenumber == NULL_PAGE) {
    Kabort("Allocate kernel page: free list is empty\n");
    Ms_Splx(x);
  }
  theaddress = (unsigned)FindKernelMapAddress(thepagenumber);
  AllocatePage(theaddress, 1);
  Ms_Splx(x);
  return (char *) theaddress;
}

char *Ms_AllocateUserPage()	/* really machine independent */
{
  int x = Ms_Spl7();
  int thepagenumber = freePageList;
  unsigned theaddress;

  if (thepagenumber == NULL_PAGE) {
    Kabort("Allocate user page: free list is empty\n");
    Ms_Splx(x);
  }
  theaddress = (unsigned)FindUserMapAddress(thepagenumber);
  AllocatePage(theaddress, 0);
  Ms_Splx(x);
  return (char *) theaddress;
}

/*********************************************************
* Internal routines
**********************************************************/

AllocatePage(pgaddr, kernel)
register unsigned pgaddr;
int kernel;
/*
 * Map a page into the current context at the given address, if one
 *  can be found.  Return OK for success.
 */
{
  PageMapEntry newpg;
  int x = Ms_Spl7();
  TRACE3(processcreation, 1, "Allocating page %x at %x%s", freePageList, 
    pgaddr, kernel?" for kernel":"");
/*  assert(pgaddr <= CONTEXT_SIZE); */
  /* Find a page, if available */
  if ( (newpg.f.pagenum = freePageList) == NULL_PAGE ) {
    printf("Allocate page: free list is empty\n");
    Ms_Splx(x);
    return -1;
  }

  /* Make sure there is a pmeg mapped to this segment */
  if (GetSegMap(pgaddr) == NULL_PMEG) {
    if (AllocatePmeg(pgaddr, kernel)) {
      printf("AllocatePage: Can't get a pmeg\n");
      Ms_Splx(x);
      return -1;
    }
  }

  /* Map in the page */
  newpg.f.valid = 1;
  newpg.f.type = ONBOARD_MEM;
  newpg.f.protection = kernel ? PAGE_WRITE | SUPER_ONLY : PAGE_WRITE;
  SetPageMap(pgaddr, newpg.u);

  /* Remove it from the free list */
  freePageList = PageMap[freePageList].next;
  FreePages--;
  Ms_Splx(x);
  return 0;
}

/* Allocate physical memory. This routine is meant to be called by the kernel
 * to allocate memory for such uses as process decriptor space, ethernet
 * buffer space, and profiling data space. Is does not check to see if the
 * memory has already been allocated.
 */
AllocateMemory( thestart, length )
unsigned thestart;
unsigned length;
{
  register unsigned ptr, theend;
  theend = thestart + length;

  for ( ptr = thestart; ptr < theend; ptr += PAGE_SIZE ) {
    if ( AllocatePage(ptr, TRUE)) {
      for ( theend = thestart; theend < ptr; theend += PAGE_SIZE ) {
	Ms_FreePage( theend );
      }
      return( -1 );
    }
  }
  return( 0 );
}

initPageMap()
{
  register int i;
  for (i = 0; i < MAXPAGES; i++) {
    PageMap[i].use = USE_FREE;
  }
}

claimPage(page, use)
unsigned int page;
int use;
{
  TRACE2(memoryinit, 5, "Claiming page %d for use %d", page, use);
  PageMap[page].use = use;
}

initFreePageList(npages)
int npages;
{
  register int i;
  freePageList = NULL_PAGE;
  for (i = 0; i < npages; i++) {
    if (PageMap[i].use == USE_FREE) {
      PageMap[i].next = freePageList;
      freePageList = i;
      FreePages++;
    }
  }
  TRACE1(memoryinit, 1, "Free page list = %x", freePageList);
}

initPmegMap()
{
  register int i;
  for (i = 0; i < NUM_PMEGS; i++) PmegMap[i] = NULL_PMEG;
}

claimPmeg(p)
Pmeg p;
{
  if (p != NULL_PMEG) {
    TRACE1(memoryinit, 5, "Claiming pmeg # %d", p);
    PmegMap[p] = p;
  }
}

freePage(p)
register unsigned int p;
/* Add a page to the free list */
{
  TRACE1(processcreation, 1, "Freeing page number %x", p);

  PageMap[p].next = freePageList;
  freePageList = p;
  FreePages++;
}

initFreePmegList()
{
  unsigned a;
  register int i;
  freePmegList = NULL_PMEG;
  for (i = 0; i < NUM_PMEGS - 1; i++) {
    if (PmegMap[i] == NULL_PMEG) {
      SetSegMap(TESTSEG, i);
      for (a = TESTSEG; a < TESTSEG + SEG_SIZE; a += PAGE_SIZE) {
	SetPageMap(a, INVALID_PTE);
      }
      SetSegMap(TESTSEG, NULL_PMEG);
      PmegMap[i] = freePmegList;
      freePmegList = i;
      FreePmegs++;
    } else {
      PmegMap[i] = NULL_PMEG;
    }
  }
}

ClearPmeg(addr)
unsigned addr;
{
  register unsigned i;
  for (i = addr; i < uptoseg(addr+1); i += PAGE_SIZE) {
    SetPageMap(i, INVALID_PTE);
  }
}

FreePmeg(p)
/* Free the pmeg numbered p, clearing out all the pme's. */
Pmeg p;
{
  unsigned a;
  TRACE1(processcreation, 1, "Freeing pmeg %d", p);

  PmegMap[p] = freePmegList;
  freePmegList = p;

  SetSegMap(TESTSEG, p);
  for (a = TESTSEG; a < TESTSEG + SEG_SIZE; a += PAGE_SIZE) {
    SetPageMap(a, INVALID_PTE);
  }
  SetSegMap(TESTSEG, NULL_PMEG);
  FreePmegs ++;
}

MapKernelPmeg(addr, newpm)
register unsigned addr;
register Pmeg newpm;
{
  register ContextId oldContext = GetContext();
  register int i;
  for (i = 0; i < NUM_CONTEXTS; i++) {
    SetContext(i);
    SetSegMap(addr, newpm);
  }
  SetContext(oldContext);
}

AllocatePmeg(addr, kernel)
unsigned addr;
int kernel;
/*
 * Map an empty pmeg into the current context at the given address, if one
 *  can be found.  Return OK for success.
 */
{
  Pmeg newpm;
  extern int XInitialized;

  /* Find a pmeg */
  newpm = freePmegList;
  TRACE1(processcreation, 1, "Allocate pmeg, free list = %d", newpm);
  if (newpm == NULL_PMEG) return -1;

  TRACE3(processcreation, 1, "Allocating pmeg %d at %x%s\n", 
    newpm, addr, kernel ? " for kernel" : "");
  /* Map it in */
  if (kernel) {
    if (XInitialized && downtopage(StackBottom) != downtopage(FetchSP())) {
      CallOnKernelStack(MapKernelPmeg,addr,newpm);
    } else {
      MapKernelPmeg(addr, newpm);
    }
  } else {
    SetSegMap(addr, newpm);
  }

  /* Remove it from the free list */
  freePmegList = PmegMap[newpm];
  PmegMap[newpm] = NULL_PMEG;
  FreePmegs --;
  return(0);
}

int GetMemorySize()
/*
 * Probe to determine size of on-board memory on a Sun-3.
 * Returns the amount in pages.
 */
{
#ifdef USEROMMEMORYSIZE
  return *romp->v_memorysize / PAGE_SIZE;
#else
  PageMapEntry oldPage, pme;
  register int pages;

  /* Save old contents of page map entry we are using */
  oldPage.u = GetPageMap(TESTPAGE);

  /* Test onboard pages until one doesn't work, for some strange
   * reason, we need to start with page 2, starting with page 0 fails */
  pages = 2;
  pme.f.valid = 1;
  pme.f.protection = PAGE_WRITE;
  pme.f.type = ONBOARD_MEM;
  while (pages < ExtractPageNumber(P_ONBOARD_RAM_LIMIT+PAGE_SIZE)) {
    pme.f.pagenum = pages;
    SetPageMap(TESTPAGE, pme.u);
    if (Probe((short *)TESTPAGE)) {
      pages++;	/* more memory here */
    } else {
      break;	/* no more */
    }
  }
  SetPageMap(TESTPAGE, oldPage.u);
  return pages;
#endif
}

unsigned CurrentBreak = KERNEL_MALLOC;

/*ARGSUSED*/
char *brk(n)
unsigned n;
{
  Kabort("Brk called");
}

char *xsbrk(n)
int n;
{
  unsigned answer = CurrentBreak;
  if (n != PAGE_SIZE) Kabort("bad sbrk");
  if (CurrentBreak >= KERNEL_MALLOC_LIMIT) {
    Kabort("sbrk attempted past limit");
  }
  if (AllocatePage(CurrentBreak, TRUE)) {
    Kabort("AllocatePage failed for sbrk");
  }
  CurrentBreak += n;
  return (char *)answer;
}

int PageNumberAt(addr)
char *addr;
{
  PageMapEntry pg;
  pg.u = GetPageMap(addr);
  return pg.f.pagenum;
}

char *FindUserMapAddress(page)
int page;
{
  int i, limit;
  char *addr;
  PageMapEntry pg;
  
  for (i = page % 16, limit = (i + 15) % 16; i != limit; i = (i + 1)%16) {
    addr = (char *)(UMAPPINGSEG + i * PAGE_SIZE);
    pg.u = GetPageMap(addr);
    if (! pg.f.valid) return addr;
  }
  Kabort("No place to map a page to this user");
  /*NOTREACHED*/
}

char *FindKernelMapAddress(page)
int page;
{
  int i, limit;
  char *addr;
  PageMapEntry pg;
  
  for (i = page % 16, limit = (i + 15) % 16; i != limit; i = (i + 1)%16) {
    addr = (char *)(KMAPPINGSEG + i * PAGE_SIZE);
    pg.u = GetPageMap(addr);
    if (! pg.f.valid) return addr;
  }
  Kabort("No place to map a page in the kernel");
  /*NOTREACHED*/
}

#ifdef REALCHECKOUTMEMORY
static struct foo {
  VirtualAddress v;
  int context;
} where[NUM_PMEGS];
static struct foo wherep[MAX_PAGES];
#endif

CheckOutMemory()
{
#ifdef REALLYCHECKOUTMEMORY
  VirtualAddress va;
  Pmeg s, save;

  PageMapEntry pg, savep, newpg;
  register int i, context, limit, oldcontext, count, errors = 0;
  int x = Ms_Spl7();
  for (i = 0; i < NUM_PMEGS; i++) {
    where[i].v.u = 3;
  }
  oldcontext = GetContext();
  for (context = 0; context < NUM_CONTEXTS; context++) {
    SetContext(context);
    /* FIXME: Currently don't check out the process stacks */
    TRACE0(fixme, 1, "Should check out the process stacks");
    limit = context == 0 ? CONTEXT_SIZE : ASPACE_LIMIT - 0x80 * SEG_SIZE;
    for (va.u = 0; va.u < limit; va.u += SEG_SIZE) {
      s = GetSegMap(va.u);
      if (s != NULL_PMEG) {
	if (where[s].v.u != 3) {
	  printf("Found pmeg number %d mapped at address %x(%d) and %x(%d)\n",
	    s, where[s].v.u, where[s].context, va.u, context);
	  errors++;
	}
	where[s].v = va;
	where[s].context = context;
      }
    }
  }
  SetContext(0);
  s = freePmegList;
  save = GetSegMap(XFERSEG);
  while (s != NULL_PMEG) {
    if (where[s].v.u != 3) {
      printf("Pmeg number %d is free and mapped at addr %x(%d)\n",
	s, where[s].v.u, where[s].context);
      errors++;
    }
    SetSegMap(XFERSEG, s);
    s = (Pmeg) GetPageMap(XFERSEG);
  }
  for (i = 0; i < MAX_PAGES; i++) {
    wherep[i].v.u = 3;
  }
  for (context = 0; context < NUM_CONTEXTS; context++) {
    SetContext(context);
    /* FIXME: Currently don't check out the process stacks */
    TRACE0(fixme, 1, "Should check out the process stacks");
    limit = context == 0 ? CONTEXT_SIZE : ASPACE_LIMIT - 0x80 * SEG_SIZE;
    for (va.u = 0; va.u < limit; va.u += PAGE_SIZE) {
      pg.u = GetPageMap(va.u);
      if (pg.f.valid && pg.f.type == ONBOARD_MEM) {
	if (pg.f.pagenum < MAX_PAGES) {
	  if (wherep[pg.f.pagenum].v.u != 3) {
	    printf("Found page number %d mapped at address %x(%d) and %x(%d)\n",
	      pg.f.pagenum, wherep[pg.f.pagenum].v.u, wherep[pg.f.pagenum].context,
	      va.u, context);
	    errors++;
	  }
	  wherep[pg.f.pagenum].v = va;
	  wherep[pg.f.pagenum].context = context;
	}
      }
    }
  }
  SetContext(0);
  AllocatePmeg(XFERSEG, 1);
  pg.u = freePageList.u;
  while( pg.u != INVALID_PTE) {
    if (pg.f.valid && pg.f.type == ONBOARD_MEM) {
      if (pg.f.pagenum < MAX_PAGES) {
	if (wherep[pg.f.pagenum].v.u != 3) {
	  printf("Found page number %d mapped at address %x(%d) and free\n",
	    pg.f.pagenum, wherep[pg.f.pagenum].v.u,
	    wherep[pg.f.pagenum].context);
	  errors++;
	}
	wherep[pg.f.pagenum].v.u = XFERSEG;
	wherep[pg.f.pagenum].context = 0;
      }
    }
    pg.f.valid = 1;
    pg.f.protection = PAGE_WRITE | SUPER_ONLY;
    SetPageMap(XFERSEG, pg.u);
    pg = *(PageMapEntry *) downtopage(XFERSEG);
  }
  FreePmeg(XFERSEG);
  SetSegMap(XFERSEG, save);
  for (i = 0, count = 0; i < MAX_PAGES; i++) {
    if (wherep[i].v.u == 3) {
      TRACE1(memoryinit, 5, "Missing page %d", i);
      count++;
    }
  }
  if (count) printf("Missing %d pages\n", count);
  SetContext(oldcontext);
  assert (!errors);
  Ms_Splx(x);
#endif
}



