/* 
 * process.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include <signal.h>
#include "assert.h"
#include "process.h"
#include "debug.h"
#include "system.h"
typedef int (*PFI)();

Process *Active;
Process *ReadyQ = (Process *)NULL;
int SignalsPossible = 0;
int inInterrupt;

long *xksplimit()
{
  return Active->stacklimit;
}

long *xkspbase()
{
  return Active->stack;
}

/*ARGSUSED*/
char *PProcess(p, v)
register Process *p;
int v;
{
#define NBUFFERS 5
  static char buff[NBUFFERS][80];
  static int nextBuf = 0;
  char *s = buff[nextBuf++ % NBUFFERS];
  if (!p) {
    sprintf(s, "No process (nil)");
  } else {
    sprintf(s, "%#x (sp=%#x, pc=%#x)", p, p->jb.sp, p->jb.pc);
  }
  return(s);
}

void AddReady(p)
Process *p;
{
  register Process *h, *f;
  TRACE1(processswitch, 3, "Adding %s to readyq", PProcess(p, 0));
  assert(p->jb.sp < p->stacklimit);
  assert(p->jb.sp > p->stack);
  for (f = NULL, h = ReadyQ; h && h->prio <= p->prio; f = h, h = h->link);
  p->link = h;
  if (f) {
    f->link = p;
  } else {
    ReadyQ = p;
  }
}

void DisplayProcesses()
{
  int x = spl7();
  register Process *p;
  printf("Active:\n");
  printf("%s\n", PProcess(Active, 1));
  printf("ReadyQ:\n");
  for (p = ReadyQ; p; p = p->link) {
    printf("%s\n", PProcess(p, 1));
  }
  splx(x);
}

xschedule()
{
  while (!ReadyQ) {
    /* 6 May 88 NCH: Set Active to NULL */
    Active = NULL;
    if (SignalsPossible) {
      wait_interrupt();
      TRACE0(processswitch, 3, "Back from wait interrupt");
    } else {
      exit(0);
    }
  }
  Active = ReadyQ;
  ReadyQ = Active->link;
  Active->link = NULL;
  TRACE1(processswitch, 1, "Running %s", PProcess(Active, 0));
  assert(Active->jb.sp < Active->stacklimit);
  assert(Active->jb.sp > Active->stack);
  splx(0);
  _longjmp(&Active->jb, 1);
}

#define MAXPROCESSES 1000
static nextFree = 0;
static Process *processarea[MAXPROCESSES];

xForAllStacks(p)
PFI p;
{
  static long stuff[2*MAXPROCESSES];
  register Process *pr;
  int i, j;
  for (i = j = 0; i < MAXPROCESSES; i++) {
    if (pr = processarea[i]) {
      if (pr == Active) {
	stuff[j++] = (long) &i;
      } else {
	stuff[j++] = (long) pr->jb.sp;
      }
      stuff[j++] = (long) pr->stacklimit;
    }
  }
  p(j/2, stuff);
}

#define JOINSTACKSIZE (4 * 1024)
static int joinstack[JOINSTACKSIZE];

join()
{
  int index;
  (void) spl7();
#ifdef sun
  asm("	movl	#_joinstack+(4*4*1020), sp");
#endif
#ifdef vax
  asm("	movl	$_joinstack+(4*4*1020), sp");
#endif
  TRACE1(processswitch, 1, "Terminating %s", PProcess(Active, 0));
  index = Active->index;
  processarea[index] = 0;
  if (index < nextFree) nextFree = index;
  free((char *)Active->stack);
  free((char *)Active);
  xschedule();
}

xpid()
{
  return Active->index;
}

/*VARARGS3*/
CreateProcess(r, extrastuff, prio, nargs, firstarg)
PFI r;
int extrastuff;
short prio;
int nargs;
int firstarg;
{
  int stack;
  int *sp;
  Process *pd;
  extern join();
  int *argv = &firstarg, i;
  int x = spl7();
  TRACE3(processcreation, 3, "CreateProcess: r %x prio %d nargs %d", r, prio, nargs);
  if ((stack = (int)malloc(STACKSIZE)) == NULL) 
    Kabort("fork: no memory");
  if ((pd = (Process *)malloc(sizeof(Process))) == NULL) 
    Kabort("fork: no memory");
  pd->extrastuff = extrastuff;
  pd->prio = prio;
  pd->stack = (long *)stack;
  pd->stacklimit = (long *)(stack + STACKSIZE);
  while (processarea[nextFree] != 0) nextFree++;
  pd->index = nextFree;
  processarea[nextFree] = pd;
  nextFree++;
  sp = (int *)(stack + STACKSIZE);
  for (i = 1; i <= nargs; i++) {
    *--sp = argv[nargs - i];
  }
#ifdef vax
  pd->jb.pc = (long *)((int)r + 2);	/* 16 bits of mask word */
  *(--sp) = nargs;
  pd->jb.ap = (long *) sp;
  *(--sp) = (int) ((int)join+2);/* old pc */
  --sp;
  *(sp) = (int) (sp - 3);	/* old fp; only used if fall through to join */
  *(--sp) = 0;			/* old argument pointer */
  *(--sp) = 1 << 29;		/* mask word */
  *(--sp) = 0;			/* handler */
  pd->jb.fp = (long *)sp;
  pd->jb.sp = (long *)sp;
#endif
#ifdef sun
  *--sp = (int)join;
  *--sp = 0;			/* the argument to _setjmp */
  pd->jb.sp = (long *) sp;
  pd->jb.pc = (int)r;
#endif
  TRACE1(processcreation, 1, "Created %s", PProcess(pd, 0));
  TRACE1(processswitch, 1, "Created %s", PProcess(pd, 0));
  AddReady(pd);
  splx(x);
  return;
}
  
void Delay(n)
int n;
{
  if (!_setjmp(&Active->jb)) {
    event_register(AddReady, (int)Active, (unsigned)n, 1);
    xschedule();
  }
}

void Yield()
{
  /* 6 May 88 NCH: Added check for Active */
  (void) spl7();
  if (Active && !_setjmp(&Active->jb)) {
    AddReady(Active);
    xschedule();
  }
}

InitSemaphore(s, n)
register Semaphore *s;
unsigned n;
{
  s->count = n;
  Q_INIT(s);
}

realP(s)
register Semaphore *s;
{
  register Process *active = Active;
  register int sr = spl7();
  TRACE2(processswitch, 3, "P on %#x by %s", s, PProcess(active, 0));
  if (inInterrupt) Kabort("P from interrupt handler");
  if (!_setjmp(&active->jb)) {
    if (s->count < 0) {
      TRACE2(processswitch, 1, "Blocking p on %#x by %s", s, PProcess(active, 0));
      Q_INSERTLAST(s, active);
      xschedule();
    } else {
      splx(sr);
    }
  }
}

realV(s)
register Semaphore *s;
{
  register Process *p;
  register int sr = spl7();
  TRACE2(processswitch, 3, "V on %#x by %s", s, PProcess(Active, 0));
  if (s->count <= 0) {
    Q_REMOVEFIRST(s, p);
    if (p) {
      AddReady(p);
      TRACE3(processswitch, 1, "Unblocking %s by V on %#x by %s", 
	PProcess(p, 0), s, PProcess(Active, 0));
    }
  }
  splx(sr);
  /* 6 May 88: Deleted Yield() from here */
}

VAll(s)
register Semaphore *s;
{
  register Process *p;
  register int sr = spl7();
  TRACE2(processswitch, 3, "VAll on %#x by %s", s, PProcess(Active, 0));
  while (s->count < 0) {
    Q_REMOVEFIRST(s, p);
    if (p) AddReady(p);
    s->count ++;
  }
  splx(sr);
  /* 6 May 88: Deleted Yield() from here */
}

#undef CreateKernelProcess
/*VARARGS3*/
CreateKernelProcess(r, prio, nargs, a1, a2, a3)
PFI r;
short prio;
int nargs;
int a1, a2, a3;
{
  CreateProcess(r, 0, prio, nargs, a1, a2, a3);
}

/*
 * setjmp, longjmp
 * 
 * 	longjmp(a, v)
 * causes a "return(v)" from the
 * last call to
 * 
 * 	setjmp(v)
 * by restoring all the registers and
 * adjusting the stack
 * 
 * jmp_buf is set up as (on sun):
 * 
 * 	_________________
 * 	|	pc	|
 * 	-----------------
 * 	| signal_mask	|
 * 	-----------------
 * 	|   onstack	|
 * 	-----------------
 * 	|	d2	|
 * 	-----------------
 * 	|	...	|
 * 	-----------------
 * 	|	d7	|
 * 	-----------------
 * 	|	a2	|
 * 	-----------------
 * 	|	...	|
 * 	-----------------
 * 	|	a7	|
 * 	-----------------
 */
#ifdef sun
asm("	.text");
asm("	.globl	__setjmp");
asm("__setjmp:");
asm("	movl	sp@(4),a0");	/* pointer to jmp_buf */
asm("	movl	sp@,a0@");		/* pc */
asm("	clrl	a0@(4)");
asm("	clrl	a0@(8)");
asm("	moveml	#0xFCFC,a0@(12)");	/* d2-d7, a2-a7 */
asm("	clrl	d0");		/* return 0 */
asm("	rts");

asm("	.globl	__longjmp");
asm("__longjmp:");
asm("	movl	sp@(4),a0");	/* pointer to jmp_buf */
asm("	movl	sp@(8),d0");	/* value returned */
asm("	bne	1f");
asm("	moveq	#1,d0");
asm("1:");
asm("	moveml	a0@(12),#0xFCFC");	/* restore d2-d7, a2-a7 */
asm("	movl	a0@,sp@");		/* restore pc of call to setjmp to stack */
asm("	rts");
#endif
#ifdef vax
asm("	.text");
asm("	.globl	__setjmp");
asm("	.align	2");
asm("__setjmp:.word 0");
asm("	movl	4(ap),r0");
asm("	movab	24(fp),4(r0)");		/* save sp of caller */
asm("	movl	8(fp),12(r0)");		/* save ap of caller */
asm("	movl	12(fp),8(r0)");		/* save frame pointer of caller */
asm("	movl	16(fp),(r0)");		/* save pc of caller */
asm("	movq	r6,16(r0)");
asm("	movq	r8,24(r0)");
asm("	movq	r10,32(r0)");
asm("	clrl	r0");
asm("	ret");

asm("	.globl	__longjmp");
asm("	.align	2");
asm("__longjmp:	.word 0");
asm("	movl	8(ap),r0");		/* return(v) */
asm("	movl	4(ap),r1");		/* fetch buffer */
asm("	movq	16(r1),r6");
asm("	movq	24(r1),r8");
asm("	movq	32(r1),r10");
asm("	movl	8(r1),fp");		/* establish fp */
asm("	movl	12(r1),ap");		/* establish ap */
asm("	movl	4(r1),sp");		/* establish sp */
asm("	jmp	*(r1)");			/* establish pc */
#endif
