/****************************************************************************

    PROGRAM: Gortrs.c

    PURPOSE: Gorta ODBC Test Program


    COMMENTS:

	This program is intended to test, demonstrate and configure the
	gorta Mini-SQL ODBC connection. 

	If you haven't ensured that the WMSQLDLL program is working, please
	ensure that it is so using the wmsqlrs.exe program.

****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <windows.h>            /* required for all Windows applications */
#include <sql.h>				/* MS standard ODBC include file */
#include <sqlext.h>				/* Extended SQL functionality */

#include "gortars.h"            /* specific to this program          */

HANDLE hInst;               /* current instance              */

/* Prototypes for internal functions */
static void DisplayError(HENV henv, HDBC hdbc, HSTMT hstmt, char *pcFunction);
static void ShowSQLTables(HWND hWnd);


/*
 * Global Variables for managing the ODBC connection.
 */
static HENV ghenv;  /* The environment Handle */
static HDBC ghdbc;  /* The Database Context */
static HDBC ghstmt; /* Statement handle used for passing information to results dialog */

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

    COMMENTS:

        Windows recognizes this function by name as the initial entry point
        for the program.  This function calls the application initialization
        routine, if no other instance of the program is running, and always
        calls the instance initialization routine.  It then executes a message
        retrieval and dispatch loop that is the top-level control structure
        for the remainder of execution.  The loop is terminated when a WM_QUIT
        message is received, at which time this function exits the application
        instance by returning the value passed by PostQuitMessage().

        If this function must abort before entering the message loop, it
        returns the conventional value NULL.

****************************************************************************/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;                /* current instance         */
HANDLE hPrevInstance;            /* previous instance        */
LPSTR lpCmdLine;                 /* command line             */
int nCmdShow;                    /* show-window type (open/icon) */
{
    MSG msg;                     /* message              */

    if (!hPrevInstance)          /* Other instances of app running? */
    if (!InitApplication(hInstance)) /* Initialize shared things */
        return (FALSE);      /* Exits if unable to initialize     */

    /* Perform initializations that apply to a specific instance */

    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

	/*
	 * Connect to ODBC.
	 */
	if (SQLAllocEnv(&ghenv) != SQL_SUCCESS)
	{
		DisplayError(SQL_NULL_HENV, SQL_NULL_HDBC, SQL_NULL_HSTMT, "SQLAllocEnv");
	    return (msg.wParam);       /* Returns the value from PostQuitMessage */
	}
	/* Alloc a connection Handle - Only use on in this program */
	if (SQLAllocConnect(ghenv, &ghdbc) != SQL_SUCCESS)
	{
		DisplayError(ghenv, SQL_NULL_HDBC, SQL_NULL_HSTMT, "SQLAllocConnect");
	    return (msg.wParam);       /* Returns the value from PostQuitMessage */
	}

    /* Acquire and dispatch messages until a WM_QUIT message is received. */

    while (GetMessage(&msg,    /* message structure              */
        NULL,          /* handle of window receiving the message */
        NULL,          /* lowest message to examine          */
        NULL))         /* highest message to examine         */
    {
    TranslateMessage(&msg);    /* Translates virtual key codes       */
    DispatchMessage(&msg);     /* Dispatches message to window       */
    }
     
    return (msg.wParam);       /* Returns the value from PostQuitMessage */
}


/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

    COMMENTS:

        This function is called at initialization time only if no other
        instances of the application are running.  This function performs
        initialization tasks that can be done once for any number of running
        instances.

        In this case, we initialize a window class by filling out a data
        structure of type WNDCLASS and calling the Windows RegisterClass()
        function.  Since all instances of this application use the same window
        class, we only need to do this when the first instance is initialized.


****************************************************************************/

BOOL InitApplication(hInstance)
HANDLE hInstance;                  /* current instance       */
{
    WNDCLASS  wc;

    /* Fill in window class structure with parameters that describe the       */
    /* main window.                                                           */

    wc.style = NULL;                    /* Class style(s).                    */
    wc.lpfnWndProc = MainWndProc;       /* Function to retrieve messages for  */
                                        /* windows of this class.             */
    wc.cbClsExtra = 0;                  /* No per-class extra data.           */
    wc.cbWndExtra = 0;                  /* No per-window extra data.          */
    wc.hInstance = hInstance;           /* Application that owns the class.   */
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  "GortaRSMenu";   /* Name of menu resource in .RC file. */
    wc.lpszClassName = "GortaRSWClass"; /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */

    return (RegisterClass(&wc));

}


/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

    COMMENTS:

        This function is called at initialization time for every instance of
        this application.  This function performs initialization tasks that
        cannot be shared by multiple instances.

        In this case, we save the instance handle in a static variable and
        create and display the main program window.

****************************************************************************/

BOOL InitInstance(hInstance, nCmdShow)
    HANDLE          hInstance;          /* Current instance identifier.       */
    int             nCmdShow;           /* Param for first ShowWindow() call. */
{
    HWND            hWnd;               /* Main window handle.                */

    /* Save the instance handle in static variable, which will be used in  */
    /* many subsequence calls from this application to Windows.            */

    hInst = hInstance;

    /* Create a main window for this application instance.  */

    hWnd = CreateWindow(
        "GortaRSWClass",                /* See RegisterClass() call.          */
        "Gorta RS Application",   /* Text for window title bar.         */
        WS_OVERLAPPEDWINDOW,            /* Window style.                      */
        CW_USEDEFAULT,                  /* Default horizontal position.       */
        CW_USEDEFAULT,                  /* Default vertical position.         */
        CW_USEDEFAULT,                  /* Default width.                     */
        CW_USEDEFAULT,                  /* Default height.                    */
        NULL,                           /* Overlapped windows have no parent. */
        NULL,                           /* Use the window class menu.         */
        hInstance,                      /* This instance owns this window.    */
        NULL                            /* Pointer not needed.                */
    );

    /* If window could not be created, return "failure" */

    if (!hWnd)
        return (FALSE);

    /* Make the window visible; update its client area; and return "success" */

    ShowWindow(hWnd, nCmdShow);  /* Show the window                        */
    UpdateWindow(hWnd);          /* Sends WM_PAINT message                 */
    return (TRUE);               /* Returns the value from PostQuitMessage */

}

/****************************************************************************

    FUNCTION: MainWndProc(HWND, UINT, WPARAM, LPARAM)

    PURPOSE:  Processes messages

    MESSAGES:

    WM_COMMAND    - application menu (About dialog box)
    WM_DESTROY    - destroy window

    COMMENTS:

    To process the IDM_ABOUT message, call MakeProcInstance() to get the
    current instance address of the About() function.  Then call Dialog
    box which will create the box according to the information in your
    gortrs.rc file and turn control over to the About() function.  When
    it returns, free the intance address.

****************************************************************************/

long CALLBACK __export MainWndProc(hWnd, message, wParam, lParam)
HWND hWnd;                      /* window handle                 */
UINT message;                   /* type of message               */
WPARAM wParam;                  /* additional information        */
LPARAM lParam;                  /* additional information        */
{
	int iRv;
    switch (message)
    {
    case WM_COMMAND:       /* message: command from application menu */
    	switch (wParam)
    	{    
        case IDM_ABOUT:
			DialogBox(hInst,        /* current instance          */
				"AboutBox",         /* resource to use           */
				hWnd,               /* parent handle             */
				About);             /* About() instance address  */
			break;
		case IDM_EXIT:
			PostQuitMessage(0);
			break;
		case IDM_ODBC_CONNECT:
			DialogBox(hInst, "SQL_CONNECT",	hWnd, DlgSQLConnect);
			break;
		case IDM_ODBC_DRIVER_CONNECT:
			DialogBox(hInst, "SQL_DRIVER_CONNECT",	hWnd, DlgSQLDriverConnect);
			break;
		case IDM_ODBC_DISCONNECT:
			iRv = SQLDisconnect(ghdbc);
			if (iRv != SQL_SUCCESS && iRv != SQL_SUCCESS_WITH_INFO)
				DisplayError(ghenv, ghdbc, SQL_NULL_HSTMT, "SQLDisconnect");
			ghdbc = NULL;
			break;
		case IDM_SQL_TABLES:
			ShowSQLTables(hWnd);
			break;
		case IDM_SQL_COLUMNS:
			DialogBox(hInst, "SQL_COLUMNS",	hWnd, DlgSQLColumns);
			break;
		case IDM_SQL_EXEC_DIRECT:
			DialogBox(hInst, "SQL_EXEC_DIRECT",	hWnd, DlgSQLExecDirect);
			break;
		case IDM_CONFIGURE:
			DialogBox(hInst, "CONFIGURE",	hWnd, DlgConfigure);
			break;
		case IDM_README:
			WinExec("write readme.wri", SW_SHOW);
			break;			
        default:                        /* Lets Windows process it   */
        	return (DefWindowProc(hWnd, message, wParam, lParam));
        }
        break;
	case WM_DESTROY:          /* message: window being destroyed */
		if (ghdbc)
			(void)SQLDisconnect(ghdbc);
		ghdbc = NULL;
		(void)SQLFreeConnect(ghdbc);
		(void)SQLFreeEnv(ghenv);
		PostQuitMessage(0);
		break;

	default:                  /* Passes it on if unproccessed    */
		return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}


/****************************************************************************

    FUNCTION: About(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "About" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

    No initialization is needed for this particular dialog box, but TRUE
    must be returned to Windows.

    Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL __export CALLBACK About(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
    switch (message)
    {
        case WM_INITDIALOG:            /* message: initialize dialog box */
            return (TRUE);

        case WM_COMMAND:               /* message: received a command */
            if (wParam == IDOK         /* "OK" box selected?          */
                || wParam == IDCANCEL) /* System menu close command?  */
            {
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
                return (TRUE);
            }
            break;
    }
    return (FALSE);               /* Didn't process a message    */
}

/****************************************************************************

    FUNCTION: DlgSQLConnect(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "SQLConnect" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

****************************************************************************/

BOOL __export CALLBACK DlgSQLConnect(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
	int iRetCode;
	
    switch (message)
    {
    case WM_INITDIALOG:
    {	
    	char szDSN[SQL_MAX_DSN_LENGTH + 1];
    	long lDSNLen, lDescLen;
    	char szDescription[255];
    	
   		iRetCode = SQLDataSources(ghenv, SQL_FETCH_FIRST, szDSN, 
    		SQL_MAX_DSN_LENGTH, &lDSNLen, szDescription, 255, &lDescLen);
		for (;;) {    	
    		if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
    		{
				DisplayError(ghenv, ghdbc, SQL_NULL_HSTMT, "SQLDataSources");
				return(FALSE);
			}
			SendDlgItemMessage(hDlg, IDC_DATA_SOURCES, CB_ADDSTRING, 0, (LPARAM) ((LPSTR) szDSN));
    		iRetCode = SQLDataSources(ghenv, SQL_FETCH_NEXT, szDSN, 
	    		SQL_MAX_DSN_LENGTH, &lDSNLen, szDescription, 255, &lDescLen);
	    	if (iRetCode == SQL_NO_DATA_FOUND)
	    		break;
	   	}
        SendDlgItemMessage(hDlg, IDC_DATA_SOURCES, CB_SETCURSEL, 0, 0);
		return (TRUE);
	}

	case WM_COMMAND:               /* message: received a command */
		switch(wParam)
		{
		case IDOK:        /* "OK" box selected?          */
		{
			char szUser[30], szPWD[30], szDSN[SQL_MAX_DSN_LENGTH + 1];
			/* Get the strings from the dialogs .
			   Attempt to do SQLConnect().
			 */
			(void)GetDlgItemText(hDlg, IDC_DATA_SOURCES, szDSN, sizeof(szDSN));
			(void)GetDlgItemText(hDlg, IDC_USER, szUser, sizeof(szUser));
			(void)GetDlgItemText(hDlg, IDC_PASSWORD, szPWD, sizeof(szPWD));
            iRetCode = SQLConnect(ghdbc, szDSN, (SWORD)strlen(szDSN), szUser, 
            	(SWORD)strlen(szUser), szPWD, (SWORD)strlen(szPWD));
			if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
				DisplayError(ghenv, ghdbc, SQL_NULL_HSTMT, "SQLConnect");
			else
				EndDialog(hDlg, TRUE); /* Exits the dialog box        */
			return (TRUE);
		}
		case IDCANCEL: /* System menu close command?  */
			EndDialog(hDlg, TRUE); /* Exits the dialog box        */
			return (TRUE);
		}
		break;
    }
    return (FALSE);               /* Didn't process a message    */
}

/****************************************************************************

    FUNCTION: DlgSQLDriverConnect(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "SQLDriverConnect" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

****************************************************************************/

BOOL __export CALLBACK DlgSQLDriverConnect(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
    switch (message)
    {
    case WM_INITDIALOG:
		SendDlgItemMessage(hDlg, IDC_DRIVER_PROMPT, BM_SETCHECK, TRUE, 0);
		return (TRUE);

	case WM_COMMAND:               /* message: received a command */
		switch(wParam)
		{
		case IDOK:        /* "OK" box selected?          */
		{
			char szConnectString[255];
			long lOutLen;
			int iRetCode;
			int iFlags = 0;

			iFlags = SendDlgItemMessage(hDlg, IDC_DRIVER_PROMPT, BM_GETCHECK, 0, 0) == 1 ?
				SQL_DRIVER_PROMPT : iFlags;
			iFlags = SendDlgItemMessage(hDlg, IDC_DRIVER_NOPROMPT, BM_GETCHECK, 0, 0) == 1 ?
				SQL_DRIVER_NOPROMPT : iFlags;
			iFlags = SendDlgItemMessage(hDlg, IDC_DRIVER_COMPLETE, BM_GETCHECK, 0, 0) == 1 ?
				SQL_DRIVER_COMPLETE : iFlags;
			iFlags = SendDlgItemMessage(hDlg, IDC_DRIVER_COMPLETE_REQUIRED, BM_GETCHECK, 0, 0) == 1 ?
				SQL_DRIVER_COMPLETE_REQUIRED : iFlags;
			
			(void)GetDlgItemText(hDlg, IDC_CONNECT_STR, szConnectString, 
				sizeof(szConnectString));
			
			iRetCode = SQLDriverConnect(ghdbc, hDlg, szConnectString, 
				strlen(szConnectString), szConnectString, 255, &lOutLen, iFlags);

    		if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
    		{
				DisplayError(ghenv, ghdbc, SQL_NULL_HSTMT, "SQLDataSources");
				break;
			}
			MessageBox(hDlg, szConnectString, "Returned Connection String", MB_OK);
			EndDialog(hDlg, TRUE); /* Exits the dialog box        */
			return (TRUE);
			break;
		}
		case IDCANCEL: /* System menu close command?  */
			EndDialog(hDlg, TRUE); /* Exits the dialog box        */
			return (TRUE);
		}
		break;
    }
    return (FALSE);               /* Didn't process a message    */
}

/****************************************************************************

    FUNCTION: DisplayError(HENV, HDBC, HSTMT)

    PURPOSE:  Displays the errors using SQLError

    COMMENTS:

****************************************************************************/
static void
DisplayError(HENV henv, HDBC hdbc, HSTMT hstmt, char *pcFunction)
{
	char szErrorBuf[200];
	char szTitleBuf[200];
	char szSQLState[6];
	long lErrorCode;
	long lErrMsgLen;
	int retcode;
		
	for (;;) {
		retcode = SQLError(henv, hdbc, hstmt, szSQLState, &lErrorCode, 
			szErrorBuf, 200, &lErrMsgLen);
		if (retcode != SQL_SUCCESS && retcode != SQL_SUCCESS_WITH_INFO)
			break;
		sprintf(szTitleBuf, "ODBC Error on %s [%.5s] (%d)", pcFunction, szSQLState, lErrorCode);
		MessageBox(NULL, szErrorBuf, szTitleBuf, MB_OK);
	}
}

/****************************************************************************

    FUNCTION: ShowSQLTables(HWND)

    PURPOSE:  Displays the result set containing the tables in the source.

    COMMENTS: May be extended later to include a dialog where the 
    		  parameters may be modified.

****************************************************************************/
static void
ShowSQLTables(HWND hWnd)
{
	int iRetCode;
	
	iRetCode = SQLAllocStmt(ghdbc, &ghstmt);
	if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
	{
		DisplayError(ghenv, ghdbc, SQL_NULL_HSTMT, "SQLAllocStmt");
		return;
	}
	iRetCode = SQLTables(ghstmt, "", 0L, "", 0L, "", 0L, "", 0L);
	if (iRetCode == SQL_SUCCESS || iRetCode == SQL_SUCCESS_WITH_INFO)
		DialogBox(hInst, "SQL_RESULT",	hWnd, DlgResults);
	else
		DisplayError(ghenv, ghdbc, ghstmt, "SQLTables");
   	iRetCode = SQLFreeStmt(ghstmt, SQL_DROP);
	if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
	{
		DisplayError(ghenv, ghdbc, ghstmt, "SQLFreeStmt");
	   	ghstmt = NULL;
		return;
	}
   	ghstmt = NULL;
   	return;
}

/****************************************************************************

    FUNCTION: DlgResults(HWND, unsigned, WORD, LONG)

    PURPOSE:  Displays and processes the results dialog

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

****************************************************************************/

BOOL __export CALLBACK DlgResults(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
    switch (message)
    {
    case WM_INITDIALOG:
    {
    	// Use the manipulation functions to determine the make-up
    	// of the result set. Then set up the list BOX.
    	// Later should change to using MFC and a grid
    	int iRetCode, iLoop, iLineLen;
    	SWORD swCols, swScale, swNullable;
 		UDWORD udColDef;
 		struct ColData {
 			char pcTitle[32];	/* Name of the Column */
 			SWORD iTitleLen;    /* Length of title */
 			SWORD SQLType;		/* Type of the data */
 			SWORD iColLen;		/* MAX length of the Column */
 			SWORD iDisplayLen;	/* Display Length of the Column */
 			char *pcData;     	/* Data returned from SQLFetch */
 			SDWORD iDataLen;		/* Length returned from SQLFetch */
 		} *pColData;
		char *pcLine;			/* Buffer to display one line */
		char szTmpBuf[12];
		    	
    	iRetCode = SQLNumResultCols(ghstmt, &swCols);
    	if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
    	{
			DisplayError(ghenv, ghdbc, SQL_NULL_HSTMT, "SQLNumResultCols");
			EndDialog(hDlg, TRUE); /* Exits the dialog box        */
			return(FALSE);
		}
		if (swCols == 0)
		{
			MessageBox(hDlg, "Operation Successful. No Result Set.", "Results",
				MB_OK);
			EndDialog(hDlg, TRUE); /* Exits the dialog box        */
			return(FALSE);
		}
		/* Allocate all the tracking Buffers */
		pColData = calloc(swCols, sizeof(struct ColData));
		iLineLen = 0;
		for (iLoop=0; iLoop<swCols; iLoop++)
		{
			iRetCode = SQLDescribeCol(ghstmt, iLoop+1, pColData[iLoop].pcTitle, 
				32, &(pColData[iLoop].iTitleLen), &(pColData[iLoop].SQLType), 
				&udColDef, &swScale, &swNullable);
			if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
			{
				DisplayError(ghenv, ghdbc, ghstmt, "SQLDescribeCol");
				free(pColData);
				EndDialog(hDlg, TRUE); /* Exits the dialog box        */
				return(FALSE);
			}
			switch (pColData[iLoop].SQLType)
			{
			case SQL_CHAR:
			case SQL_VARCHAR:
				pColData[iLoop].iColLen = udColDef;
				pColData[iLoop].iDisplayLen = udColDef;
				break;                  
			case SQL_INTEGER:
				pColData[iLoop].iColLen = 4;
				pColData[iLoop].iDisplayLen = 10;
				break;
			case SQL_SMALLINT:
				pColData[iLoop].iColLen = 2;
				pColData[iLoop].iDisplayLen = 5;
				break;
			default:
			{
				char szMessage[200];
				sprintf(szMessage, "Field %s is of type %d.\n"
					"This type is not handled", pColData[iLoop].pcTitle,
					pColData[iLoop].SQLType);
				MessageBox(hDlg, szMessage, "Error on SQLDescribeCol", MB_OK);
				free(pColData);
				EndDialog(hDlg, TRUE); /* Exits the dialog box        */
				return(FALSE);
			}
			}
			iLineLen += pColData[iLoop].iDisplayLen;
		}
		/* Allocate the data buffers */
		for (iLoop=0; iLoop<swCols; iLoop++)
			pColData[iLoop].pcData = malloc(pColData[iLoop].iColLen);
		pcLine = malloc(iLineLen+1);
		
		/* Set up the binding for the fetch */
		for (iLoop=0; iLoop<swCols; iLoop++)
		{
			switch (pColData[iLoop].SQLType)
			{
			case SQL_CHAR:
			case SQL_VARCHAR:
				iRetCode = SQLBindCol(ghstmt, iLoop+1, SQL_C_CHAR, pColData[iLoop].pcData, 
					pColData[iLoop].iColLen, &(pColData[iLoop].iDataLen));
				break;                  
			case SQL_INTEGER:
				iRetCode = SQLBindCol(ghstmt, iLoop+1, SQL_C_LONG, pColData[iLoop].pcData, 
					0, &(pColData[iLoop].iDataLen));
				break;
			case SQL_SMALLINT:
				iRetCode = SQLBindCol(ghstmt, iLoop+1, SQL_C_SHORT, pColData[iLoop].pcData, 
					0, &(pColData[iLoop].iDataLen));
				break;
			}
			if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
			{
				DisplayError(ghenv, ghdbc, ghstmt, "SQLBindCol");
				for (iLoop=0; iLoop<swCols; iLoop++)
					free(pColData[iLoop].pcData);
				free(pColData);
				free(pcLine);
				EndDialog(hDlg, TRUE); /* Exits the dialog box        */
				return(FALSE);
			}

		}
		/* Now move through the data */
		while ((iRetCode = SQLFetch(ghstmt)) == SQL_SUCCESS || 
			iRetCode == SQL_SUCCESS_WITH_INFO)
		{
			iLineLen = 0;			
			for (iLoop=0; iLoop<swCols; iLoop++)
			{
				if (pColData[iLoop].iDataLen == SQL_NULL_DATA)
					sprintf(&(pcLine[iLineLen]), "%-*.*s", pColData[iLoop].iDisplayLen,
						pColData[iLoop].iDisplayLen, "NULL");
				else
					switch (pColData[iLoop].SQLType)
					{
					case SQL_CHAR:
					case SQL_VARCHAR:
						sprintf(&(pcLine[iLineLen]), "%-*.*s", pColData[iLoop].iDisplayLen,
							pColData[iLoop].iDisplayLen, pColData[iLoop].pcData);
						break;                  
					case SQL_INTEGER:
						sprintf(szTmpBuf, "%ld", *(long *)pColData[iLoop].pcData);
						sprintf(&(pcLine[iLineLen]), "%-*.*s", pColData[iLoop].iDisplayLen,
							pColData[iLoop].iDisplayLen, szTmpBuf);
						break;                  
					case SQL_SMALLINT:
						sprintf(szTmpBuf, "%d", *(int *)pColData[iLoop].pcData);
						sprintf(&(pcLine[iLineLen]), "%-*.*s", pColData[iLoop].iDisplayLen,
							pColData[iLoop].iDisplayLen, szTmpBuf);
						break;
					}
				iLineLen += pColData[iLoop].iDisplayLen;
			}
			SendDlgItemMessage(hDlg, IDC_LIST3, LB_ADDSTRING, 0, pcLine);
		}
		if (iRetCode != SQL_NO_DATA_FOUND)
		{
			DisplayError(ghenv, ghdbc, ghstmt, "SQLFetch");
		}
		for (iLoop=0; iLoop<swCols; iLoop++)
			free(pColData[iLoop].pcData);
		free(pColData);
		free(pcLine);
		{		
			/*
			 * Set up the horizontal Extent length for horizontal Scrolling.
			 */
			SWORD swExt;
			TEXTMETRIC tm;
			HDC hdc;
			hdc = GetDC(hDlg);
			GetTextMetrics(hdc, &tm);
			swExt = tm.tmAveCharWidth * iLineLen;
    		SendDlgItemMessage(hDlg, IDC_LIST3, LB_SETHORIZONTALEXTENT, swExt, 0L);
	    	ReleaseDC(hDlg, hdc);
		}
    	return(TRUE);
    }
    
	case WM_COMMAND:               /* message: received a command */
		switch(wParam)
		{
		case IDOK:        /* "OK" box selected?          */
			EndDialog(hDlg, TRUE); /* Exits the dialog box        */
			return (TRUE);
		}
		break;
    }
    return (FALSE);               /* Didn't process a message    */
}


/****************************************************************************

    FUNCTION: DlgSQLColumns(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "SQLColumns" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

****************************************************************************/

BOOL __export CALLBACK DlgSQLColumns(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
    switch (message)
    {
    case WM_INITDIALOG:
		return (TRUE);

	case WM_COMMAND:               /* message: received a command */
		switch(wParam)
		{
		case IDOK:        /* "OK" box selected?          */
		{
			int iRetCode;
			char szQualifier[100], szOwner[100], szName[100], szColumnName[100];
			
			(void)GetDlgItemText(hDlg, IDC_TABLE_QUALIFIER, szQualifier, sizeof(szQualifier));
			(void)GetDlgItemText(hDlg, IDC_TABLE_NAME, szName, sizeof(szName));
			(void)GetDlgItemText(hDlg, IDC_TABLE_OWNER, szOwner, sizeof(szOwner));
			(void)GetDlgItemText(hDlg, IDC_COLUMN_NAME, szColumnName, 
				sizeof(szColumnName));
	
			iRetCode = SQLAllocStmt(ghdbc, &ghstmt);
			if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
			{
				DisplayError(ghenv, ghdbc, SQL_NULL_HSTMT, "SQLAllocStmt");
				return(TRUE);
			}
			iRetCode = SQLColumns(ghstmt, szQualifier, strlen(szQualifier),
				szOwner, strlen(szOwner), szName, strlen(szName),
				szColumnName, strlen(szColumnName));
			if (iRetCode == SQL_SUCCESS || iRetCode == SQL_SUCCESS_WITH_INFO)
				DialogBox(hInst, "SQL_RESULT",	hDlg, DlgResults);
			else
				DisplayError(ghenv, ghdbc, ghstmt, "SQLColumns");
		   	iRetCode = SQLFreeStmt(ghstmt, SQL_DROP);
			if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
			{
				DisplayError(ghenv, ghdbc, ghstmt, "SQLFreeStmt");
	   			ghstmt = NULL;
				return(TRUE);
			}
   			ghstmt = NULL;
   			return(TRUE);
		}
		case IDCANCEL: /* System menu close command?  */
			EndDialog(hDlg, TRUE); /* Exits the dialog box        */
			return (TRUE);
		}
		break;
    }
    return (FALSE);               /* Didn't process a message    */
}


/****************************************************************************

    FUNCTION: DlgSQLExecDirect(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "SQLExecDirect" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

****************************************************************************/

BOOL __export CALLBACK DlgSQLExecDirect(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
    switch (message)
    {
    case WM_INITDIALOG:
		return (TRUE);

	case WM_COMMAND:               /* message: received a command */
		switch(wParam)
		{
		case IDOK:        /* "OK" box selected?          */
		{
			int iRetCode;
			char szSQLRequest[1000];
			
			(void)GetDlgItemText(hDlg, IDC_SQL, szSQLRequest, sizeof(szSQLRequest));
	
			iRetCode = SQLAllocStmt(ghdbc, &ghstmt);
			if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
			{
				DisplayError(ghenv, ghdbc, SQL_NULL_HSTMT, "SQLAllocStmt");
				return(TRUE);
			}
			iRetCode = SQLExecDirect(ghstmt, szSQLRequest, strlen(szSQLRequest));
			if (iRetCode == SQL_SUCCESS || iRetCode == SQL_SUCCESS_WITH_INFO)
				DialogBox(hInst, "SQL_RESULT",	hDlg, DlgResults);
			else
				DisplayError(ghenv, ghdbc, ghstmt, "SQLExecDirect");
		   	iRetCode = SQLFreeStmt(ghstmt, SQL_DROP);
			if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
			{
				DisplayError(ghenv, ghdbc, ghstmt, "SQLFreeStmt");
	   			ghstmt = NULL;
				return(TRUE);
			}
   			ghstmt = NULL;
   			return(TRUE);
		}
		case IDCANCEL: /* System menu close command?  */
			EndDialog(hDlg, TRUE); /* Exits the dialog box        */
			return (TRUE);
		}
		break;
    }
    return (FALSE);               /* Didn't process a message    */
}


/****************************************************************************

    FUNCTION: DlgConfigure(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "Configure Gorta ODBC Driver" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

****************************************************************************/

BOOL __export CALLBACK DlgConfigure(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
	int iRetCode;
	static char *szLastDSN[SQL_MAX_DSN_LENGTH + 1];
	
    switch (message)
    {
    case WM_INITDIALOG:
    {	
    	char szDSN[SQL_MAX_DSN_LENGTH + 1];
    	long lDSNLen, lDescLen;
    	char szDescription[255];
    	
   		iRetCode = SQLDataSources(ghenv, SQL_FETCH_FIRST, szDSN, 
    		SQL_MAX_DSN_LENGTH, &lDSNLen, szDescription, 255, &lDescLen);
		for (;;) {    	
    		if (iRetCode != SQL_SUCCESS && iRetCode != SQL_SUCCESS_WITH_INFO)
    		{
				DisplayError(ghenv, ghdbc, SQL_NULL_HSTMT, "SQLDataSources");
				return(FALSE);
			}
			SendDlgItemMessage(hDlg, IDC_DATA_SOURCES, LB_ADDSTRING, 0, (LPARAM) ((LPSTR) szDSN));
    		iRetCode = SQLDataSources(ghenv, SQL_FETCH_NEXT, szDSN, 
	    		SQL_MAX_DSN_LENGTH, &lDSNLen, szDescription, 255, &lDescLen);
	    	if (iRetCode == SQL_NO_DATA_FOUND)
	    		break;
	   	}
		return (TRUE);
	}
	
	case WM_COMMAND:               /* message: received a command */
		switch(wParam)
		{
		case IDOK:        /* "OK" box selected?          */
		{
			char szBuffer[100];
							
			/*
			 * A Data Source Name has been selected. Set the
			 * Initialial Values in the 3 Edit Boxes
			 */
			GetDlgItemText(hDlg, IDC_Description, szBuffer, sizeof(szBuffer));
			WritePrivateProfileString(szLastDSN, "Description",
				szBuffer, "ODBC.INI");
			GetDlgItemText(hDlg, IDC_Server, szBuffer, sizeof(szBuffer));
			WritePrivateProfileString(szLastDSN, "Server",
				szBuffer, "ODBC.INI");
			GetDlgItemText(hDlg, IDC_Database, szBuffer, sizeof(szBuffer));
			WritePrivateProfileString(szLastDSN, "Database",
				szBuffer, "ODBC.INI");
			EndDialog(hDlg, TRUE); /* Exits the dialog box        */
   			return(TRUE);
		}
		case IDCANCEL: /* System menu close command?  */
			EndDialog(hDlg, TRUE); /* Exits the dialog box        */
			return (TRUE);
		default:
			switch (HIWORD(lParam))
			{
			case LBN_DBLCLK:
				{
					char *szCurDSN[SQL_MAX_DSN_LENGTH + 1];
					DWORD dwIndex;
					
					(void)GetDlgItemText(hDlg, IDC_DATA_SOURCES, szCurDSN, sizeof(szCurDSN));

					dwIndex = SendDlgItemMessage(hDlg, IDC_DATA_SOURCES, LB_GETCURSEL, 0, 0);
					if (dwIndex != LB_ERR)
					{
    					SendDlgItemMessage(hDlg, IDC_DATA_SOURCES, LB_GETTEXT, (WPARAM)dwIndex, 
    						(LPARAM) ((LPSTR) szCurDSN));

						if (strcmp(szCurDSN, szLastDSN))
						{
							char szBuffer[100];
							
							strcpy(szLastDSN, szCurDSN);
							/*
							 * A Data Source Name has been selected. Set the
							 * Initialial Values in the 3 Edit Boxes
							 */
							GetPrivateProfileString(szCurDSN, "Description",
								"(Not Set)", szBuffer, sizeof(szBuffer), "ODBC.INI");
							SetDlgItemText(hDlg, IDC_Description, szBuffer);
							GetPrivateProfileString(szCurDSN, "Server",
								"(Not Set)", szBuffer, sizeof(szBuffer), "ODBC.INI");
							SetDlgItemText(hDlg, IDC_Server, szBuffer);
							GetPrivateProfileString(szCurDSN, "Database",
								"(Not Set)", szBuffer, sizeof(szBuffer), "ODBC.INI");
							SetDlgItemText(hDlg, IDC_Database, szBuffer);
						}
					}
    	    		return(TRUE);
				}
			}
		}
		break;
    }
    return (FALSE);               /* Didn't process a message    */
}

