"""
t3Metaclass.py

Provides various classes representing t3 metaclass objects. This code is based
on the Tads 3 reference implementation by Michael J. Roberts. 
"""

import struct, cStringIO
from t3DataHolder import *
from decimal import Decimal

class T3Metaclass:
	def __init__(self, data):
		self.interpret_data(data)

	def report(self):
		return repr(self.to_python_val())
		
	def report_deep(self, image):
		return self.report()


class MetaTadsObject(T3Metaclass):
	"""
	Implementation of the t3 tads-object metaclass.
	"""
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		mcdata = {}
		supcnt, pcnt, flgs = struct.unpack("<3H", datastream.read(6))
		superclasses = []
		for i in xrange(supcnt):
			objid = int(struct.unpack("<I", datastream.read(4))[0])
			superclasses.append(objid)
		self.l_superclasses = superclasses
		properties = []
		for i in xrange(pcnt):
			propid = int(struct.unpack("<H", datastream.read(2))[0])
			dhs = datastream.read(5)
			value = get_data_holder(dhs)
			properties.append((propid, value))
		self.l_properties = properties
		datastream.close()
		
	def to_python_val(self):
		return self
		
	def report(self):
		sl = []
		sl.append("TADS-Object")
		sl.append("\n\tsuperclasses: " + str(self.l_superclasses))
		sl.append("\n\tproperties: " + str(self.l_properties))
		return "".join(sl)
		
	def report_deep(self, image):
		sl = []
		sl.append("TADS-Object")
		sl.append("\n\tsuperclasses: " + str(self.l_superclasses))
		propvals = [prop[1].get_reference(image) for prop in self.l_properties]
		sl.append("\n\tproperties: " + str(propvals))
		return "".join(sl)
	
		
class MetaIntClassMod(MetaTadsObject):
	pass

		
class MetaList(T3Metaclass):
	"""
	Implementation of the t3 list metaclass. As lists are a basic type in
	python this is all fairly simple.
	"""
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		items = []
		itemcount = struct.unpack("<H", datastream.read(2))[0]
		for i in xrange(itemcount):
			items.append(get_data_holder(datastream.read(5)))
		self.l_items = items
		datastream.close()

	def to_python_val(self):
		return self.l_items
		
	def report_deep(self, image):
		sl = []
		sl.append(str([item.get_reference(image) for item in self.l_items]))
		return "".join(sl)


class MetaBigNumber(T3Metaclass):
	"""
	Implementation of the t3 BigNumber class. At the expense of limiting
	ourselves to python 2.4 and above, we use the decimal module from the
	library to represent the BigNumber.
	"""
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		num, exp, flag = struct.unpack("<HhB", datastream.read(5))[0]
		sign = flag & 1
		special = flag & 6
		iszero = flag & 8
		if special == 2: # NaN
			self.o_number = Decimal("NaN")
			datastream.close()
			return
		elif special == 4:
			if sign == 0:	# positive infinity
				self.o_number = Decimal("INFINITY")
			else:	# negative infinity
				self.o_number = Decimal("-INFINITY")
			datastream.close()
			return
		elif iszero:
			if sign == 0: # zero
				self.o_number = Decimal("0")
			else:	#  negative zero
				self.o_number = Decimal("-0")
			datastream.close()
			return
		bytes = []
		for i in xrange((num+1)/2):
			high, low = self.packed_bcd_to_ints(struct.unpack("<B",
											datastream.read(1))[0])
			bytes.append(high)
			bytes.append(low)
		datastream.close()
		self.o_number = Decimal((sign, bytes, exp))
		
	def packed_bcd_to_ints(self, byte):
		return (byte & 240, byte & 15)
		
	def to_python_val(self):
		return self.o_number.to_eng_string()

class MetaByteArray(T3Metaclass):
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		n = struct.unpack("<I", datastream.read(4))[0]
		bytes = []
		for i in xrange(n):
			bytes.append(struct.unpack("<B", datastream.read(1))[0])
		datastream.close()
		self.l_bytes = bytes
		
	def to_python_val(self):
		return self.l_bytes
	
	
class MetaCharacterSet(T3Metaclass):
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		n = struct.unpack("<H", datastream.read(2))[0]
		self.s_name = datastream.read(n)
		datastream.close()
		
	def to_python_val(self):
		return self.s_name

		
# collections do not appear in image files, so we'll ignore them for now.


class MetaDictionary(T3Metaclass):
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		compar_obj_id, ecount = struct.unpack("<IH", datastream.read(6))
		self.comparator_id = compar_obj_id
		tdict = {}
		for i in xrange(ecount):
			klen = struct.unpack("<B", datastream.read(1))[0]
			key = [chr(ord(char) ^ 0xBD) for char in datastream.read(klen)]
			k = "".join(key)
			tdict[k] = []
			nsub = struct.unpack("<H", datastream.read(2))[0]
			for j in xrange(nsub):
				tdict[k].append(struct.unpack("<IH", datastream.read(6)))
		self.d_dictionary = tdict
		datastream.close()
		
	def to_python_val(self):
		return self.d_dictionary
		
	def report_deep(self, image):
		newdict = {}
		for k in self.d_dictionary.keys():
			valuelist = self.d_dictionary[k]
			newvaluelist = []
			for object, prop in valuelist:
				obj = image.objectDict[object]
				newvaluelist.append((obj, prop))
			newdict[k] = newvaluelist
		s = ["Dictionary"]
		for k in newdict:
			for obj, prop in newdict[k]:
				s.append("\t" + str(obj) + " " + str(prop))
		return "\n".join(s)


		
class MetaFile(T3Metaclass):
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		charsetID = struct.unpack("<I", datastream.read(4))[0]
		self.i_charset_id = charsetID
		modeb, accessb = struct.unpack("<BB", datastream.read(2))[0]
		self.s_access = {1: "r", 2: "w", 3: "rw", 4: "rw+"}[accessb]
		self.s_mode = {1: "", 2: "b", 3: "b"}[modeb]
		# actually, mode 2 is data, which in TADS automatically converts TADS
		# data formats to their binary representations. But we'll pretend it's
		# just binary for now
		flags = struct.unpack("<I", datastream.read(4))[0]
		self.b_outOfSync = bool(flags & 1)
		self.b_stdioBuffersDirty = bool(flags & 2)
		self.b_lastOpWrite = bool(flags & 4)
		self.b_isResource = bool(flags & 8)
		datastream.close()
		
	def to_python_val(self):
		return self
		
	def report(self):
		sl = []
		sl.append("T3FileObject[ ")
		sl.append("mode: " + self.self.s_access + self.s_mode + "]")
		return "".join(sl)
		
class MetaGrammarProduction(T3Metaclass):
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		altcount = struct.unpack("<H", datastream.read(2))[0]
		alternatives = []
		for i in xrange(altcount):
			altdict = {}
			scr, bdns, pobjid, ntok = struct.unpack("<hhIH",
											datastream.read(10))
			altdict["i_score"] = scr
			altdict["i_badness"] = bdns
			altdict["i_processor_object_id"] = pobjid
			tokens = []
			for j in xrange(ntok):
				tokdict = {}
				prop_assoc, matchtyp = struct.unpack("<HB", datastream.read(3))
				tokdict["i_property_association"] = prop_assoc
				matchtype = {1: "VGRAM_MATCH_PROD",
							 2: "VGRAM_MATCH_SPEECH",
							 3: "VGRAM_MATCH_LITERAL",
							 4: "VGRAM_MATCH_TOKTYPE",
							 5: "VGRAM_MATCH_STAR",
							 6: "VGRAM_MATCH_NSPEECH"}[matchtyp]
				tokdict["s_matchtype"] = matchtype
				datadict = {}
				if matchtyp == 1:
					datadict["i_production_object_id"] = struct.unpack("<I",
														datastream.read(4))[0]
				elif matchtyp == 2:
					datadict["i_vocabulary_property"] = struct.unpack("<H",
														datastream.read(2))[0]
				elif matchtyp == 3:
					nstr = struct.unpack("<H", datastream.read(2))[0]
					datadict["s_literal_string"] = datastream.read(nstr)
				elif matchtyp == 4:
					datadict["i_token_enumerator_id"] = struct.unpack("<I",
														datastream.read(4))[0]
				elif matchtyp == 6:
					n = struct.unpack("<H", datastream.read(2))[0]
					vprops = struct.unpack("<" + str(int(n)) + "H",
									datastream.read(n * 2))
					datadict["l_vocabulary_properties"] = vprops
				tokdict["d_data"] = datadict
				tokens.append(tokdict)
			altdict["l_tokens"] = tokens
			alternatives.append(altdict)
		self.l_alternatives = alternatives
		datastream.close()
		
	def to_python_val(self):
		return self.l_alternatives
		
	def report(self):
		sl = []
		sl.append("GrammarProduction")
		for alt in self.l_alternatives:
			sl.append("\n\tscore: " + str(alt["i_score"]))
			sl.append("\n\tbadness: " + str(alt["i_badness"]))
			sl.append("\n")
			for token in alt["l_tokens"]:
				mt = token["s_matchtype"]
				sl.append("\t" + token["s_matchtype"])
				if mt == "VGRAM_MATCH_PROD":
					sl.append(" production object: " + str(token["d_data"]
											["i_production_object_id"]))
				elif mt == "VGRAM_MATCH_SPEECH":
					sl.append(" vocab property: " + str(token["d_data"]
											["i_vocabulary_property"]))
				elif mt == "VGRAM_MATCH_LITERAL":
					sl.append(" string: " + str(token["d_data"]
											["s_literal_string"]))
				elif mt == "VGRAM_MATCH_TOKTYPE":
					sl.append(" enumerator id: " + str(token["d_data"]
											["i_token_enumerator_id"]))
				elif mt == "VGRAM_MATCH_NSPEECH":
					sl.append(" vocab properties: " + str(token["d_data"]
											["l_vocabulary_properties"]))
				sl.append("\n")
		return "".join(sl)
		

class MetaIntrinsicClass(T3Metaclass):
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		# ignore the byte count as it's always 8 and we have no useful way of
		# doing anything different if it turns out not to be, except perhaps
		# ignoring the rest of the data
		datastream.read(2)
		deptabidx = struct.unpack("<H", datastream.read(2))[0]
		mod_obj_id = struct.unpack("<I", datastream.read(4))[0]
		self.i_dependencyTableIndex = deptabidx
		self.i_modifierObjectID = mod_obj_id
		datastream.close()
	
	def to_python_val(self):
		return (self.i_dependencyTableIndex, self.i_modifierObjectID)


class MetaIterator(T3Metaclass):
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		self.o_collection_value = get_data_holder(datastream.read(5))
		idx, first, last, flags = struct.unpack("<4I", datastream.read(16))
		self.i_currentIndex = idx
		self.i_firstValid = first
		self.i_lastValid = last
		self.b_ObjIterIdxUndo = bool(flags & 1)
		datastream.close()

	def to_python_val(self):
		# python has iterators built in to all sequence types anyway, so this
		# should just be for show
		return (self.i_currentIndex,
				self.i_firstValid,
				self.i_lastValid,
				self.b_ObjIterIdxUndo)
				

		
class MetaLookupTable(T3Metaclass):
	def interpret_data(self, data):
		# As python dictionaries does all the bucket stuff under the hood, we
		# don't actually need to care about it. We just get enough information
		# for a standard dictionary.
		datastream = cStringIO.StringIO(data)
		bc, vc, ffi = struct.unpack("<3H", datastream.read(6))
		self.i_bucketCount = bc
		self.i_valueCount = vc
		self.i_firstFreeIndex = ffi
		datastream.read(bc*2)
		tdict = {}
		for i in xrange(ffi):
			key = get_data_holder(datastream.read(5))
			value = get_data_holder(datastream.read(5))
			datastream.read(2)
			tdict[key] = value
		self.d_lookupTable = tdict
		datastream.close()
		
	def to_python_val(self):
		return self.d_lookupTable
		
	def report_deep(self, image):
		sl = ["LookupTable"]
		for k in self.d_lookupTable:
			sl.append("\n\t" + str(k.get_reference(image)))
			sl.append(" : " + str(self.d_lookupTable[k].get_reference(image)))
		return "".join(sl)

		
class MetaWeakRefLookupTable(MetaLookupTable):
	pass

class MetaRexPattern(T3Metaclass):
	def interpret_data(self, data):
		# we don't need to bother actually opening up a stream for this as the
		# internal representation is a data holder.
		self.o_pattern = get_data_holder(data)
		
	def to_python_val(self):
		return self.o_pattern
		
	def report_deep(self, image):
		return repr(self.o_pattern.get_reference(image))
		
class MetaString(T3Metaclass):
	def interpret_data(self, data):
		# this one's easy - the first two bytes are the length of the string,
		# the rest of it is the string, which is all we want.
		self.string = data[2:]
		
	def to_python_val(self):
		return self.string
		
	
class MetaStringComparator(T3Metaclass):
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		trlen, flags, eq_mapc, eq_tvc = struct.unpack("<4H", 
											datastream.read(8))
		self.i_truncationLength = trlen
		self.b_caseSensitive = bool(flags & 1)
		self.l_equivalencyMappings = []
		for i in xrange(eq_mapc):
			eq_map = {}
			refc, vcc, ucresf, lcresf = struct.unpack("<HBII",
											datastream.read(11))
			eq_map["us_referenceChar"] = unichr(refc)
			eq_map["i_UpperResultFlags"] = ucresf
			eq_map["i_LowerResultFlags"] = lcresf
			uvals = struct.unpack("<" + str(int(vcc)) + "H",
											datastream.read(vcc*2))
			vchrs = "".join([unichr(uval) for uval in uvals])
			eq_map["s_valueChars"] = vchrs
			self.l_equivalencyMappings.append(eq_map)
		datastream.close()
		
	def to_python_val(self):
		return self
		
	def report(self):
		sl = ["string-comparator"]
		sl.append("\n\ttruncation length: " + str(self.i_truncationLength))
		sl.append("\n\tcase sensitive: " + str(self.b_caseSensitive))
		for eqm in self.l_equivalencyMappings:
			sl.append("\n\t" + eqm["s_valueChars"])
		return "".join(sl)
			

class MetaVector(T3Metaclass):
	def interpret_data(self, data):
		datastream = cStringIO.StringIO(data)
		nalloc, nused = struct.unpack("<2H", datastream.read(4))
		self.l_elements = []
		for n in xrange(nused):
			self.l_elements.append(get_data_holder(datastream.read(5)))
		datastream.close()
		
	def to_python_val(self):
		return self.l_elements
		
	def deep_report(self, image):
		return str([el.get_reference(image) for el in self.l_elements])
		

class MetaAnonFnPtr(MetaVector):
	pass
	
class MetaUnknownClass(T3Metaclass):
	def interpret_data(self, data):
		self.data = data
		
	def to_python_val(self):
		return repr(self.data)




metaclassdict = {	"tads-object": MetaTadsObject,
					"list": MetaList,
					"dictionary2": MetaDictionary,
					"grammar-production": MetaGrammarProduction,
					"vector": MetaVector,
					"anon-fn-ptr": MetaAnonFnPtr,
					"int-cls-mod": MetaIntClassMod,
					"intrinsic-class": MetaIntrinsicClass,
					"iterator": MetaIterator,
					"indexed-iterator": MetaIterator,
					"character-set": MetaCharacterSet,
					"bytearray": MetaByteArray,
					"string": MetaString,
					"regex-pattern": MetaRexPattern,
					"lookuptable": MetaLookupTable,
					"weakreflookuptable": MetaLookupTable,
					"lookuptable-iterator": MetaIterator,
					"file": MetaFile,
					"string-comparator": MetaStringComparator }