(*
$File: Compiler/RunLambda.sml $
$Date: 1992/09/17 14:18:04 $
$Revision: 1.1 $
$Locker:  $
*)

(*$RunLambda:
	LVARS LAMBDA_EXP RUNTIME APPLY FINMAP CRASH RUN_LAMBDA
 *)

(* RunLambda - a quickly hacked up interpreter for the lambda intermediate
   language. We still have explicit environments because we haven't done the
   closure analysis in the lambda language. *)

functor RunLambda(structure Lvars: LVARS	(* for `eqtype lvar' *)

		  structure LambdaExp: LAMBDA_EXP
		    sharing type LambdaExp.lvar = Lvars.lvar

		  structure Runtime: RUNTIME
		    sharing type Runtime.Objects.LambdaExp = LambdaExp.LambdaExp
		        and type Runtime.Objects.lvar = Lvars.lvar

		  structure Apply: APPLY
		    sharing type Apply.object = Runtime.Objects.object

		  structure FinMap: FINMAP
		    sharing type LambdaExp.map = FinMap.map

		  structure Crash: CRASH
		 ): RUN_LAMBDA =
  struct
    open LambdaExp
    open Runtime		(* bring in `Objects' and `DynamicEnv' *)
    type DEnv = DynamicEnv.DEnv
    type object = Objects.object
    infix plus
    val (op plus) = DynamicEnv.plus

   (* exceptions: PACKET is used for raising and handling exception
      packets. Should one propagate to the outside world, we turn it
      into an UNCAUGHT so that the main loop can print it. *)

    exception PACKET of object
    exception UNCAUGHT of object

    fun printPacket pack: string =
      let
	open Objects
      in
	Crash.unimplemented "RAISE(VOID) still in place";
	deString(deRef(select(1, pack)))
      end

    fun switch(runner: LambdaExp -> object,
	       unpack: object -> ''a,
	       SWITCH{arg, selections, wildcard}: ''a Switch
	      ) =
      let
	val const = unpack(runner arg)
      in
	case FinMap.lookup selections const
	  of Some lamb => runner lamb
	   | None =>
	       (case wildcard of Some w => runner w
	     		       | None => Crash.impossible "switch(w=NONE)"
	       )
      end

    fun run env lamb =
      case lamb
	of VAR lv => DynamicEnv.lookup env lv
	 | INTEGER x => Objects.integer x
	 | STRING x => Objects.string x
	 | REAL x => Objects.real x

	 | FN(lv, lamb) => Objects.closure{arg=lv, body=lamb, bodyEnv=env}

	 | FIX(lvs, lambs, lamb) =>
	     let
	       val closures = map (run env) lambs
					(* the closures don't have the
					   top layer of recursive env... *)
	       val recEnv =
		 let
		   fun f(lv :: lvs, c :: cs) =
		         DynamicEnv.declare(lv, c, f(lvs, cs))
		     | f(nil, nil) =
		         DynamicEnv.emptyDEnv
		     | f _ =
		         Crash.impossible "RunLambda(FIX)"
		 in
		   f(lvs, closures)
		 end
	     in
	       run (env plus (DynamicEnv.REC recEnv)) lamb
	     end

	 | APP(lamb1, lamb2) =>
	     let
	       val func = run env lamb1
	       val arg = run env lamb2
	       val {arg=lv, body, bodyEnv, recEnv} = Objects.deClosure func
	     in
	       run (DynamicEnv.declare(lv, arg,
				       bodyEnv plus (DynamicEnv.REC recEnv)
				      )
		   ) body
	     end

	 | PRIM_APP(n, lamb) =>
	     Apply.apply(n, run env lamb)

	 | VECTOR lambs =>
	     Objects.vector(map (run env) lambs)

	 | SELECT(n, lamb) =>
	     Objects.select(n, run env lamb)

	 | SWITCH_I sw => switch(run env, Objects.deInteger, sw)
	 | SWITCH_S sw => switch(run env, Objects.deString, sw)
	 | SWITCH_R sw => switch(run env, Objects.deReal, sw)

	 | RAISE lamb => raise PACKET(run env lamb)

	 | HANDLE(lamb1, lamb2) =>
	     (run env lamb1
	      handle PACKET pack =>	(* Apply lamb2 to the packet. *)
		let
		  val func = run env lamb2
		  val {arg=lv, body, bodyEnv, recEnv} = Objects.deClosure func
		in
		  run (DynamicEnv.declare(lv, pack,
					  bodyEnv plus (DynamicEnv.REC recEnv)
					 )
		      ) body
		end
	     )

	 | REF lamb => Objects.Ref(run env lamb)

	 | VOID => Objects.void

    val run = fn env => fn lamb =>
      run env lamb
      handle PACKET pack => raise UNCAUGHT pack

    fun RE_RAISE pack = raise(PACKET pack)
    val FAIL_USE = Apply.FAIL_USE
  end;
