(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: MethodExpr.m3                                         *)
(* Last modified on Tue Feb  4 08:37:32 PST 1992 by kalsow         *)
(*      modified on Tue Feb 19 01:32:23 1991 by muller         *)

MODULE MethodExpr;

IMPORT Expr, ExprRep, Type, Emit, ObjectType, Temp, MBuf, Target;
IMPORT Value, ProcType, Method, Error, String;

TYPE
  P = Expr.T BRANDED "MethodExpr.T" OBJECT
        object      : Type.T;
        name        : String.T;
        method      : Value.T;
        holder      : Type.T;
      OVERRIDES
        typeOf       := TypeOf;
        check        := Check;
        compile      := Compile;
        evaluate     := ExprRep.Self;
        fprint       := FPrinter;
        write        := Writer;
        isEqual      := EqCheck;
        getBounds    := ExprRep.NoBounds;
        isWritable   := ExprRep.IsNever;
        isDesignator := ExprRep.IsNever;
	isZeroes     := ExprRep.IsNever;
	note_write   := ExprRep.NotWritable;
	genLiteral   := GenLiteral;
      END;

PROCEDURE New (object: Type.T;  name: String.T;
                  method: Value.T;  holder: Type.T): Expr.T =
  VAR p := NEW (P);
  BEGIN
    ExprRep.Init (p);
    p.object  := object;
    p.name    := name;
    p.method  := method;
    p.holder  := holder;
    RETURN p;
  END New;

PROCEDURE TypeOf (p: P): Type.T =
  VAR offset: INTEGER;  override: BOOLEAN;  mType: Type.T;
  BEGIN
    Method.SplitX (p.method, offset, override, mType);
    RETURN ProcType.MethodSigAsProcSig (mType, p.object);
  END TypeOf;

PROCEDURE Check (p: P;  VAR cs: Expr.CheckState) =
  BEGIN
    Type.Check (p.object);
    Type.Check (p.holder);
    Value.TypeCheck (p.method, cs);
    Type.Check (TypeOf (p));
  END Check;

PROCEDURE EqCheck (a: P;  e: Expr.T): BOOLEAN =
  BEGIN
    TYPECASE e OF
    | NULL => RETURN FALSE;
    | P(b) => RETURN (a.name = b.name)
                 AND Type.IsEqual (a.object, b.object, NIL);
    ELSE      RETURN FALSE;
    END;
  END EqCheck;

PROCEDURE Compile (p: P): Temp.T =
  BEGIN
    Type.Compile (p.object);
    RETURN Temp.AllocMacro (p, FALSE);
  END Compile;

PROCEDURE Writer (p: P;  <*UNUSED*> t1, t2: Temp.T) =
  VAR x: INTEGER;
  BEGIN
    x := ObjectType.MethodOffset (p.holder);
    Emit.OpF ("(((@_methods*)", p.holder);
    Emit.OpF ("(@_TC->defaultMethods + ", p.object);
    IF (x < 0)
      THEN Emit.OpF ("@_TC->methodOffset", p.holder);
      ELSE Emit.OpI ("@", x DIV Target.CHARSIZE);
    END;
    Emit.OpS ("))->@)", p.name);
  END Writer;

PROCEDURE FPrinter (p: P;  map: Type.FPMap;  wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "DEFAULT_METHOD ");
    Type.Fingerprint (p.object, map, wr);
    MBuf.PutText (wr, " . ");
    String.Put (wr, p.name);
  END FPrinter;

PROCEDURE GenLiteral (p: P) =
  BEGIN
    Error.Str (p.name,
    "SRC Modula-3 restriction: default method is not a compile-time constant");
    Emit.Op ("0");
  END GenLiteral;

BEGIN
END MethodExpr.
