/* eval.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: eval.c,v 1.6 1995/05/22 21:02:41 brianp Exp $

$Log: eval.c,v $
 * Revision 1.6  1995/05/22  21:02:41  brianp
 * Release 1.2
 *
 * Revision 1.5  1995/03/04  19:29:44  brianp
 * 1.1 beta revision
 *
 * Revision 1.4  1995/03/02  19:10:24  brianp
 * fixed a bug in Map2Texture1 computation
 *
 * Revision 1.3  1995/02/26  22:00:22  brianp
 * removed debug printf's
 *
 * Revision 1.2  1995/02/25  22:08:16  brianp
 * extensive changes, introduced eval_xd_yd() functions
 *
 * Revision 1.1  1995/02/24  14:22:00  brianp
 * Initial revision
 *
 */



#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "context.h"
#include "draw.h"
#include "list.h"



/*
 * Uncomment this to try computing surface normals by derivative.  This
 * doesn't work yet...
 */
/*#define DERIVATIVE*/




/**********************************************************************/
/*                          Private Functions                         */
/**********************************************************************/

static GLboolean init_flag = 0;
static GLfloat comb_table[MAX_EVAL_ORDER][MAX_EVAL_ORDER];


/*
 * Compute n!
 */
static GLuint fac( GLuint n )
{
   GLuint i, p;

   p = 1;
   for (i=2;i<=n;i++) {
      p = p * i;
   }
   return p;
}


/*
 * Compute combinations nCr.
 */
static GLuint comb( GLuint n, GLuint r )
{
   return fac(n) / (fac(r)*fac(n-r));
/*   return comb_table[n][r];*/
}


/*
 * Compute x^n.
 */
static GLdouble power( GLdouble x, GLint n )
{
   register GLuint i;
   register GLdouble p = 1.0;

   if (n==0) {
      return 1.0;
   }
   else if (n>0) {
      for (i=0;i<n;i++) {
	 p *= x;
      }
      return p;
   }
   else {
      /* n<0 */
      for (i=0;i<-n;i++) {
	 p *= x;
      }
      return 1.0F / p;
   }
}


/*
 * Evaluate ith Bernstein polynomial of degree n for u.
 */
static GLfloat B( GLuint n, GLuint i, GLfloat u )
{
   GLdouble b;

   b = comb_table[n][i] * power(u,i) * power( 1.0-u, n-i );
   if (b<1.0e-35) {
      return 0.0;
   }
   else {
      return (GLfloat) b;
   }
}


#ifdef DERIVATIVE
/*
 * Evaluate the first derivative of the ith Bernstein polynomial of
 * degree n at u.
 */
static GLfloat dB( GLint n, GLint i, GLfloat u )
{
/*
   return comb_table[n][i] * (
			       power(u,i)*(i-n)*power(1.0-u,n-i-1)
			     + i*power(u,i-1)*power(1.0-u,n-i)
			    );
*/
   return comb_table[n][i] * power(1.0-u,n-i-1) * power(u,i-1) *
             ( u*(i-n) + i*(1.0-u) );
}
#endif


/*
 * Return the number of components per control point for any type of
 * evaluator.  Return 0 if bad target.
 */
static GLint components( GLenum target )
{
   switch (target) {
      case GL_MAP1_VERTEX_3:		return 3;
      case GL_MAP1_VERTEX_4:		return 4;
      case GL_MAP1_INDEX:		return 1;
      case GL_MAP1_COLOR_4:		return 4;
      case GL_MAP1_NORMAL:		return 3;
      case GL_MAP1_TEXTURE_COORD_1:	return 1;
      case GL_MAP1_TEXTURE_COORD_2:	return 2;
      case GL_MAP1_TEXTURE_COORD_3:	return 3;
      case GL_MAP1_TEXTURE_COORD_4:	return 4;
      case GL_MAP2_VERTEX_3:		return 3;
      case GL_MAP2_VERTEX_4:		return 4;
      case GL_MAP2_INDEX:		return 1;
      case GL_MAP2_COLOR_4:		return 4;
      case GL_MAP2_NORMAL:		return 3;
      case GL_MAP2_TEXTURE_COORD_1:	return 1;
      case GL_MAP2_TEXTURE_COORD_2:	return 2;
      case GL_MAP2_TEXTURE_COORD_3:	return 3;
      case GL_MAP2_TEXTURE_COORD_4:	return 4;
      default:				return 0;
   }
}



/*
 * Copy evaluator control points from user-specified memory space to
 * a buffer of contiguous control points.
 * Input:  see glMap*f for details
 * Return:  pointer to buffer of contiguous control points or NULL if out
 *          of memory.
 */
static GLfloat *copy_points_f( GLenum target,
			        GLint ustride, GLint uorder,
			        GLint vstride, GLint vorder,
			        const GLfloat *points )
{
   GLfloat *buffer, *p;
   GLuint i, j, k, size;

   size = components( target );

   buffer = (GLfloat *) malloc( uorder * vorder * size * sizeof(GLfloat) );

   if (buffer) {
      p = buffer;
      for (i=0;i<uorder;i++) {
	 for (j=0;j<vorder;j++) {
	    for (k=0;k<size;k++) {
	       *p++ = points[i*ustride+j*vstride+k];
	    }
	 }
      }
   }

   return buffer;
}



/*
 * Same as above but convert doubles to floats.
 */
static GLfloat *copy_points_d( GLenum target,
			        GLint ustride, GLint uorder,
			        GLint vstride, GLint vorder,
			        const GLdouble *points )
{
   GLfloat *buffer, *p;
   GLuint i, j, k, size;

   size = components( target );

   buffer = (GLfloat *) malloc( uorder * vorder * size * sizeof(GLfloat) );

   if (buffer) {
      p = buffer;
      for (i=0;i<uorder;i++) {
	 for (j=0;j<vorder;j++) {
	    for (k=0;k<size;k++) {
	       *p++ = (GLfloat) points[i*ustride+j*vstride+k];
	    }
	 }
      }
   }

   return buffer;
}



/**********************************************************************/
/*                     Internal library functions                      */
/**********************************************************************/


/*
 * Do one-time initialization for evaluators.
 */
void gl_init_eval( void )
{
   /* Compute a table of nCr (combination) values used by the
    * Bernstein polynomial generator.
    */

   if (init_flag==0) {
      GLuint n, r;
      for (n=0;n<MAX_EVAL_ORDER;n++) {
	 for (r=0;r<=n;r++) {
	    comb_table[n][r] = (GLfloat) (fac(n) / (fac(r)*fac(n-r)));
	 }
      }
   }
   init_flag = 1;
}


/*
 * Control points and info are shared by all contexts in the address space.
 * The discard flag indicates whether the current control point data can be
 * free()'d when new control points are given via glMap[12][fd].  It can't
 * freed be when the current control points are also in a display list.
 */

/* TODO: should these be per-context or shared? */

/* Map 1, Vertex_3 */
static GLuint Map1Vertex3order;
static GLfloat Map1Vertex3u1, Map1Vertex3u2;
static GLfloat *Map1Vertex3 = NULL;
static GLboolean DiscardMap1Vertex3 = GL_FALSE;

/* Map 1, Vertex_4 */
static GLuint Map1Vertex4order;
static GLfloat Map1Vertex4u1, Map1Vertex4u2;
static GLfloat *Map1Vertex4 = NULL;
static GLboolean DiscardMap1Vertex4 = GL_FALSE;

/* Map 1, Index */
static GLuint Map1Indexorder;
static GLfloat Map1Indexu1, Map1Indexu2;
static GLfloat *Map1Index = NULL;
static GLboolean DiscardMap1Index = GL_FALSE;

/* Map 1, Color_4 */
static GLuint Map1Color4order;
static GLfloat Map1Color4u1, Map1Color4u2;
static GLfloat *Map1Color4 = NULL;
static GLboolean DiscardMap1Color4 = GL_FALSE;

/* Map 1, Normal */
static GLuint Map1Normalorder;
static GLfloat Map1Normalu1, Map1Normalu2;
static GLfloat *Map1Normal = NULL;
static GLboolean DiscardMap1Normal = GL_FALSE;

/* Map 1, Texture_1 */
static GLuint Map1Texture1order;
static GLfloat Map1Texture1u1, Map1Texture1u2;
static GLfloat *Map1Texture1 = NULL;
static GLboolean DiscardMap1Texture1 = GL_FALSE;

/* Map 1, Texture_2 */
static GLuint Map1Texture2order;
static GLfloat Map1Texture2u1, Map1Texture2u2;
static GLfloat *Map1Texture2 = NULL;
static GLboolean DiscardMap1Texture2 = GL_FALSE;

/* Map 1, Texture_3 */
static GLuint Map1Texture3order;
static GLfloat Map1Texture3u1, Map1Texture3u2;
static GLfloat *Map1Texture3 = NULL;
static GLboolean DiscardMap1Texture3 = GL_FALSE;

/* Map 1, Texture_4 */
static GLuint Map1Texture4order;
static GLfloat Map1Texture4u1, Map1Texture4u2;
static GLfloat *Map1Texture4 = NULL;
static GLboolean DiscardMap1Texture4 = GL_FALSE;


/* Map 2, Vertex_3 */
static GLuint Map2Vertex3uorder;
static GLuint Map2Vertex3vorder;
static GLfloat Map2Vertex3u1, Map2Vertex3u2;
static GLfloat Map2Vertex3v1, Map2Vertex3v2;
static GLfloat *Map2Vertex3 = NULL;
static GLboolean DiscardMap2Vertex3 = GL_FALSE;

/* Map 2, Vertex_4 */
static GLuint Map2Vertex4uorder;
static GLuint Map2Vertex4vorder;
static GLfloat Map2Vertex4u1, Map2Vertex4u2;
static GLfloat Map2Vertex4v1, Map2Vertex4v2;
static GLfloat *Map2Vertex4 = NULL;
static GLboolean DiscardMap2Vertex4 = GL_FALSE;

/* Map 2, Index */
static GLuint Map2Indexuorder;
static GLuint Map2Indexvorder;
static GLfloat Map2Indexu1, Map2Indexu2;
static GLfloat Map2Indexv1, Map2Indexv2;
static GLfloat *Map2Index = NULL;
static GLboolean DiscardMap2Index = GL_FALSE;

/* Map 2, Color_4 */
static GLuint Map2Color4uorder;
static GLuint Map2Color4vorder;
static GLfloat Map2Color4u1, Map2Color4u2;
static GLfloat Map2Color4v1, Map2Color4v2;
static GLfloat *Map2Color4 = NULL;
static GLboolean DiscardMap2Color4 = GL_FALSE;

/* Map 2, Normal */
static GLuint Map2Normaluorder;
static GLuint Map2Normalvorder;
static GLfloat Map2Normalu1, Map2Normalu2;
static GLfloat Map2Normalv1, Map2Normalv2;
static GLfloat *Map2Normal = NULL;
static GLboolean DiscardMap2Normal = GL_FALSE;

/* Map 2, Texture_1 */
static GLuint Map2Texture1uorder;
static GLuint Map2Texture1vorder;
static GLfloat Map2Texture1u1, Map2Texture1u2;
static GLfloat Map2Texture1v1, Map2Texture1v2;
static GLfloat *Map2Texture1 = NULL;
static GLboolean DiscardMap2Texture1 = GL_FALSE;

/* Map 2, Texture_2 */
static GLuint Map2Texture2uorder;
static GLuint Map2Texture2vorder;
static GLfloat Map2Texture2u1, Map2Texture2u2;
static GLfloat Map2Texture2v1, Map2Texture2v2;
static GLfloat *Map2Texture2 = NULL;
static GLboolean DiscardMap2Texture2 = GL_FALSE;

/* Map 2, Texture_3 */
static GLuint Map2Texture3uorder;
static GLuint Map2Texture3vorder;
static GLfloat Map2Texture3u1, Map2Texture3u2;
static GLfloat Map2Texture3v1, Map2Texture3v2;
static GLfloat *Map2Texture3 = NULL;
static GLboolean DiscardMap2Texture3 = GL_FALSE;

/* Map 2, Texture_4 */
static GLuint Map2Texture4uorder;
static GLuint Map2Texture4vorder;
static GLfloat Map2Texture4u1, Map2Texture4u2;
static GLfloat Map2Texture4v1, Map2Texture4v2;
static GLfloat *Map2Texture4 = NULL;
static GLboolean DiscardMap2Texture4 = GL_FALSE;



/*
 * Given a Map target, return a pointer to the corresponding Discard*
 * variable.
 */
static GLboolean *discard_target( GLenum target )
{
   switch (target) {
      case GL_MAP1_VERTEX_3:		return &DiscardMap1Vertex3;
      case GL_MAP1_VERTEX_4:		return &DiscardMap1Vertex4;
      case GL_MAP1_INDEX:		return &DiscardMap1Index;
      case GL_MAP1_COLOR_4:		return &DiscardMap1Color4;
      case GL_MAP1_NORMAL:		return &DiscardMap1Normal;
      case GL_MAP1_TEXTURE_COORD_1:	return &DiscardMap1Texture1;
      case GL_MAP1_TEXTURE_COORD_2:	return &DiscardMap1Texture2;
      case GL_MAP1_TEXTURE_COORD_3:	return &DiscardMap1Texture3;
      case GL_MAP1_TEXTURE_COORD_4:	return &DiscardMap1Texture4;
      case GL_MAP2_VERTEX_3:		return &DiscardMap2Vertex3;
      case GL_MAP2_VERTEX_4:		return &DiscardMap2Vertex4;
      case GL_MAP2_INDEX:		return &DiscardMap2Index;
      case GL_MAP2_COLOR_4:		return &DiscardMap2Color4;
      case GL_MAP2_NORMAL:		return &DiscardMap2Normal;
      case GL_MAP2_TEXTURE_COORD_1:	return &DiscardMap2Texture1;
      case GL_MAP2_TEXTURE_COORD_2:	return &DiscardMap2Texture2;
      case GL_MAP2_TEXTURE_COORD_3:	return &DiscardMap2Texture3;
      case GL_MAP2_TEXTURE_COORD_4:	return &DiscardMap2Texture4;
      default:
	 gl_error( GL_INVALID_ENUM, "discard_target" );
	 return NULL;
   }
}



/*
 * This function is called by the display list deallocator function to
 * specify that a given set of control points are no longer needed.
 * Under certain conditions, we can deallocate the control points memory,
 * otherwise, we mark it as discard-able.
 */
void gl_free_control_points( GLenum target, GLfloat *data )
{
   switch (target) {
      case GL_MAP1_VERTEX_3:
         if (data==Map1Vertex3) {
	    /* The control points in the display list are currently */
	    /* being used so we can mark them as discard-able. */
	    DiscardMap1Vertex3 = GL_TRUE;
	 }
	 else {
	    /* The control points in the display list are not currently */
	    /* being used. */
	    free( data );
	 }
	 break;
      case GL_MAP1_VERTEX_4:
         if (data==Map1Vertex4)
	    DiscardMap1Vertex4 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP1_INDEX:
         if (data==Map1Index)
	    DiscardMap1Index = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP1_COLOR_4:
         if (data==Map1Vertex4)
	    DiscardMap1Vertex4 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP1_NORMAL:
         if (data==Map1Normal)
	    DiscardMap1Normal = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP1_TEXTURE_COORD_1:
         if (data==Map1Texture1)
	    DiscardMap1Texture1 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP1_TEXTURE_COORD_2:
         if (data==Map1Texture2)
	    DiscardMap1Texture2 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP1_TEXTURE_COORD_3:
         if (data==Map1Texture3)
	    DiscardMap1Texture3 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP1_TEXTURE_COORD_4:
         if (data==Map1Texture4)
	    DiscardMap1Texture4 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP2_VERTEX_3:
         if (data==Map2Vertex3)
	    DiscardMap2Vertex3 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP2_VERTEX_4:
         if (data==Map2Vertex4)
	    DiscardMap2Vertex4 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP2_INDEX:
         if (data==Map2Index)
	    DiscardMap2Index = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP2_COLOR_4:
         if (data==Map2Color4)
	    DiscardMap2Color4 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP2_NORMAL:
         if (data==Map2Normal)
	    DiscardMap2Normal = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP2_TEXTURE_COORD_1:
         if (data==Map2Texture1)
	    DiscardMap2Texture1 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP2_TEXTURE_COORD_2:
         if (data==Map2Texture2)
	    DiscardMap2Texture2 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP2_TEXTURE_COORD_3:
         if (data==Map2Texture3)
	    DiscardMap2Texture3 = GL_TRUE;
	 else
	    free( data );
	 break;
      case GL_MAP2_TEXTURE_COORD_4:
         if (data==Map2Texture4)
	    DiscardMap2Texture4 = GL_TRUE;
	 else
	    free( data );
	 break;
      default:
	 gl_error( GL_INVALID_ENUM, "gl_free_control_points" );
   }
}




/**********************************************************************/
/*              Control Point Definition Functions                    */
/**********************************************************************/


/*
 * Internal glMap1{fd} function.  Note that points must be a contiguous
 * array of control points.
 */
void gl_map1( GLenum target, GLfloat u1, GLfloat u2, GLint stride,
	      GLint order, const GLfloat *points )
{
   GLuint k;

   if (CC.Mode!=0) {
      gl_error( GL_INVALID_OPERATION, "glMap1" );
      return;
   }

   if (u1==u2) {
      gl_error( GL_INVALID_VALUE, "glMap1(u1,u2)" );
      return;
   }

   if (order<1 || order>MAX_EVAL_ORDER) {
      gl_error( GL_INVALID_VALUE, "glMap1(order)" );
      return;
   }

   k = components( target );
   if (k==0) {
      gl_error( GL_INVALID_ENUM, "glMap1(target)" );
   }

   if (stride < k) {
      gl_error( GL_INVALID_VALUE, "glMap1(stride)" );
      return;
   }

   switch (target) {
      case GL_MAP1_VERTEX_3:
         Map1Vertex3order = order;
	 Map1Vertex3u1 = u1;
	 Map1Vertex3u2 = u2;
	 if (Map1Vertex3 && DiscardMap1Vertex3) {
	    free( Map1Vertex3 );
	 }
	 DiscardMap1Vertex3 = GL_FALSE;
	 Map1Vertex3 = (GLfloat *) points;
	 break;
      case GL_MAP1_VERTEX_4:
         Map1Vertex4order = order;
	 Map1Vertex4u1 = u1;
	 Map1Vertex4u2 = u2;
	 if (Map1Vertex4 && DiscardMap1Vertex4) {
	    free( Map1Vertex4 );
	 }
	 DiscardMap1Vertex4 = GL_FALSE;
	 Map1Vertex4 = (GLfloat *) points;
	 break;
      case GL_MAP1_INDEX:
         Map1Indexorder = order;
	 Map1Indexu1 = u1;
	 Map1Indexu2 = u2;
	 if (Map1Index && DiscardMap1Index) {
	    free( Map1Index );
	 }
	 DiscardMap1Index = GL_FALSE;
	 Map1Index = (GLfloat *) points;
	 break;
      case GL_MAP1_COLOR_4:
         Map1Color4order = order;
	 Map1Color4u1 = u1;
	 Map1Color4u2 = u2;
	 if (Map1Color4 && DiscardMap1Color4) {
	    free( Map1Color4 );
	 }
	 DiscardMap1Color4 = GL_FALSE;
	 Map1Color4 = (GLfloat *) points;
	 break;
      case GL_MAP1_NORMAL:
         Map1Normalorder = order;
	 Map1Normalu1 = u1;
	 Map1Normalu2 = u2;
	 if (Map1Normal && DiscardMap1Normal) {
	    free( Map1Normal );
	 }
	 DiscardMap1Normal = GL_FALSE;
	 Map1Normal = (GLfloat *) points;
	 break;
      case GL_MAP1_TEXTURE_COORD_1:
         Map1Texture1order = order;
	 Map1Texture1u1 = u1;
	 Map1Texture1u2 = u2;
	 if (Map1Texture1 && DiscardMap1Texture1) {
	    free( Map1Texture1 );
	 }
	 DiscardMap1Texture1 = GL_FALSE;
	 Map1Texture1 = (GLfloat *) points;
	 break;
      case GL_MAP1_TEXTURE_COORD_2:
         Map1Texture2order = order;
	 Map1Texture2u1 = u1;
	 Map1Texture2u2 = u2;
	 if (Map1Texture2 && DiscardMap1Texture2) {
	    free( Map1Texture2 );
	 }
	 DiscardMap1Texture2 = GL_FALSE;
	 Map1Texture2 = (GLfloat *) points;
	 break;
      case GL_MAP1_TEXTURE_COORD_3:
         Map1Texture3order = order;
	 Map1Texture3u1 = u1;
	 Map1Texture3u2 = u2;
	 if (Map1Texture3 && DiscardMap1Texture3) {
	    free( Map1Texture3 );
	 }
	 DiscardMap1Texture3 = GL_FALSE;
	 Map1Texture3 = (GLfloat *) points;
	 break;
      case GL_MAP1_TEXTURE_COORD_4:
         Map1Texture4order = order;
	 Map1Texture4u1 = u1;
	 Map1Texture4u2 = u2;
	 if (Map1Texture4 && DiscardMap1Texture4) {
	    free( Map1Texture4 );
	 }
	 DiscardMap1Texture4 = GL_FALSE;
	 Map1Texture4 = (GLfloat *) points;
	 break;
      default:
         gl_error( GL_INVALID_ENUM, "glMap1(target)" );
   }
}



void glMap1f( GLenum target, GLfloat u1, GLfloat u2, GLint stride,
	      GLint order, const GLfloat *points )
{
   float *p;

   p = copy_points_f( target, stride, order, 0, 1,  points );
   if (!p) {
      gl_error( GL_OUT_OF_MEMORY, "glMap1f" );
      return;
   }

   if (CC.CompileFlag) {
      gl_save_map1( target, u1, u2, stride, order, p );
   }
   if (CC.ExecuteFlag) {
      gl_map1( target, u1, u2, stride, order, p );
      if (!CC.CompileFlag) {
	 /* get pointer to the discard flag for the given target */
	 GLboolean *discard = discard_target( target );
  	 /* the control points can be discarded when new ones are bound */
	 *discard = GL_TRUE;
      }
   }
}



void glMap1d( GLenum target, GLdouble u1, GLdouble u2, GLint stride,
	      GLint order, const GLdouble *points )
{
   float *p;

   p = copy_points_d( target, stride, order, 0, 1,  points );
   if (!p) {
      gl_error( GL_OUT_OF_MEMORY, "glMap1d" );
      return;
   }

   if (CC.CompileFlag) {
      gl_save_map1( target, u1, u2, stride, order, p );
   }
   if (CC.ExecuteFlag) {
      gl_map1( target, u1, u2, stride, order, p );
      if (!CC.CompileFlag) {
	 /* get pointer to the discard flag for the given target */
	 GLboolean *discard = discard_target( target );
  	 /* the control points can be discarded when new ones are bound */
	 *discard = GL_TRUE;
      }
   }
}




void gl_map2( GLenum target,
	      GLfloat u1, GLfloat u2, GLint ustride, GLint uorder,
	      GLfloat v1, GLfloat v2, GLint vstride, GLint vorder,
	      const GLfloat *points )
{
   GLuint k;

   if (CC.Mode!=0) {
      gl_error( GL_INVALID_OPERATION, "glMap2" );
      return;
   }

   if (u1==u2) {
      gl_error( GL_INVALID_VALUE, "glMap2(u1,u2)" );
      return;
   }

   if (v1==v2) {
      gl_error( GL_INVALID_VALUE, "glMap2(v1,v2)" );
      return;
   }

   if (uorder<1 || uorder>MAX_EVAL_ORDER) {
      gl_error( GL_INVALID_VALUE, "glMap2(uorder)" );
      return;
   }

   if (vorder<1 || vorder>MAX_EVAL_ORDER) {
      gl_error( GL_INVALID_VALUE, "glMap2(vorder)" );
      return;
   }

   k = components( target );
   if (k==0) {
      gl_error( GL_INVALID_ENUM, "glMap2(target)" );
   }

   if (ustride < k) {
      gl_error( GL_INVALID_VALUE, "glMap2(ustride)" );
      return;
   }
   if (vstride < k) {
      gl_error( GL_INVALID_VALUE, "glMap2(vstride)" );
      return;
   }

   switch (target) {
      case GL_MAP2_VERTEX_3:
         Map2Vertex3uorder = uorder;
	 Map2Vertex3u1 = u1;
	 Map2Vertex3u2 = u2;
         Map2Vertex3vorder = vorder;
	 Map2Vertex3v1 = v1;
	 Map2Vertex3v2 = v2;
	 if (Map2Vertex3 && DiscardMap2Vertex3) {
	    free( Map2Vertex3 );
	 }
	 DiscardMap2Vertex3 = GL_FALSE;
	 Map2Vertex3 = (GLfloat *) points;
	 break;
      case GL_MAP2_VERTEX_4:
         Map2Vertex4uorder = uorder;
	 Map2Vertex4u1 = u1;
	 Map2Vertex4u2 = u2;
         Map2Vertex4vorder = vorder;
	 Map2Vertex4v1 = v1;
	 Map2Vertex4v2 = v2;
	 if (Map2Vertex4 && DiscardMap2Vertex4) {
	    free( Map2Vertex4 );
	 }
	 DiscardMap2Vertex4 = GL_FALSE;
	 Map2Vertex4 = (GLfloat *) points;
	 break;
      case GL_MAP2_INDEX:
         Map2Indexuorder = uorder;
	 Map2Indexu1 = u1;
	 Map2Indexu2 = u2;
         Map2Indexvorder = vorder;
	 Map2Indexv1 = v1;
	 Map2Indexv2 = v2;
	 if (Map2Index && DiscardMap2Index) {
	    free( Map2Index );
	 }
	 DiscardMap2Index = GL_FALSE;
	 Map2Index = (GLfloat *) points;
	 break;
      case GL_MAP2_COLOR_4:
         Map2Color4uorder = uorder;
	 Map2Color4u1 = u1;
	 Map2Color4u2 = u2;
         Map2Color4vorder = vorder;
	 Map2Color4v1 = v1;
	 Map2Color4v2 = v2;
	 if (Map2Color4 && DiscardMap2Color4) {
	    free( Map2Color4 );
	 }
	 DiscardMap2Color4 = GL_FALSE;
	 Map2Color4 = (GLfloat *) points;
	 break;
      case GL_MAP2_NORMAL:
         Map2Normaluorder = uorder;
	 Map2Normalu1 = u1;
	 Map2Normalu2 = u2;
         Map2Normalvorder = vorder;
	 Map2Normalv1 = v1;
	 Map2Normalv2 = v2;
	 if (Map2Normal && DiscardMap2Normal) {
	    free( Map2Normal );
	 }
	 DiscardMap2Normal = GL_FALSE;
	 Map2Normal = (GLfloat *) points;
	 break;
      case GL_MAP2_TEXTURE_COORD_1:
         Map2Texture1uorder = uorder;
	 Map2Texture1u1 = u1;
	 Map2Texture1u2 = u2;
         Map2Texture1vorder = vorder;
	 Map2Texture1v1 = v1;
	 Map2Texture1v2 = v2;
	 if (Map2Texture1 && DiscardMap2Texture1) {
	    free( Map2Texture1 );
	 }
	 DiscardMap2Texture1 = GL_FALSE;
	 Map2Texture1 = (GLfloat *) points;
	 break;
      case GL_MAP2_TEXTURE_COORD_2:
         Map2Texture2uorder = uorder;
	 Map2Texture2u1 = u1;
	 Map2Texture2u2 = u2;
         Map2Texture2vorder = vorder;
	 Map2Texture2v1 = v1;
	 Map2Texture2v2 = v2;
	 if (Map2Texture2 && DiscardMap2Texture2) {
	    free( Map2Texture2 );
	 }
	 DiscardMap2Texture2 = GL_FALSE;
	 Map2Texture2 = (GLfloat *) points;
	 break;
      case GL_MAP2_TEXTURE_COORD_3:
         Map2Texture3uorder = uorder;
	 Map2Texture3u1 = u1;
	 Map2Texture3u2 = u2;
         Map2Texture3vorder = vorder;
	 Map2Texture3v1 = v1;
	 Map2Texture3v2 = v2;
	 if (Map2Texture3 && DiscardMap2Texture3) {
	    free( Map2Texture3 );
	 }
	 DiscardMap2Texture3 = GL_FALSE;
	 Map2Texture3 = (GLfloat *) points;
	 break;
      case GL_MAP2_TEXTURE_COORD_4:
         Map2Texture4uorder = uorder;
	 Map2Texture4u1 = u1;
	 Map2Texture4u2 = u2;
         Map2Texture4vorder = vorder;
	 Map2Texture4v1 = v1;
	 Map2Texture4v2 = v2;
	 if (Map2Texture4 && DiscardMap2Texture4) {
	    free( Map2Texture4 );
	 }
	 DiscardMap2Texture4 = GL_FALSE;
	 Map2Texture4 = (GLfloat *) points;
	 break;
      default:
         gl_error( GL_INVALID_ENUM, "glMap1f(target)" );
   }

}


   
void glMap2f( GLenum target,
	      GLfloat u1, GLfloat u2, GLint ustride, GLint uorder,
	      GLfloat v1, GLfloat v2, GLint vstride, GLint vorder,
	      const GLfloat *points )
{
   GLfloat *p;

   p = copy_points_f( target, ustride, uorder, vstride, vorder, points );
   if (!p) {
      gl_error( GL_OUT_OF_MEMORY, "glMap2f" );
      return;
   }

   if (CC.CompileFlag) {
      gl_save_map2( target, u1, u2, ustride, uorder,
		    v1, v2, vstride, vorder, p );
   }
   if (CC.ExecuteFlag) {
      gl_map2( target, u1, u2, ustride, uorder,
	       v1, v2, vstride, vorder, p );
      if (!CC.CompileFlag) {
	 /* get pointer to the discard flag for the given target */
	 GLboolean *discard = discard_target( target );
  	 /* the control points can be discarded when new ones are bound */
	 *discard = GL_TRUE;
      }
   }
}



void glMap2d( GLenum target,
	      GLdouble u1, GLdouble u2, GLint ustride, GLint uorder,
	      GLdouble v1, GLdouble v2, GLint vstride, GLint vorder,
	      const GLdouble *points )
{
   GLfloat *p;

   p = copy_points_d( target, ustride, uorder, vstride, vorder, points );
   if (!p) {
      gl_error( GL_OUT_OF_MEMORY, "glMap2d" );
      return;
   }

   if (CC.CompileFlag) {
      gl_save_map2( target, u1, u2, ustride, uorder,
		    v1, v2, vstride, vorder, p );
   }
   if (CC.ExecuteFlag) {
      gl_map2( target, u1, u2, ustride, uorder,
	       v1, v2, vstride, vorder, p );
      if (!CC.CompileFlag) {
	 /* get pointer to the discard flag for the given target */
	 GLboolean *discard = discard_target( target );
  	 /* the control points can be discarded when new ones are bound */
	 *discard = GL_TRUE;
      }
   }
}




/**********************************************************************/
/*                        Evaluation functions                        */
/**********************************************************************/


/*
 * Evaluate a curve at u to generate a 1-D coordinate.
 */
static void eval_1d_1d( GLfloat u, GLuint order, const GLfloat *cp,
		        GLfloat *coord )
{
   register GLuint i, n;
   register GLfloat b, t0;

   n = order - 1;
   t0 = 0.0;
   for (i=0;i<=n;i++) {
      b = B( n, i, u );
      t0 += b * *cp++;
   }
   coord[0] = t0;
}


/*
 * Evaluate a curve at u to generate a 2-D coordinate.
 */
static void eval_1d_2d( GLfloat u, GLuint order, const GLfloat *cp,
		        GLfloat *coord )
{
   register GLuint i, n;
   register GLfloat b, t0, t1;

   n = order - 1;
   t0 = t1 = 0.0;
   for (i=0;i<=n;i++) {
      b = B( n, i, u );
      t0 += b * *cp++;
      t1 += b * *cp++;
   }
   coord[0] = t0;
   coord[1] = t1;
}


/*
 * Evaluate a curve at u to generate a 3-D coordinate.
 */
static void eval_1d_3d( GLfloat u, GLuint order, const GLfloat *cp,
		        GLfloat *coord )
{
   register GLuint i, n;
   register GLfloat b, t0, t1, t2;

   n = order - 1;
   t0 = t1 = t2 = 0.0;
   for (i=0;i<=n;i++) {
      b = B( n, i, u );
      t0 += b * *cp++;
      t1 += b * *cp++;
      t2 += b * *cp++;
   }
   coord[0] = t0;
   coord[1] = t1;
   coord[2] = t2;
}


/*
 * Evaluate a curve at u to generate a 4-D coordinate.
 */
static void eval_1d_4d( GLfloat u, GLuint order, const GLfloat *cp,
		        GLfloat *coord )
{
   register GLuint i, n;
   register GLfloat b, t0, t1, t2, t3;

   n = order - 1;
   t0 = t1 = t2 = t3 = 0.0;
   for (i=0;i<=n;i++) {
      b = B( n, i, u );
      t0 += b * *cp++;
      t1 += b * *cp++;
      t2 += b * *cp++;
      t3 += b * *cp++;
   }
   coord[0] = t0;
   coord[1] = t1;
   coord[2] = t2;
   coord[3] = t3;
}



void gl_evalcoord1( GLfloat u )
{
   GLfloat vertex[4];
   GLfloat index;
   GLfloat color[4];
   GLfloat normal[3];
   GLfloat texcoord[4];
   register GLfloat *cp;
   register GLfloat uu;
   register GLuint i;
   register GLfloat t0, t1, t2, t3;

   /** Vertex **/
   if (CC.Eval.Map1Vertex4) {
      uu = (u-Map1Vertex4u1) / (Map1Vertex4u2-Map1Vertex4u1);
      eval_1d_4d( uu, Map1Vertex4order, Map1Vertex4, vertex );
   }
   else if (CC.Eval.Map1Vertex3) {
      uu = (u-Map1Vertex3u1) / (Map1Vertex3u2-Map1Vertex3u1);
      eval_1d_3d( uu, Map1Vertex3order, Map1Vertex3, vertex );
      vertex[3] = 1.0;
   }

   /** Color Index **/
   if (CC.Eval.Map1Index) {
      uu = (u-Map1Indexu1) / (Map1Indexu2-Map1Indexu1);
      eval_1d_1d( uu, Map1Indexorder, Map1Index, &index );
   }
   else {
      index = (GLfloat) CC.Current.Index;
   }

   /** Color **/
   if (CC.Eval.Map1Color4) {
      uu = (u-Map1Color4u1) / (Map1Color4u2-Map1Color4u1);
      eval_1d_4d( uu, Map1Color4order, Map1Color4, color );
   }
   else {
      color[0] = CC.Current.Color[0];
      color[1] = CC.Current.Color[1];
      color[2] = CC.Current.Color[2];
      color[3] = CC.Current.Color[3];
   }

   /** Normal Vector **/
   if (CC.Eval.Map1Normal) {
      uu = (u-Map1Normalu1) / (Map1Normalu2-Map1Normalu1);
      eval_1d_3d( uu, Map1Normalorder, Map1Normal, normal );
   }
   else {
      normal[0] = CC.Current.Normal[0];
      normal[1] = CC.Current.Normal[1];
      normal[2] = CC.Current.Normal[2];
   }

   /** Texture Coordinates **/
   if (CC.Eval.Map1TextureCoord4) {
      uu = (u-Map1Texture4u1) / (Map1Texture4u2-Map1Texture4u1);
      eval_1d_4d( uu, Map1Texture4order, Map1Texture4, texcoord );
   }
   else if (CC.Eval.Map1TextureCoord3) {
      uu = (u-Map1Texture3u1) / (Map1Texture3u2-Map1Texture3u1);
      eval_1d_3d( uu, Map1Texture3order, Map1Texture3, texcoord );
      texcoord[3] = 1.0;
   }
   else if (CC.Eval.Map1TextureCoord2) {
      uu = (u-Map1Texture2u1) / (Map1Texture2u2-Map1Texture2u1);
      eval_1d_2d( uu, Map1Texture2order, Map1Texture2, texcoord );
      texcoord[2] = 0.0;
      texcoord[3] = 1.0;
   }
   else if (CC.Eval.Map1TextureCoord1) {
      uu = (u-Map1Texture1u1) / (Map1Texture1u2-Map1Texture1u1);
      eval_1d_1d( uu, Map1Texture1order, Map1Texture1, texcoord );
      texcoord[1] = 0.0;
      texcoord[2] = 0.0;
      texcoord[3] = 1.0;
   }
   else {
      texcoord[0] = CC.Current.TexCoord[0];
      texcoord[1] = CC.Current.TexCoord[1];
      texcoord[2] = CC.Current.TexCoord[2];
      texcoord[3] = CC.Current.TexCoord[3];
   }

   gl_eval_vertex( vertex, normal, color, index, texcoord );
}



void glEvalCoord1f( GLfloat u )
{
   if (CC.CompileFlag) {
      gl_save_evalcoord1( u );
   }
   if (CC.ExecuteFlag) {
      gl_evalcoord1( u );
   }
}


void glEvalCoord1d( GLdouble u )
{
   if (CC.CompileFlag) {
      gl_save_evalcoord1( (GLfloat) u );
   }
   if (CC.ExecuteFlag) {
      gl_evalcoord1( (GLfloat) u );
   }
}




/*
 * Evaluate a surface at (u,v) to generate a 1-D coordinate.
 */
static void eval_2d_1d( GLfloat u, GLfloat v,
		        GLuint uorder, GLuint vorder,
		        const GLfloat *cp,
		        GLfloat *coord )
{
   register GLuint n, m, i, j;
   register GLfloat t0;
   register GLfloat bi, b;

   n = uorder-1;
   m = vorder-1;
   t0 = 0.0;
   for (i=0;i<=n;i++) {
      bi = B( n, i, u );
      for (j=0;j<=m;j++) {
	 b = bi * B( m, j, v );
	 t0 += b * *cp++;
      }
   }
   coord[0] = t0;
}



/*
 * Evaluate a surface at (u,v) to generate a 2-D coordinate.
 */
static void eval_2d_2d( GLfloat u, GLfloat v,
		        GLuint uorder, GLuint vorder,
		        const GLfloat *cp,
		        GLfloat *coord )
{
   register GLuint n, m, i, j;
   register GLfloat t0, t1;
   register GLfloat bi, b;

   n = uorder-1;
   m = vorder-1;
   t0 = t1 = 0.0;
   for (i=0;i<=n;i++) {
      bi = B( n, i, u );
      for (j=0;j<=m;j++) {
	 b =  bi * B( m, j, v );
	 t0 += b * *cp++;
	 t1 += b * *cp++;
      }
   }
   coord[0] = t0;
   coord[1] = t1;
}



/*
 * Evaluate a surface at (u,v) to generate a 3-D coordinate.
 */
static void eval_2d_3d( GLfloat u, GLfloat v,
		        GLuint uorder, GLuint vorder,
		        const GLfloat *cp,
		        GLfloat *coord )
{
   register GLuint n, m, i, j;
   register GLfloat t0, t1, t2;
   register GLfloat bi, b;

   n = uorder-1;
   m = vorder-1;
   t0 = t1 = t2 = 0.0;
   for (i=0;i<=n;i++) {
      bi = B( n, i, u );
      for (j=0;j<=m;j++) {
	 b =  bi * B( m, j, v );
	 t0 += b * *cp++;
	 t1 += b * *cp++;
	 t2 += b * *cp++;
      }
   }
   coord[0] = t0;
   coord[1] = t1;
   coord[2] = t2;
}



/*
 * Evaluate a surface at (u,v) to generate a 4-D coordinate.
 */
static void eval_2d_4d( GLfloat u, GLfloat v,
		        GLuint uorder, GLuint vorder,
		        const GLfloat *cp,
		        GLfloat *coord )
{
   register GLuint n, m, i, j;
   register GLfloat t0, t1, t2, t3;
   register GLfloat bi, b;

   n = uorder-1;
   m = vorder-1;
   t0 = t1 = t2 = t3 = 0.0;
   for (i=0;i<=n;i++) {
      bi = B( n, i, u );
      for (j=0;j<=m;j++) {
	 b =  bi * B( m, j, v );
	 t0 += b * *cp++;
	 t1 += b * *cp++;
	 t2 += b * *cp++;
	 t3 += b * *cp++;
      }
   }
   coord[0] = t0;
   coord[1] = t1;
   coord[2] = t2;
   coord[3] = t3;
}



void gl_evalcoord2( GLfloat u, GLfloat v )
{
   GLfloat vertex[4];
   GLfloat normal[3];
   GLfloat color[4];
   GLfloat texcoord[4];
   GLfloat index;
   GLfloat uu, vv;

   /** Vertex **/
   if (CC.Eval.Map2Vertex4) {
      uu = (u-Map2Vertex4u1) / (Map2Vertex4u2-Map2Vertex4u1);
      vv = (v-Map2Vertex4v1) / (Map2Vertex4v2-Map2Vertex4v1);
      eval_2d_4d( uu, vv, Map2Vertex4uorder, Map2Vertex4vorder, Map2Vertex4,
		  vertex );
   }
   else if (CC.Eval.Map2Vertex3) {
      uu = (u-Map2Vertex3u1) / (Map2Vertex3u2-Map2Vertex3u1);
      vv = (v-Map2Vertex3v1) / (Map2Vertex3v2-Map2Vertex3v1);
      eval_2d_3d( uu, vv, Map2Vertex3uorder, Map2Vertex3vorder, Map2Vertex3,
		  vertex );
      vertex[3] = 1.0;
   }

   /** Color Index **/
   if (CC.Eval.Map2Index) {
      uu = (u-Map2Indexu1) / (Map2Indexu2-Map2Indexu1);
      vv = (v-Map2Indexv1) / (Map2Indexv2-Map2Indexv1);
      eval_2d_1d( uu, vv, Map2Indexuorder, Map2Indexvorder, Map2Index, &index);
   }
   else {
      index = (GLfloat) CC.Current.Index;
   }

   /** Color **/
   if (CC.Eval.Map2Color4) {
      uu = (u-Map2Color4u1) / (Map2Color4u2-Map2Color4u1);
      vv = (v-Map2Color4v1) / (Map2Color4v2-Map2Color4v1);
      eval_2d_4d( uu, vv, Map2Color4uorder, Map2Color4vorder, Map2Color4,
		  color );
   }
   else {
      color[0] = CC.Current.Color[0];
      color[1] = CC.Current.Color[1];
      color[2] = CC.Current.Color[2];
      color[3] = CC.Current.Color[3];
   }

   /** Normal **/
   if (CC.Eval.AutoNormal && (CC.Eval.Map2Vertex3 || CC.Eval.Map2Vertex4)) {
#ifdef DERIVATIVE
      /* Compute surface normal using derivatives.  This doesn't work! */
      GLuint i, j, n, m;
      GLfloat t0, t1, t2, t3;
      GLfloat b, bi;
      GLfloat *cp;
      GLfloat du[3], dv[3], mag;

      if (CC.Eval.Map2Vertex4) {
	 uu = (u-Map2Vertex4u1) / (Map2Vertex4u2-Map2Vertex4u1);
	 vv = (v-Map2Vertex4v1) / (Map2Vertex4v2-Map2Vertex4v1);
	 n = Map2Vertex4uorder-1;
	 m = Map2Vertex4vorder-1;

	 /* Compute derivative at (uu,vv) with respect to u */
	 t0 = t1 = t2 = t3 = 0.0;
	 cp = Map2Vertex4;
	 for (i=0;i<=n;i++) {
	    bi = dB( n, i, uu );
	    for (j=0;j<=m;j++) {
	       b = bi * B( m, j, vv );
	       t0 += b * *cp++;
	       t1 += b * *cp++;
	       t2 += b * *cp++;
	       t3 += b * *cp++;
	    }
	 }
	 du[0] = t0/t3;  du[1] = t1/t3;  du[2] = t2/t3;

	 /* Compute derivative at (uu,vv) with respect to v */
	 t0 = t1 = t2 = t3 = 0.0;
	 cp = Map2Vertex4;
	 for (i=0;i<=n;i++) {
	    bi = B( n, i, uu );
	    for (j=0;j<=m;j++) {
	       b = bi * dB( m, j, vv );
	       t0 += b * *cp++;
	       t1 += b * *cp++;
	       t2 += b * *cp++;
	       t3 += b * *cp++;
	    }
	 }
	 dv[0] = t0/t3;  dv[1] = t1/t3;  dv[2] = t2/t3;
      }
      else {
	 /* Map2Vertex3 */
	 uu = (u-Map2Vertex3u1) / (Map2Vertex3u2-Map2Vertex3u1);
	 vv = (v-Map2Vertex3v1) / (Map2Vertex3v2-Map2Vertex3v1);
	 n = Map2Vertex3uorder-1;
	 m = Map2Vertex3vorder-1;

	 /* Compute derivative at (uu,vv) with respect to u */
	 t0 = t1 = t2 = 0.0;
	 cp = Map2Vertex3;
	 for (i=0;i<=n;i++) {
	    bi = dB( n, i, uu );
	    for (j=0;j<=m;j++) {
	       b = bi * B( m, j, vv );
	       t0 += b * *cp++;
	       t1 += b * *cp++;
	       t2 += b * *cp++;
	    }
	 }
	 du[0] = t0;  du[1] = t1;  du[2] = t2;

	 /* Compute derivative at (uu,vv) with respect to v */
	 t0 = t1 = t2 = 0.0;
	 cp = Map2Vertex3;
	 for (i=0;i<=n;i++) {
	    bi = B( n, i, uu );
	    for (j=0;j<=m;j++) {
	       b = bi * dB( m, j, vv );
	       t0 += b * *cp++;
	       t1 += b * *cp++;
	       t2 += b * *cp++;
	    }
	 }
	 dv[0] = t0;  dv[1] = t1;  dv[2] = t2;
      }

      /* Compute normal = du X dv */
      normal[0] =   du[1]*dv[2] - du[2]*dv[1];
      normal[1] = -(du[0]*dv[2] - du[2]*dv[0]);
      normal[2] =   du[0]*dv[1] - du[1]*dv[0];
      mag = sqrt( normal[0]*normal[0] + normal[1]*normal[1]
		 + normal[2]*normal[2] );
      if (mag>0.000001) {
	 normal[0] /= mag;
	 normal[1] /= mag;
	 normal[2] /= mag;
      }
#else
      /* compute normal analytically */
      float p[4], q[4], vp[3], vq[3];
      float d = 0.02;  /*0.002*/  /* TODO: is this OK??? */
      float mag;
      GLfloat sign = 1.0;

      if (CC.Eval.Map2Vertex4) {
	 uu = (u-Map2Vertex4u1) / (Map2Vertex4u2-Map2Vertex4u1);
	 vv = (v-Map2Vertex4v1) / (Map2Vertex4v2-Map2Vertex4v1);
	 /* Compute point P */
	 if (uu==1.0) {
	    eval_2d_4d( uu-d, vv, Map2Vertex4uorder, Map2Vertex4vorder,
		        Map2Vertex4, p );
	    sign *= -1.0;
	 }
	 else {
	    eval_2d_4d( uu+d, vv, Map2Vertex4uorder, Map2Vertex4vorder,
		        Map2Vertex4, p );
	 }
	 p[0] /= p[3];
	 p[1] /= p[3];
	 p[2] /= p[3];
	 /* Compute point Q */
	 if (vv==1.0) {
	    eval_2d_4d( uu, vv-d, Map2Vertex4uorder, Map2Vertex4vorder,
		        Map2Vertex4, q );
	 }
	 else {
	    eval_2d_4d( uu, vv+d, Map2Vertex4uorder, Map2Vertex4vorder,
		        Map2Vertex4, q );
	 }
	 q[0] /= q[3];
	 q[1] /= q[3];
	 q[2] /= q[3];
      }
      else {
	 /* Map2Vertex3 */
	 uu = (u-Map2Vertex3u1) / (Map2Vertex3u2-Map2Vertex3u1);
	 vv = (v-Map2Vertex3v1) / (Map2Vertex3v2-Map2Vertex3v1);
	 /* Compute point P */
	 if (uu==1.0) {
	    eval_2d_3d( uu-d, vv,
		        Map2Vertex3uorder, Map2Vertex3vorder, Map2Vertex3, p );
	    sign *= -1.0;
	 }
	 else {
	    eval_2d_3d( uu+d, vv,
		        Map2Vertex3uorder, Map2Vertex3vorder, Map2Vertex3, p );
	 }
	 /* Compute point Q */
	 if (vv==1.0) {
	    eval_2d_3d( uu, vv-d,
		        Map2Vertex3uorder, Map2Vertex3vorder, Map2Vertex3, q );
	    sign *= -1.0;
	 }
	 else {
	    eval_2d_3d( uu, vv+d,
		        Map2Vertex3uorder, Map2Vertex3vorder, Map2Vertex3, q );
	 }
      }
      /* compute normal via cross product */
      /* TODO: vp X vq  or  vq X vp ??? */
      vp[0] = p[0] - vertex[0];
      vp[1] = p[1] - vertex[1];
      vp[2] = p[2] - vertex[2];
      vq[0] = q[0] - vertex[0];
      vq[1] = q[1] - vertex[1];
      vq[2] = q[2] - vertex[2];
      normal[0] =  (vp[1]*vq[2] - vp[2]*vq[1]) * sign;
      normal[1] = -(vp[0]*vq[2] - vp[2]*vq[0]) * sign;
      normal[2] =  (vp[0]*vq[1] - vp[1]*vq[0]) * sign;
/*
      normal[0] =   vq[1]*vp[2] - vq[2]*vp[1];
      normal[1] = -(vq[0]*vp[2] - vq[2]*vp[0]);
      normal[2] =   vq[0]*vp[1] - vq[1]*vp[0];
*/
      mag = sqrt( normal[0]*normal[0] + normal[1]*normal[1]
		 + normal[2]*normal[2] );
      if (mag>0.0) {
	 normal[0] /= mag;
	 normal[1] /= mag;
	 normal[2] /= mag;
      }
      else {
	 /* TODO: The surface normal should never = <0,0,0>.  This is a hack */
	 normal[0] = 0.0;
	 normal[1] = 0.0;
	 normal[2] = 1.0;
      }
#endif
   }
   else if (CC.Eval.Map2Normal) {
      uu = (u-Map2Normalu1) / (Map2Normalu2-Map2Normalu1);
      vv = (v-Map2Normalv1) / (Map2Normalv2-Map2Normalv1);
      eval_2d_3d( uu, vv, Map2Normaluorder, Map2Normalvorder, Map2Normal,
		  normal );
   }
   else {
      normal[0] = CC.Current.Normal[0];
      normal[1] = CC.Current.Normal[1];
      normal[2] = CC.Current.Normal[2];
   }

   /** Texture Coordinates **/
   if (CC.Eval.Map2TextureCoord4) {
      uu = (u-Map2Texture4u1) / (Map2Texture4u2-Map2Texture4u1);
      vv = (v-Map2Texture4v1) / (Map2Texture4v2-Map2Texture4v1);
      eval_2d_4d( uu, vv, Map2Texture4uorder, Map2Texture4vorder, Map2Texture4,
		  texcoord );
   }
   else if (CC.Eval.Map2TextureCoord3) {
      uu = (u-Map2Texture3u1) / (Map2Texture3u2-Map2Texture3u1);
      vv = (v-Map2Texture3v1) / (Map2Texture3v2-Map2Texture3v1);
      eval_2d_3d( uu, vv, Map2Texture3uorder, Map2Texture3vorder, Map2Texture3,
		  texcoord );
      texcoord[3] = 1.0;
   }
   else if (CC.Eval.Map2TextureCoord2) {
      uu = (u-Map2Texture2u1) / (Map2Texture2u2-Map2Texture2u1);
      vv = (v-Map2Texture2v1) / (Map2Texture2v2-Map2Texture2v1);
      eval_2d_2d( uu, vv, Map2Texture2uorder, Map2Texture2vorder, Map2Texture2,
		  texcoord );
      texcoord[2] = 0.0;
      texcoord[3] = 1.0;
   }
   else if (CC.Eval.Map2TextureCoord1) {
      uu = (u-Map2Texture1u1) / (Map2Texture1u2-Map2Texture1u1);
      vv = (v-Map2Texture1v1) / (Map2Texture1v2-Map2Texture1v1);
      eval_2d_1d( uu, vv, Map2Texture1uorder, Map2Texture1vorder, Map2Texture1,
		  texcoord );
      texcoord[1] = 0.0;
      texcoord[2] = 0.0;
      texcoord[3] = 1.0;
   }
   else {
      texcoord[0] = CC.Current.TexCoord[0];
      texcoord[1] = CC.Current.TexCoord[1];
      texcoord[2] = CC.Current.TexCoord[2];
      texcoord[3] = CC.Current.TexCoord[3];
   }

   gl_eval_vertex( vertex, normal, color, index, texcoord );
}



void glEvalCoord2f( GLfloat u, GLfloat v )
{
   if (CC.CompileFlag) {
      gl_save_evalcoord2( u, v );
   }
   if (CC.ExecuteFlag) {
      gl_evalcoord2( u, v );
   }
}


void glEvalCoord2d( GLdouble u, GLdouble v )
{
   if (CC.CompileFlag) {
      gl_save_evalcoord2( (GLfloat) u, (GLfloat) v );
   }
   if (CC.ExecuteFlag) {
      gl_evalcoord2( (GLfloat) u, (GLfloat) v );
   }
}





/**********************************************************************/
/*                             Meshes                                 */
/**********************************************************************/


void gl_mapgrid1( GLint un, GLfloat u1, GLfloat u2 )
{
   if (CC.Mode!=0) {
      gl_error( GL_INVALID_OPERATION, "glMapGrid1f" );
      return;
   }
   if (un<1) {
      gl_error( GL_INVALID_VALUE, "glMapGrid1f" );
      return;
   }
   CC.Eval.MapGrid1un = un;
   CC.Eval.MapGrid1u1 = u1;
   CC.Eval.MapGrid1u2 = u2;
}


void glMapGrid1f( GLint un, GLfloat u1, GLfloat u2 )
{
   if (CC.CompileFlag) {
      gl_save_mapgrid1( un, u1, u2 );
   }
   if (CC.ExecuteFlag) {
      gl_mapgrid1( un, u1, u2 );
   }
}


void glMapGrid1d( GLint un, GLdouble u1, GLdouble u2 )
{
   if (CC.CompileFlag) {
      gl_save_mapgrid1( un, (GLfloat) u1, (GLfloat) u2 );
   }
   if (CC.ExecuteFlag) {
      gl_mapgrid1( un, (GLfloat) u1, (GLfloat) u2 );
   }
}



void gl_mapgrid2( GLint un, GLfloat u1, GLfloat u2,
		  GLint vn, GLfloat v1, GLfloat v2 )
{
   if (CC.Mode!=0) {
      gl_error( GL_INVALID_OPERATION, "glMapGrid2f" );
      return;
   }
   if (un<1) {
      gl_error( GL_INVALID_VALUE, "glMapGrid2f(un)" );
      return;
   }
   if (vn<1) {
      gl_error( GL_INVALID_VALUE, "glMapGrid2f(vn)" );
      return;
   }
   CC.Eval.MapGrid2un = un;
   CC.Eval.MapGrid2u1 = u1;
   CC.Eval.MapGrid2u2 = u2;
   CC.Eval.MapGrid2vn = vn;
   CC.Eval.MapGrid2v1 = v1;
   CC.Eval.MapGrid2v2 = v2;
}


void glMapGrid2f( GLint un, GLfloat u1, GLfloat u2,
		  GLint vn, GLfloat v1, GLfloat v2 )
{
   if (CC.CompileFlag) {
      gl_save_mapgrid2( un, u1, u2, vn, v1, v2 );
   }
   if (CC.ExecuteFlag) {
      gl_mapgrid2( un, u1, u2, vn, v1, v2 );
   }
}


void glMapGrid2d( GLint un, GLdouble u1, GLdouble u2,
		  GLint vn, GLdouble v1, GLdouble v2 )
{
   if (CC.CompileFlag) {
      gl_save_mapgrid2( un, (GLfloat) u1, (GLfloat) u2,
		        vn, (GLfloat) v1, (GLfloat) v2 );
   }
   if (CC.ExecuteFlag) {
      gl_mapgrid2( un, (GLfloat) u1, (GLfloat) u2,
		   vn, (GLfloat) v1, (GLfloat) v2 );
   }
}



void glEvalPoint1( GLint i )
{
   if (CC.CompileFlag) {
      gl_save_evalpoint1( i );
   }
   if (CC.ExecuteFlag) {
      GLfloat u, du;

      if (i==0) {
	 u = CC.Eval.MapGrid1u1;
      }
      else if (i==CC.Eval.MapGrid1un) {
	 u = CC.Eval.MapGrid1u2;
      }
      else {
	 du = (CC.Eval.MapGrid1u2 - CC.Eval.MapGrid1u1)
	      / (GLfloat) CC.Eval.MapGrid1un;
	 u = i * du + CC.Eval.MapGrid1u1;
      }
      gl_evalcoord1( u );
   }
}



void glEvalPoint2( GLint i, GLint j )
{
   if (CC.CompileFlag) {
      gl_save_evalpoint2( i, j );
   }
   if (CC.ExecuteFlag) {
      GLfloat u, du;
      GLfloat v, dv;

      if (i==0) {
	 u = CC.Eval.MapGrid2u1;
      }
      else if (i==CC.Eval.MapGrid2un) {
	 u = CC.Eval.MapGrid2u2;
      }
      else {
	 du = (CC.Eval.MapGrid2u2 - CC.Eval.MapGrid2u1)
	       / (GLfloat) CC.Eval.MapGrid2un;
	 u = i * du + CC.Eval.MapGrid2u1;
      }

      if (j==0) {
	 v = CC.Eval.MapGrid2v1;
      }
      else if (j==CC.Eval.MapGrid2vn) {
	 v = CC.Eval.MapGrid2v2;
      }
      else {
	 dv = (CC.Eval.MapGrid2v2 - CC.Eval.MapGrid2v1)
	       / (GLfloat) CC.Eval.MapGrid2vn;
	 v = j * dv + CC.Eval.MapGrid2v1;
      }

      gl_evalcoord2( u, v );
   }
}



void glEvalMesh1( GLenum mode, GLint i1, GLint i2 )
{
   GLint i;
   GLfloat u, du;
   GLenum prim;

   if (CC.CompileFlag) {
      gl_save_evalmesh1( mode, i1, i2 );
   }
   if (CC.ExecuteFlag) {

      if (CC.Mode!=0) {
	 gl_error( GL_INVALID_OPERATION, "glEvalMesh1" );
	 return;
      }

      switch (mode) {
	 case GL_POINT:
	    prim = GL_POINTS;
	    break;
	 case GL_LINE:
	    prim = GL_LINE_STRIP;
	    break;
	 default:
	    gl_error( GL_INVALID_ENUM, "glEvalMesh1(mode)" );
	    return;
      }

      du = (CC.Eval.MapGrid1u2 - CC.Eval.MapGrid1u1)
	       / (GLfloat) CC.Eval.MapGrid1un;

      gl_begin( prim );
      for (i=i1;i<=i2;i++) {
	 if (i==0) {
	    u = CC.Eval.MapGrid1u1;
	 }
	 else if (i==CC.Eval.MapGrid1un) {
	    u = CC.Eval.MapGrid1u2;
	 }
	 else {
	    u = i * du + CC.Eval.MapGrid1u1;
	 }
	 gl_evalcoord1( u );
      }
      gl_end();
   }
}



void glEvalMesh2( GLenum mode, GLint i1, GLint i2, GLint j1, GLint j2 )
{
   GLint i, j;
   GLfloat u, du, v, dv, v1, v2;

   if (CC.CompileFlag) {
      gl_save_evalmesh2( mode, i1, i2, j1, j2 );
   }
   if (CC.ExecuteFlag) {

      if (CC.Mode!=0) {
	 gl_error( GL_INVALID_OPERATION, "glEvalMesh2" );
	 return;
      }

      du = (CC.Eval.MapGrid2u2 - CC.Eval.MapGrid2u1)
	       / (GLfloat) CC.Eval.MapGrid2un;
      dv = (CC.Eval.MapGrid2v2 - CC.Eval.MapGrid2v1)
	       / (GLfloat) CC.Eval.MapGrid2vn;

#define I_TO_U( I, U )				\
	   if ((I)==0) {		       	\
	      U = CC.Eval.MapGrid2u1;		\
	   }					\
	   else if ((I)==CC.Eval.MapGrid2un) {	\
	      U = CC.Eval.MapGrid2u2;		\
	   }					\
	   else {				\
	      U = (I) * du + CC.Eval.MapGrid2u1;\
	   }

#define J_TO_V( J, V )				\
	   if ((J)==0) {			\
	      V = CC.Eval.MapGrid2v1;		\
	   }					\
	   else if ((J)==CC.Eval.MapGrid2vn) {	\
	      V = CC.Eval.MapGrid2v2;		\
	   }					\
	   else {				\
	      V = (J) * dv + CC.Eval.MapGrid2v1;\
	   }

      switch (mode) {
	 case GL_POINT:
	    gl_begin( GL_POINTS );
	    for (j=j1;j<=j2;j++) {
	       J_TO_V( j, v );
	       for (i=i1;i<=i2;i++) {
		  I_TO_U( i, u );
		  gl_evalcoord2( u, v );
	       }
	    }
	    gl_end();
	    break;
	 case GL_LINE:
	    for (j=j1;j<=j2;j++) {
	       J_TO_V( j, v );
	       gl_begin( GL_LINE_STRIP );
	       for (i=i1;i<=i2;i++) {
		  I_TO_U( i, u );
		  gl_evalcoord2( u, v );
	       }
	       gl_end();
	    }
	    for (i=i1;i<=i2;i++) {
	       I_TO_U( i, u );
	       gl_begin( GL_LINE_STRIP );
	       for (j=j1;j<=j2;j++) {
		  J_TO_V( j, v );
		  gl_evalcoord2( u, v );
	       }
	       gl_end();
	    }
	    break;
	 case GL_FILL:
	    for (j=j1;j<j2;j++) {
	       /* NOTE: a quad strip can't be used because the four */
	       /* can't be guaranteed to be coplanar! */
	       gl_begin( GL_TRIANGLE_STRIP );
	       J_TO_V( j, v1 );
	       J_TO_V( j+1, v2 );
	       for (i=i1;i<=i2;i++) {
		  I_TO_U( i, u );
		  gl_evalcoord2( u, v1 );
		  gl_evalcoord2( u, v2 );
	       }
	       gl_end();
	    }
	    break;
	 default:
	    gl_error( GL_INVALID_ENUM, "glEvalMesh2(mode)" );
	    return;
      }

#undef I_TO_U
#undef J_TO_V
   }
}

