 /*
  * Khoros: $Id: gw_main.c,v 1.2 1991/07/15 05:59:03 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: gw_main.c,v 1.2 1991/07/15 05:59:03 khoros Exp $";
#endif

 /*
  * $Log: gw_main.c,v $
 * Revision 1.2  1991/07/15  05:59:03  khoros
 * HellPatch1
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.
 * 
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *----------------------------------------------------------------------
 */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "writer.h"
#include "gw_str.h"

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>    Ghostwriter Routines to Generate *.c file          <<<<
   >>>>                                                       <<<<
   >>>>                gw_generate_cfile()                    <<<<
   >>>>                                                       <<<<
   >>>>                gw_print_main()                        <<<<
   >>>>                gw_print_usage()                       <<<<
   >>>>                gw_print_getargs()                     <<<<
   >>>>                                                       <<<<
   >>>>                gw_print_getarg_pstrings()   	      <<<<
   >>>>                gw_print_pstr()     	  	      <<<<
   >>>>                gw_print_get_required_args()    	      <<<<
   >>>>                gw_print_get_optional_args()    	      <<<<
   >>>>                gw_print_get_toggle_prompt()    	      <<<<
   >>>>                gw_print_get_toggle_commline()  	      <<<<
   >>>>                gw_print_get_mut_excl_prompt()         <<<<
   >>>>                gw_print_get_mut_excl_commline()       <<<<
   >>>>                gw_print_me_choices()	  	      <<<<
   >>>>                gw_print_debug_stmts()	  	      <<<<
   >>>>                gw_print_debug_stmt()	  	      <<<<
   >>>>                gw_print_toggle_debug_stmt()  	      <<<<
   >>>>                gw_print_variable_tag()  	      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */



/************************************************************
*
*  Routine Name: gw_generate_cfile
*
*      Purpose: This routine creates the main program file *.c, including
*		the main() program, gw_get_args(), and gw_usage()
*
*        Input: database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*
*    Called by: lwriter
*
*   Written By: Danielle Argiro
*
*************************************************************/

int  gw_generate_cfile(database, prog_spec, name, filename, type)
char **database;
char **prog_spec;
char *name;
char *filename;
char *type;
{
     FILE *file, *tmpfile;
     char  temp[MaxLength], resp;

     /*
      *  see if *.c file already exists.  If so, prompt to over-write,
      *  return true if the answer is NO
      */
     if (!writer->force)
     {
        if (tmpfile = fopen(filename, "r"))
        {
            fprintf(stderr, "\nOverwrite file '%s'?  [y/n]: ", filename);
            fgets(temp, MaxLength, stdin);
            sscanf(temp, "%c", &resp);
            if ((resp != 'y') && (resp != 'Y')) return(true);
        }
     }

     /*
      *  create & open *.c file
      */
     if ((file = fopen(filename, "w"))== NULL)
     {
          fprintf(stderr, "\nghost_main:\n");
          fprintf(stderr, "  Could not create file '%s'\n", filename);
          return(false);
     }

    /*
     * begin *.c file with RCS header
     */
     fprintf(file, " /*\n  * Khoros: %cId%c\n  */\n\n", '$','$');
     fprintf(file, "#if !defined(lint) && !defined(SABER)\n");
     fprintf(file, "static char rcsid[] = \"Khoros: %cId%c\";\n", '$','$');
     fprintf(file, "#endif\n\n");
     fprintf(file, " /*\n  * %cLog%c\n  */\n\n", '$','$');
      
     /*
      *  generate the main program
      */
     gw_print_main(file, database, prog_spec, name);
     fprintf(file, "\n\n");

     /*
      *  generate gw_usage() routine
      */
     gw_print_usage(file, database, prog_spec, name, type);
     fprintf(file, "\n\n");

     /*
      *  generate gw_get_args() routine
      */
     gw_print_getargs(file, database, prog_spec, name);
     fprintf(file, "\n\n");


     fclose(file);

     fprintf(stderr, "ghostwriter:  done generating '%s'\n", filename);
     return(true);

}


/************************************************************
*
*      Routine: gw_print_main
*
*      Purpose: This routine prints out the main program in file *.c
*
*        Input: file     - open stream to .c file
*		database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*
*    Called by: gw_generate_cfile
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_main(file, database, prog_spec, name)
FILE *file;
char **database;
char **prog_spec;
char *name;
{

     time_t  clock;
     char  temp[MaxLength];
     char  *textblock, **textlines;
     char *copyright;

     /*
      * begin .c file with UNM copyright
      */
     copyright = gw_copyright("long");
     if (copyright == NULL) {
         fprintf (stderr, "error in gw_generate_cfile ");
         fprintf (stderr, "- could not print copyright\n");
     }
     else {
         fprintf(file, "%s\n", copyright);
     }

    /*
     *  add header for .c file
     */
     fprintf(file, "%s", filehdr_top);
     fprintf(file, " >>>>\n%s%s.c\n >>>>\n", filehdr_file, name);
     fprintf(file, "%s%s\n >>>>\n",   filehdr_name, name);
     fprintf(file, "%s\n", filehdr_desc);
     if (xvf_strlen(prog_spec[ShortProgDesc]) > 0) 
     {
        textlines = get_text_lines(prog_spec[ShortProgDesc], writer->format);
        textblock = add_gw_format(textlines, " >>>>	"); 
	fprintf(file, "%s\n", textblock); 
     }
     else fprintf(file, " >>>>\n");

     if (xvf_strlen(prog_spec[Authors]) > 0) 
 	 fprintf(file, "%s%s >>>>\n", filehdr_auth, prog_spec[Authors]);
     else
         fprintf(file, "\n >>>> \n");

     clock = (time_t) time(NULL);
     sprintf(temp, "%s%s\n", filehdr_date, ctime(&clock));
     temp[xvf_strlen(temp)-2] = '\0';
     fprintf(file, "%s\n >>>>\n", temp);
     fprintf(file, "%smain- the main program for %s\n", filehdr_mods, name);
     fprintf(file, " >>>> \t\t gw_usage - gives usage of the program\n");
     fprintf(file, " >>>> \t\t gw_args  - gets arguments of program from command line\n");
     fprintf(file, " >>>>\n%s\n\n\n", filehdr_bottom);

    /*
     *  add in #includes
     */
     fprintf(file, "#include \"%s.h\"\n", name);
     fprintf(file, "\n\n");

    /*
     *   put in header for main program
     */
     fprintf(file, "%s", comm_beg);
     fprintf(file, " *\n * Routine Name:  main program for %s\n *\n", name);
     fprintf(file, " *       Input:  \n");
     gw_print_syntax(file, database, name, MainHeader);
     fprintf(file, " %s\n\n", comm_end);

    /*
     *   put in beginning of main program
     */
     fprintf(file, "main(argc, argv)\n");
     fprintf(file, "int argc;\n");
     fprintf(file, "char *argv[];\n{\n\n");

    /*
     *   add in main_variable_list from .prog file
     */
     fprintf(file,  "/* -main_variable_list */\n");
     if (xvf_strlen(prog_spec[MainVariables]) > 0)
        fprintf(file,  "%s", prog_spec[MainVariables]);
     fprintf(file,  "/* -main_variable_list_end */\n\n");

     fprintf(file, "\tprogram = VStrcpy(argv[0]);\n");
     fprintf(file, "\tac = argc;\n");
     fprintf(file, "\tav = argv;\n\n");
     fprintf(file, "\tkhoros_init();\n\n");

     fprintf(file, "\tgw_get_args();\n\n");

    /*
     *   if -debug specified, write out print stmts
     */
     if ((writer->debug) && (arg_total != 0))
	gw_print_debug_stmts(file, database, name);

    /*
     *   add in main_before_lib_call from .prog file
     */
     fprintf(file,  "/* -main_before_lib_call */\n");
     if (xvf_strlen(prog_spec[MainBeforeLib]) > 0)
        fprintf(file,  "%s", prog_spec[MainBeforeLib]);
     fprintf(file,  "/* -main_before_lib_call_end */\n\n");

    /*
     *   add in main_library_call from .prog file
     */
     fprintf(file,  "/* -main_library_call */\n");
     if (xvf_strlen(prog_spec[MainLibCall]) > 0)
        fprintf(file,  "%s", prog_spec[MainLibCall]);
     fprintf(file,  "/* -main_library_call_end */\n\n");

    /*
     *   add in main_after_lib_call from .prog file
     */
     fprintf(file,  "/* -main_after_lib_call */\n");
     if (xvf_strlen(prog_spec[MainAfterLib]) > 0)
        fprintf(file,  "%s", prog_spec[MainAfterLib]);
     fprintf(file,  "/* -main_after_lib_call_end */\n\n");


     fprintf(file, "\n\tkhoros_close();\n");
     fprintf(file, "}\n");   

}


/************************************************************
*
*      Routine: gw_print_usage
*
*      Purpose: This routine prints the gw_usage() routine in file *.c
*
*        Input: file     - open stream to .c file
*		database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*
*    Called by: gw_print_cfile
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_usage(file, database, prog_spec, name, type)
FILE *file;
char **database;
char **prog_spec;
char *name;
char *type;
{
	char **textlines;
	time_t   clock;
	char   temp[MaxLength];
	int    i;

    /*
     *   put in header for main program
     */
     fprintf(file, "%s", comm_beg);
     fprintf(file, "*\n%s  gw_usage \n*\n", comm_name);
     fprintf(file, "*\n%s  prints out the usage for %s \n*\n", 
			comm_purp, name);
     fprintf(file, "%s  none \n*\n", comm_in);
     fprintf(file, "%s  none \n*\n", comm_out);
     fprintf(file, "%s  automatically generated by ghostwriter\n*\n",comm_auth);
     fprintf(file, "%s\n\n", comm_end);

     clock = (time_t) time(NULL);
     sprintf(temp, "%s", ctime(&clock));
     temp[xvf_strlen(temp)-1] = '\0';

    /*
     *   put in beginning of gw_usage routine
     */
     fprintf(file, "gw_usage()\n{\n");
     fprintf(file, "\tfprintf(stderr, \"%s :\\n\");\n", name);
     if (xvf_strlen(prog_spec[ShortProgDesc]) > 0) 
     {
        textlines = get_text_lines(prog_spec[ShortProgDesc], writer->format);
	i = 0;
	while (textlines[i] != NULL)
	{
	    textlines[i][xvf_strlen(textlines[i])-1] = '\0';
            fprintf(file, "\tfprintf(stderr, \"\\t%s\\n\");\n", textlines[i]);
	    i++;
	}
     }

    /*
     *   add in usage_additions from .prog file
     */
     fprintf(file,  "\n/* -usage_additions */\n");
     if (xvf_strlen(prog_spec[UsageAdd]) > 0)
        fprintf(file,  "%s", prog_spec[UsageAdd]);
     fprintf(file,  "/* -usage_additions_end */\n\n");

    /*
     *   print syntax of program in format for the gw_usage() routine
     */
     gw_print_syntax(file, database, name, VGCUsage);


    /*
     *   print hardwired usage stmts applicable to all generated routines
     */
     if ((strcmp(type, "xprog") == 0) || (strcmp(type, "fxprog") == 0))
     {
        fprintf(file, 
        "\tfprintf(stderr, \"\\t[-jr filename.jp] creates a journal recording\\n\");\n");
        fprintf(file, 
        "\tfprintf(stderr, \"\\t[-jp filename.jp] executes a journal playback \\n\");\n");
        fprintf(file, 
        "\tfprintf(stderr, \"\\t[-fn] specifies the font for %s\\n\");\n", name);
     }

     fprintf(file, 
     "\tfprintf(stderr, \"\\t[-V] Gives the version for %s\\n\");\n", name);

     fprintf(file, 
     "\tfprintf(stderr, \"\\t[-U] Gives the usage for %s\\n\");\n", name);

     fprintf(file, 
     "\tfprintf(stderr, \"\\t[-P] Prompts for command line options\\n\");\n");

     fprintf(file, 
     "\tfprintf(stderr, \"\\t[-A [file1]] Creates the answer file called ");
     fprintf(file, "%s.ans or file1 \\n\");\n", name);

     fprintf(file, 
     "\tfprintf(stderr, \"\\t[-a [file1]] Uses %s.ans or file1", name);
     fprintf(file, " as the answer file \\n\\n\\n\");\n");

     fprintf(file, "\n}\n");


}




/************************************************************
*
*      Routine: gw_print_getargs
*
*      Purpose: This routine prints the gw_get_args() routine in file *.c
*
*        Input: file     - open stream to .c file
*		database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*
*    Called by: gw_generate_cfile
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_getargs(file, database, prog_spec, name)
FILE *file;
char **database;
char **prog_spec;
char *name;
{
     char *variables[50];

    /*
     *   put in header for getargs routine
     */
     fprintf(file, "%s", comm_beg);
     fprintf(file, "*\n%s  gw_getargs \n*\n", comm_name);
     fprintf(file, "*\n%s  gets arguments off command line for %s\n*\n", 
		   comm_purp, name);
     fprintf(file, "%s  none \n*\n", comm_in);
     fprintf(file, "%s  none \n*\n", comm_out);
     fprintf(file, "%s  automatically generated by ghostwriter\n*\n",comm_auth);
     fprintf(file, "%s\n\n", comm_end);

    /*
     *   put in beginning of gw_get_args routine
     */
     fprintf(file, "gw_get_args()\n{\n\n");

    /*
     *  print prompt strings for all arguments
     */
     gw_print_getarg_pstrings(file, database, prog_spec, name);

    /*
     *  print declarations for gw_get_args routine
     */
     fprintf(file, "char  answer_file[512], temp[512];\n\n");
     fprintf(file, "\n\nint\tfid = -1,\n\terror, query, ok, count;\n");
     fprintf(file, "\tint prompt_flag = FALSE;\n\n");

    /*
     *  print code to get [-V] flag for version
     */
     fprintf(file, "\t/*\n\t * print version if necessary\n\t */\n");
     fprintf(file, "\tvgparml(fid, \"-V\", \"noprompt\", &query, 0, 0, &error);\n");
     fprintf(file, "\tif (error == 0)\n");
     fprintf(file, "\t{\n");

     fprintf(file, "\t    printf (\"\\n%%s: %%s: Version %%d.%%d\\n\\n\",\n");
     fprintf(file, "\t    \"%s\", RELEASE_NAME, RELEASE_MAJOR, RELEASE_MINOR);\n", name);

     fprintf(file, "\t    exit(1);\n");
     fprintf(file, "\t}\n\n");

    /*
     *  print code to get [-U] flag for usage
     */
     fprintf(file, "\t/*\n\t * print usage if necessary\n\t */\n");
     fprintf(file, "\tvgparml(fid, \"-U\", \"noprompt\", &query, 0, 0, &error);\n");
     fprintf(file, "\tif (error == 0)\n");
     fprintf(file, "\t{\n");
     fprintf(file, "\t    gw_usage();\n");
     fprintf(file, "\t    exit(1);\n");
     fprintf(file, "\t}\n\n");

    /*
     *  print code to get [-P] flag for interactive prompting
     */
     fprintf(file, "\t/*\n\t * see if interactive prompting is desired\n\t */\n");
     fprintf(file, "\tvgparml(fid, \"-P\", \"noprompt\", &query, 0, 0, &error);\n");
     fprintf(file, "\tif (error == 0)\n");
     fprintf(file, "\t    prompt_flag = TRUE;\n\n");

     
    /*
     *  print code to get [-a] flag for use of answer file
     */
     fprintf(file, "\t/*\n\t * see if answer file is to be used\n\t */\n");
     fprintf(file, "\tvgparms(fid, \"-a\", \"noprompt\", answer_file, 0, 0, &error);\n");
     fprintf(file, "\tif (error == 0)\n");
     fprintf(file, "\t{\n");
     fprintf(file, "\t     if (VStrlen(answer_file) == 0)\n");
     fprintf(file, "\t\tsprintf(answer_file, \"%s.ans\");\n", name);
     fprintf(file, "\t     while (fid < 0)\n");
     fprintf(file, "\t     {\n");
     fprintf(file, "\t\tif ((fid = open(answer_file, O_RDONLY, 0666))== -1)\n");
     fprintf(file, "\t\t{\n");
     fprintf(file, "\t\t    fprintf(stderr, \"%s: can't open answer file \\\"%%s\\\"\",\n\t\t\t    answer_file); \n", name);
     fprintf(file, "\t\t    fprintf(stderr, \"please re-enter: \");\n", name);
     fprintf(file, "\t\t    (void) gets(answer_file);\n");
     fprintf(file, "\t\t    if (VStrlen(answer_file) == 0)\n");
     fprintf(file, "\t\t\tsprintf(answer_file, \"%s.ans\");\n", name);
     fprintf(file, "\t\t}\n\n");
     fprintf(file, "\t     }\n\n");
     fprintf(file, "\t}\n\n");

    /*
     *  print code to allocate room for argument structure
     */

     if (arg_total != 0)
     {
         fprintf(file, "\t/*\n\t * allocate the %s structure\n\t */\n", name);
         fprintf(file, "\t%s = (%s_struct *) \n\t       calloc((unsigned)1, (unsigned) sizeof (%s_struct));\n\n", name, name, name);
     }


    /*
     *  print code to get required arguments off command line
     */
     if (gw_num_req > 0)
         gw_print_get_required_args(file, database, prog_spec, name);


    /*
     *  print code to get required toggles off command line
     */
     if (gw_num_tog_req > 0)
     {
          fprintf(file, "\n\t/*\n\t * get required toggles for %s\n\t */\n", 
		        name);
          gw_print_get_toggle_prompt(file, database, prog_spec, name, true);
          gw_print_get_toggle_commline(file, database, prog_spec, name, true); 
     }

  
    /*
     *  print code to get required ME groups off command line
     */
     if (gw_num_me_req > 0)
     {
         fprintf(file, "\t/*\n\t * get required mutually exclusive groups for %s\n\t */\n", name);
         gw_print_get_mut_excl_prompt(file, database, prog_spec, 
				      name, true, variables);

         gw_print_get_mut_excl_commline(file, database, prog_spec, 
				        name, true, variables);
     }

    /*
     *  print code to get optional arguments off command line
     */
     if (gw_num_opt > 0)
         gw_print_get_optional_args(file, database, prog_spec, name);


    /*
     *  print code to get optional toggles off command line
     */
     if (gw_num_tog_opt > 0)
     {
         fprintf(file, "\n\t/*\n\t * get optional toggles for %s\n\t */\n", 
			name);
         gw_print_get_toggle_prompt(file, database, prog_spec, name, false);
         gw_print_get_toggle_commline(file, database, prog_spec, name, false);
     }


    /*
     *  print code to get optional ME groups off command line
     */
     if (gw_num_me_opt > 0)
     {
         fprintf(file, "\t/*\n\t * get optional mutually exclusive groups for %s\n\t */\n", name);
         gw_print_get_mut_excl_prompt(file, database, prog_spec, 
				      name, false, variables);

         gw_print_get_mut_excl_commline(file, database, prog_spec, 
				      name, false, variables);
     }
     fprintf(file, "\n}\n\n");
}


/************************************************************
*
*      Routine: gw_print_getarg_pstrings
*
*      Purpose: This routine prints the prompt strings for the
*		gw_get_args routine in file *.c
*
*        Input: file     - open stream to .c file
*               database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*
*    Called by: gw_print_getargs
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_getarg_pstrings(file, database, prog_spec, name)
FILE *file;
char **database;
char **prog_spec;
char *name;
{
    Line_Info line_info;
    int index, mut_excl_count, toggle_count;
    mut_excl_link *mut_excl_ptr;
    gw_toggle_struct *toggle_ptr;


    xvf_clear_line_info(&line_info);
    
    /*
     * print out prompt strings for required arguments
     */
    index = 0;
    while (index < gw_num_req)
    {
	xvf_gen_parse(database[gw_req_args[index]], &line_info);
	gw_print_pstr(file, &line_info);
	index++;
    }

    /*
     * print out prompt strings for any required toggles
     */
    toggle_count = 0;
    while(toggle_count < gw_toggle_num)
    {
	toggle_ptr = gw_toggle[toggle_count];
	if (toggle_ptr->req == true)
        {
	    fprintf(file, "char\t*%s_pstr = \"\\nEnter: (%s) %s -- \\n      ",
		    toggle_ptr->variable, toggle_ptr->variable,
                    toggle_ptr->desc);
	    gw_print_toggle_vals(file, toggle_ptr, VGCpstrs);
	    fprintf(file, "     : \"; \n");
	    if (toggle_ptr->type == StringOpt)
	    {
	        xvf_gen_parse(database[toggle_ptr->def-1], &line_info);
		if (line_info.string_def != NULL)
	            fprintf(file, "char\t*%s_def = \"%s\";\n", 
				  line_info.variable, line_info.string_def);
	    }
	}
	toggle_count++;
    }

    /*
     * print out prompt strings for required mutually exclusive groups
     */
    mut_excl_count = 0;
    while (mut_excl_count < gw_mut_excl_num)
    {
        mut_excl_ptr = gw_mut_excl[mut_excl_count];
        if (mut_excl_ptr->type == true)  /* mut excl group is required */
        {
            mut_excl_ptr = mut_excl_ptr->next;
            while (mut_excl_ptr != NULL)
            {
	        xvf_gen_parse(database[mut_excl_ptr->index], 
			      &line_info);
	        gw_print_pstr(file, &line_info);
                mut_excl_ptr = mut_excl_ptr->next;
            }
        }
        mut_excl_count++;
    }

    /*
     * print out prompt strings for optional arguments
     */
    index = 0;
    while (index < gw_num_opt)
    {
	xvf_gen_parse(database[gw_opt_args[index]], &line_info);
	gw_print_pstr(file, &line_info);
	index++;
    }

    /*
     * print out prompt strings for any optional toggles
     */
    toggle_count = 0;
    while(toggle_count < gw_toggle_num)
    {
        toggle_ptr = gw_toggle[toggle_count];
        if (toggle_ptr->req == false)
        {
	    fprintf(file, "char\t*%s_pstr = \"\\nEnter: (%s) %s -- \\n      ",
		    toggle_ptr->variable, toggle_ptr->variable,
                    toggle_ptr->desc);
	    gw_print_toggle_vals(file, toggle_ptr, VGCpstrs);
	    fprintf(file, "  [%s]:  \"; \n", ascii_def_toggle_value(toggle_ptr));
	    if (toggle_ptr->type == StringOpt)
	    {
	        xvf_gen_parse(database[toggle_ptr->def-1], &line_info);
		if (line_info.string_def != NULL)
	            fprintf(file, "char\t*%s_def = \"%s\";\n", 
			          line_info.variable, line_info.string_def);
	    }
        }
        toggle_count++;
    }

    /*
     * print out prompt strings for optional mutually exclusive groups
     */
    mut_excl_count = 0;
    while (mut_excl_count < gw_mut_excl_num)
    {
        mut_excl_ptr = gw_mut_excl[mut_excl_count];
        if (mut_excl_ptr->type == false)  /* mut excl group is optional */
        {
            mut_excl_ptr = mut_excl_ptr->next;
            while (mut_excl_ptr != NULL)
            {
	        xvf_gen_parse(database[mut_excl_ptr->index], 
			      &line_info);
	        gw_print_pstr(file, &line_info);
                mut_excl_ptr = mut_excl_ptr->next;
            }
        }
        mut_excl_count++;
    }
}

/************************************************************
*
*      Routine: gw_print_pstr
*
*      Purpose: This routine prints a single prompt string
*		to the gw_getargs() routine being generated.
*		if the line_info structure indicates a string
*		argument, the line defining the default string is 
*		also printed.
*
*        Input: file     - open stream to .c file
*		line_info - pointer to the line_info struct
*
*    Called by: gw_print_getarg_pstrings
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_pstr(file, line_info)
FILE *file;
Line_Info *line_info;
{
    /*
     *	prompt string for logical argument
     */
    if (line_info->typeflag == LogicOpt)
    {
	/* print the base of the prompt string */
        fprintf(file, 
	        "char\t*%s_pstr = \n\t\"\\nEnter: (%s) %s \\n        {%s, (y/n) ",
	        line_info->variable, line_info->variable,
	        line_info->description, ascii_name(line_info->typeflag)); 

	/* tack on appropriate default info if argument is optional */
	if ((line_info->optional == true) && 
	    (line_info->logical_def == true))
	     fprintf(file, " [yes] ");
	else if ((line_info->optional == true) && 
		 (line_info->logical_def == false))
	     fprintf(file, " [no] ");

	/* finish off definition of the prompt string */
	fprintf(file, "}: \"; \n");


    }

    /*
     *	prompt string for integer argument
     */
    else if (line_info->typeflag == IntegerOpt)
    {
	/* print the base of the prompt string */
	fprintf(file, 
	        "char\t*%s_pstr = \n\t\"\\nEnter: (%s) %s \\n        {%s",
		line_info->variable, line_info->variable,
		line_info->description, ascii_name(line_info->typeflag)); 

	/* print the bounds into the prompt string */
	gw_print_bounds_int(file, line_info, VGCUsage);

	/* tack on appropriate default info if argument is optional */
	if (line_info->optional == true)
  	    fprintf(file, " [%d] ", line_info->int_def);

	/* tack on appropriate default info if argument is optional */
  	fprintf(file, "}: \"; \n");
    }

    /*
     *	prompt string for float argument
     */
    else if (line_info->typeflag == FloatOpt)
    {
	/* print the base of the prompt string */
	fprintf(file, 
	        "char\t*%s_pstr = \n\t\"\\nEnter: (%s) %s \\n        {%s",
	        line_info->variable, line_info->variable,
	        line_info->description, ascii_name(line_info->typeflag)); 

	/* print the bounds into the prompt string */
	gw_print_bounds_float(file, line_info, VGCUsage);

	/* tack on appropriate default info if argument is optional */
	if (line_info->optional == true)
	   fprintf(file, " [%f] ", line_info->float_def);

	/* finish off definition of the prompt string */
  	fprintf(file, "}: \"; \n");
    }

    /*
     *	prompt string for string argument
     */
    else if (line_info->typeflag == StringOpt)
    {
	/* print the base of the prompt string */
        fprintf(file, 
 		"char\t*%s_pstr = \n\t\"\\nEnter: (%s) %s \\n        {%s",
		line_info->variable, line_info->variable,
		line_info->description, ascii_name(line_info->typeflag)); 

	/* tack on appropriate default info if argument is optional */
	if ((line_info->string_def != NULL) && (line_info->optional == true))
	     fprintf(file, " ['%s'] ", line_info->string_def);

	/* finish off definition of the prompt string */
	fprintf(file, "}: \"; \n");

        /* print definition of default string */
	if (line_info->string_def != NULL) 
             fprintf(file, "char\t*%s_def = \"%s\";\n",
                     line_info->variable, line_info->string_def);
	else
             fprintf(file, "char\t*%s_def = NULL;\n", line_info->variable);
    }

    /*
     *	prompt strings for inputfile & outputfile arguments
     */
    else if ((line_info->typeflag == InputFile) || 
	     (line_info->typeflag == OutputFile))
    {
	/* print the base of the prompt string */
        fprintf(file, 
 		"char\t*%s_pstr = \n\t\"\\nEnter: (%s) %s \\n        {%s",
		line_info->variable, line_info->variable,
		line_info->description, ascii_name(line_info->typeflag)); 

	/* tack on appropriate default info if argument is optional */
	if ((line_info->file_def != NULL) && (line_info->optional == true))
	     fprintf(file, " [%s] ", line_info->file_def);

	/* finish off definition of the prompt string */
	fprintf(file, "}: \"; \n");
    }
}


/************************************************************
*
*      Routine: gw_print_get_required_args
*
*      Purpose: This routine prints the code for getting required
*		arguments from the command line
*
*        Input: file     - open stream to .c file
*               database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*
*    Called by: gw_print_getargs
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_get_required_args(file, database, prog_spec, name)
FILE *file;
char **database;
char **prog_spec;
char *name;
{
    Line_Info line_info;
    int index;

    /*
     *  only print out code if there are some required arguments to 
     *  get off the command line - don't need to return since following
     *  while loop won't execute anyway 
     */
    if (gw_num_req > 0)
    {
       fprintf(file, "\t/*\n\t * get required arguments for %s\n\t */\n", name);
       fprintf(file, "\tif (prompt_flag == TRUE)\n");
       fprintf(file, "\t    fprintf(stderr,\"\\nRequired Arguments:\\n\\n\");\n\n");
    }

    index = 0;
    while (index < gw_num_req)
    {
	/* 
	 * look up info about required arguent, print out the code to get
	 * that argument off the command line, with protections & error stmts
	 */
	xvf_gen_parse(database[gw_req_args[index]], &line_info);
        fprintf(file, "\tif (!(");
	gw_print_vgparm_call(file, &line_info, name, "\t\t\t");
        fprintf(file, "))\n\t{\n");
	fprintf(file, "\t    if (prompt_flag == TRUE)\n");
	fprintf(file, "\t    {\n");
	fprintf(file, "\t\twhile(!(%s->%s_flag))\n", name, line_info.variable);
        fprintf(file, "\t\t{\n");
	fprintf(file, "\t\t    fprintf(stderr, \"\\t\\t%s is required, please re-enter: \\n\");\n", line_info.variable);
	fprintf(file, "\t\t    ");   
	gw_print_vgparm_call(file, &line_info, name, "\t\t    ");
	fprintf(file, ";\n\n"); 
	fprintf(file, "\t\t}\n"); 
	fprintf(file, "\t    }\n");
	fprintf(file, "\t    else\n");
	fprintf(file, "\t    {\n");
	fprintf(file, "\t\tfprintf(stderr, \"Error: '-%s' is a required argument\\n\");\n", line_info.variable);
	fprintf(file, "\t\tgw_usage();\n");
	fprintf(file, "\t\texit(1);\n");
	fprintf(file, "\t    }\n");
	fprintf(file, "\t}\n\n");
		
	index++;
    }

}


/************************************************************
*
*      Routine: gw_print_get_optional_args
*
*      Purpose: This routine prints the code for getting optional
*		arguments from the command line
*
*        Input: file     - open stream to .c file
*               database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*
*    Called by: gw_print_getargs
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_get_optional_args(file, database, prog_spec, name)
FILE *file;
char **database;
char **prog_spec;
char *name;
{
    Line_Info line_info;
    int index;

    if (gw_num_opt > 0)
    {
       fprintf(file, "\t/*\n\t * get optional arguments for %s\n\t */\n", name);
       fprintf(file, "\tif (prompt_flag == TRUE)\n");
       fprintf(file, "\t   fprintf(stderr,\"\\nOptional Arguments:\\n\\n\");\n\n");
    }
    index = 0;
    while (index < gw_num_opt)
    {
	xvf_gen_parse(database[gw_opt_args[index]], &line_info);
	fprintf(file, "\t");
	gw_print_vgparm_call(file, &line_info, name, "\t");
	fprintf(file, ";\n\n");
	if (line_info.typeflag == StringOpt)
	{
	    fprintf(file, "\tif (prompt_flag == FALSE)\n\t{\n");
	    if (writer->tag == true)
	    fprintf(file, 
		    "\t    if ((%s->%s_flag == 1)&&(%s->%s_string == NULL))\n",
			    name, line_info.variable, name, line_info.variable);
	    else
	    fprintf(file, 
		    "\t    if ((%s->%s_flag == 1)&&(%s->%s == NULL))\n",
			    name, line_info.variable, name, line_info.variable);
	    fprintf(file, "    \t{\n\t\tfprintf(stderr, \"%s:\\n\");\n", name);
	    fprintf(file, "\t\tfprintf(stderr, \"you must specify a name following '-%s'\\n\");\n", line_info.variable);
	    fprintf(file, "\t\tgw_usage();\n");
	    fprintf(file, "\t\texit(1);\n");
	    fprintf(file, "\t\t}\n\t}\n");

	}
	index++;
    }
}



/************************************************************
*
*      Routine: gw_print_get_toggle_prompt
*
*      Purpose: This routine prints the code for getting 
*		toggle values from the command line
*		when the -P prompt option is used
*
*        Input: file     - open stream to .c file
*               database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*		flag	 -  tells if toggle is required or not
*
*    Called by: gw_print_getargs
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_get_toggle_prompt(file, database, prog_spec, name, flag)
FILE *file;
char **database;
char **prog_spec;
char *name;
int   flag;
{
    int i, toggle_count = 0;
    gw_toggle_struct *toggle_ptr;
    char *variable;

    fprintf(file, "\tif (prompt_flag == TRUE)\n\t{\n");

    /* 
     * for each group of toggles in the pane...
     */
    while(toggle_count < gw_toggle_num)
    {
	toggle_ptr = gw_toggle[toggle_count];

	/* 
	 * if the required field matches the flag passed in... 
         */
	if (toggle_ptr->req == flag)
	{
	     /* print top of the loop */
	     variable = xvf_strcpy(toggle_ptr->variable);
	     fprintf(file, "\t    ok = FALSE;\n");
	     fprintf(file, "\t    while(!ok)\n");
	     fprintf(file, "\t    {\n");

	     switch(toggle_ptr->type) {
		
		/*
		 *   Integer Toggle
		 */
		case IntegerOpt:
		     /* 
		      * print the call to vget_int() 
		      */
	     	     fprintf(file, 
			     "\t\t%s->%s_flag = vget_int(fid, prompt_flag,\n",
                    	      name, variable);
	     	     fprintf(file, "\t\t\"-%s\", %s_pstr, ", 
			     variable, variable);

		     /* append _toggle, or not, as desired by user */
		     if (writer->tag == true)
		         fprintf(file, "&(%s->%s_toggle),\n", name, variable);
		     else
		         fprintf(file, "&(%s->%s),\n", name, variable);
	             fprintf(file, "\t\t%d, 0, 0);\n\n", 
				toggle_ptr->intvals[toggle_ptr->def-1]);

		     /*
		      *  print lines to errorcheck input for toggle
		      */
		     if (flag == true)
		         fprintf(file, "\t\tif ((");
		     else
		         fprintf(file, "\t\tif (");
		     i = 0;
		     while (i < toggle_ptr->elem_num)
		     {
			if (writer->tag == true)
			    fprintf(file, "(%s->%s_toggle == %d )", 
				name, variable, toggle_ptr->intvals[i]);
			else
			    fprintf(file, "(%s->%s == %d )", 
				name, variable, toggle_ptr->intvals[i]);
			if ((i < toggle_ptr->elem_num - 1) && (flag == true))
			    fprintf(file, " || \n\t\t     ");
			else if ((i < toggle_ptr->elem_num - 1) 
				 && (flag == false))
			    fprintf(file, " || \n\t\t    ");
			else 
			    fprintf(file, ")");
			i++;
		     }

		     break;

		/*
		 *   Float Toggle
		 */
		case FloatOpt:
		     /* 
		      * print the call to vget_float() 
		      */
	     	     fprintf(file, 
			     "\t\t%s->%s_flag = vget_float(fid, prompt_flag,\n",
                    	     name, variable);
	     	     fprintf(file, "\t\t\"-%s\", %s_pstr, ", 
			     variable, variable);

		     /* append _toggle, or not, as desired by user */
		     if (writer->tag == true)
		         fprintf(file, "&(%s->%s_toggle),\n", name, variable);
		     else
		         fprintf(file, "&(%s->%s),\n", name, variable);
	             fprintf(file, "\t\t%f, 0.0, 0.0);\n\n", 
				toggle_ptr->floatvals[toggle_ptr->def-1]);

		     /*
		      *  print lines to errorcheck input for toggle
		      *  special case to check input for floats - must put
		      *  ascii rep. of float into a temp string, and use
		      *  strcmp, instead of a direct comparison.  this is
		      *  because of difficulty in correctly representing 
		      *  floats in binary.
		      */
		     if (writer->tag == true)
		         fprintf(file, 
			     "\t\tsprintf(temp, \"%%.6f\", %s->%s_toggle);\n", 
			      name, variable);
		     else
		         fprintf(file, 
			     "\t\tsprintf(temp, \"%%.6f\", %s->%s);\n", 
			      name, variable);

		     /* append _toggle, or not, as desired by user */
		     if (writer->tag == true)
		         fprintf(file, "\t\t%s->%s_toggle = atof(temp);\n", 
				 name, variable);
		     else 
		         fprintf(file, "\t\t%s->%s = atof(temp);\n", 
				 name, variable);
			
		     if (flag == true)
		         fprintf(file, "\t\tif ((");
		     else
		         fprintf(file, "\t\tif (");
		     i = 0;
		     while (i < toggle_ptr->elem_num)
		     {
			fprintf(file, "(strcmp(temp, \"%.6f\") != 0 )", 
				toggle_ptr->floatvals[i]);
			if (i < toggle_ptr->elem_num - 1)
			    fprintf(file, " && \n\t\t     ");
			else 
			    fprintf(file, ")");
			i++;
		     }
		     break;

		/*
		 *   String Toggle
		 */
		case StringOpt:
		     /* 
		      * print the call to vget_string() 
		      */
	     	     fprintf(file, 
			     "\t\t%s->%s_flag = vget_string(fid, prompt_flag,",
                    	      name, variable);
	     	     fprintf(file, "\n\t\t\"-%s\", %s_pstr, ", 
			     variable, variable);
		     /* append _toggle, or not, as desired by user */
		     if (writer->tag == true)
		         fprintf(file, "&(%s->%s_toggle),\n", name, variable);
		     else
		         fprintf(file, "&(%s->%s),\n", name, variable);
	             fprintf(file, "\t\t\"%s\");\n\n", 
				toggle_ptr->stringvals[toggle_ptr->def-1]);

		     /*
		      *  print lines to errorcheck input for toggle
		      */
		     if (flag == true)
		         fprintf(file, "\t\tif ((");
		     else
		         fprintf(file, "\t\tif (");
		     i = 0;
		     while (i < toggle_ptr->elem_num)
		     {
			if (writer->tag == true)
			    fprintf(file, 
				    "((strcmp(%s->%s_toggle, \"%s\") == 0))",
				    name, variable, toggle_ptr->stringvals[i]);
			else
			    fprintf(file, 
				    "((strcmp(%s->%s, \"%s\") == 0))",
				    name, variable, toggle_ptr->stringvals[i]);
			if (i < toggle_ptr->elem_num - 1)
			    fprintf(file, " || \n\t\t     ");
			else 
			    fprintf(file, ")");
			i++;
		     }
		     break;

		/*
		 *   Logical Toggle
		 */
		case LogicOpt:
	     	     fprintf(file, 
			     "\t\t%s->%s_flag = vget_int(fid, prompt_flag,\n",
                    	     name, variable);
	     	     fprintf(file, "\t\t\"-%s\", %s_pstr, ", 
			     variable, variable);

		     /* append _toggle, or not, as desired by user */
		     if (writer->tag == true)
		         fprintf(file, "&(%s->%s_toggle),\n", name, variable);
		     else
		         fprintf(file, "&(%s->%s),\n", name, variable);
	             fprintf(file, "\t\t%d, 0, 0);\n\n", 
				toggle_ptr->logicvals[toggle_ptr->def-1]);
		     /*
		      *  print lines to errorcheck input for toggle
		      */
		     if (flag == true)
		         fprintf(file, "\t\tif ((");
		     else
		         fprintf(file, "\t\tif (");
		     i = 0;
		     while (i < toggle_ptr->elem_num)
		     {
			if (writer->tag == true)
			    fprintf(file, "(%s->%s_toggle == %d )", 
				name, variable, toggle_ptr->logicvals[i]);
			else
			    fprintf(file, "(%s->%s == %d )", 
				name, variable, toggle_ptr->logicvals[i]);

			if (i < toggle_ptr->elem_num - 1)
			    fprintf(file, " || \n\t\t     ");
			else 
			    fprintf(file, ")");
			i++;
		     }
		     break;

	     } /* end switch */


	    /*
	     *  at this point, the code that errorchecks value of toggle
	     *  input from command line is still unfinished.  Need to finish
	     *  it up.
	     */

	     if ((flag == true) && (toggle_ptr->type != FloatOpt))
		fprintf(file, " &&\n\t\t     (%s->%s_flag))\n", 
			name, variable);
	     else if (toggle_ptr->type == FloatOpt)
	     {
		fprintf(file, " ||\n\t\t     (!(%s->%s_flag)))\n", 
			name, variable);
	        fprintf(file, "\t\t{\n");
	        if (flag == true)
	 	   fprintf(file,  "\t\t    fprintf(stderr, \"\\n%s is a required argument.\\n\");\n", variable);
	           /*fprintf(file, "\t\t    fprintf(stderr, \"allowed values for %s are: \");\n", variable);
	           gw_print_toggle_vals(file, toggle_ptr, VGCGetArgs); */
	          /* fprintf(file,"\t\t    fprintf(stderr, \"\\n\");\n");
	           fprintf(file,"\t\t    fprintf(stderr, \"please re-enter: \");\n"); */
	           fprintf(file, "\t\t}\n");
		   fprintf(file, "\t\telse\n");
		   fprintf(file, "\t\t     ok = TRUE;\n");
	           fprintf(file, "\t    }\n");

	     }
	
	     else
		fprintf(file, "\n");
	     
	     if (toggle_ptr->type != FloatOpt)
	     {
	         fprintf(file, "\t\t      ok = TRUE;\n");
	         fprintf(file, "\t\telse\n");
	         fprintf(file, "\t\t{\n");
	         if (flag == true)
		    fprintf(file,  "\t\t    fprintf(stderr, \"\\n%s is a required argument.\\n\");\n", variable);
	/*
	         fprintf(file, "\t\t    fprintf(stderr, \"allowed values for %s are: \");\n", variable);
	         gw_print_toggle_vals(file, toggle_ptr, VGCGetArgs); 
	         fprintf(file,"\t\t    fprintf(stderr, \"\\n\");\n");
	         fprintf(file,"\t\t    fprintf(stderr, \"please re-enter: :\");\n"); */
	         fprintf(file, "\t\t}\n");
	         fprintf(file, "\t    }\n");
	     }
	}
	toggle_count++;
    }
    fprintf(file, "\t}\n");

}

/************************************************************
*
*      Routine: gw_print_get_toggle_commline
*
*      Purpose: This routine prints the code for getting 
*		toggle values from the command line when the
*		-P prompt option is not used
*
*        Input: file     - open stream to .c file
*               database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*		flag	 -  tells if toggle is required or not
*
*    Called by: gw_print_getargs
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_get_toggle_commline(file, database, prog_spec, name, flag)
FILE *file;
char **database;
char **prog_spec;
char *name;
int   flag;
{
    int i, toggle_count = 0;
    gw_toggle_struct *toggle_ptr;
    char *variable;

    fprintf(file, "\telse\n\t{\n");

    /* 
     * for each group of toggles in the pane...
     */
    while(toggle_count < gw_toggle_num)
    {
	toggle_ptr = gw_toggle[toggle_count];

	/* 
	 * if the required field matches the flag passed in... 
         */
	if (toggle_ptr->req == flag)
	{
	     variable = xvf_strcpy(toggle_ptr->variable);
	     switch(toggle_ptr->type) {
		
		/*
		 *   Integer Toggle
		 */
		case IntegerOpt:
		     /* 
		      * print the call to vget_int() 
		      */
	     	     fprintf(file, 
			     "\t\t%s->%s_flag = vget_int(fid, prompt_flag,\n",
                    	      name, variable);
	     	     fprintf(file, "\t\t\"-%s\", %s_pstr, ",
			     variable, variable);
		     /* append _toggle, or not, as desired by user */
		     if (writer->tag)
		         fprintf(file, "&(%s->%s_toggle),\n", name, variable);
		     else 
		         fprintf(file, "&(%s->%s),\n", name, variable);
	             fprintf(file, "\t\t%d, 0, 0);\n\n", 
				toggle_ptr->intvals[toggle_ptr->def-1]);
		     /*
		      *  print lines to errorcheck input for toggle
		      */
		     if (flag == true)
		         fprintf(file, "\t\tif ((");
		     else
		         fprintf(file, "\t\tif (");
		     i = 0;
		     while (i < toggle_ptr->elem_num)
		     {
		        /* append _toggle, or not, as desired by user */
			if (writer->tag == true)
			    fprintf(file, "(%s->%s_toggle != %d )", 
				    name, variable, toggle_ptr->intvals[i]);
			else 
			    fprintf(file, "(%s->%s != %d )", 
				    name, variable, toggle_ptr->intvals[i]);
			if ((i < toggle_ptr->elem_num - 1) && (flag == true))
			    fprintf(file, " && \n\t\t     ");
			else if ((i < toggle_ptr->elem_num - 1) 
				 && (flag == false))
			    fprintf(file, " && \n\t\t    ");
			else 
			    fprintf(file, ")");
			i++;
		     }
		     break;

		/*
		 *   Float Toggle
		 */
		case FloatOpt:
		     /* 
		      * print the call to vget_float() 
		      */
	     	     fprintf(file, 
			     "\t\t%s->%s_flag = vget_float(fid, prompt_flag,\n",
                    	      name, variable);
	     	     fprintf(file, "\t\t\"-%s\", %s_pstr,", variable, variable);
		     /* append _toggle, or not, as desired by user */
		     if (writer->tag == true)
		         fprintf(file, " &(%s->%s_toggle),\n", name, variable);
		     else 
		         fprintf(file, " &(%s->%s),\n", name, variable);
	             fprintf(file, "\t\t%f, 0.0, 0.0);\n\n", 
				toggle_ptr->floatvals[toggle_ptr->def-1]);
		     /*
		      *  print lines to errorcheck input for toggle
		      */
		     if (flag == true)
		         fprintf(file, "\t\tif ((");
		     else
		         fprintf(file, "\t\tif (");
		     i = 0;
		     while (i < toggle_ptr->elem_num)
		     {
		        /* append _toggle, or not, as desired by user */
			if (writer->tag == true)
			    fprintf(file, "(%s->%s_toggle != %f )", 
			 	    name, variable, toggle_ptr->floatvals[i]);
			else
			    fprintf(file, "(%s->%s != %f )", 
			 	    name, variable, toggle_ptr->floatvals[i]);
			if (i < toggle_ptr->elem_num - 1)
			    fprintf(file, " && \n\t\t     ");
			else 
			    fprintf(file, ")");
			i++;
		     }
		     break;

		/*
		 *   String Toggle
		 */
		case StringOpt:
	     	     fprintf(file, 
			   "\t\t%s->%s_flag = vget_string(fid, prompt_flag,\n",
                    	    name, variable);
	     	     fprintf(file, "\t\t\"-%s\", %s_pstr, ",variable, variable);
		     /* append _toggle, or not, as desired by user */
		     if (writer->tag == true)
		         fprintf(file, "&(%s->%s_toggle),\n", name, variable);
		     else
		         fprintf(file, "&(%s->%s),\n", name, variable);
	             fprintf(file, "\t\t\"%s\");\n\n", 
				toggle_ptr->stringvals[toggle_ptr->def-1]);
		     /*
		      *  print lines to errorcheck input for toggle
		      */
		     if (flag == true)
		         fprintf(file, "\t\tif ((");
		     else
		         fprintf(file, "\t\tif (");
		     i = 0;
		     while (i < toggle_ptr->elem_num)
		     {
			if (writer->tag == true)
			   fprintf(file, "((strcmp(%s->%s_toggle, \"%s\") != 0))",
				   name, variable, toggle_ptr->stringvals[i]);
			else
			   fprintf(file, "((strcmp(%s->%s, \"%s\") != 0))",
				   name, variable, toggle_ptr->stringvals[i]);
			if (i < toggle_ptr->elem_num - 1)
			    fprintf(file, " && \n\t\t     ");
			else 
			    fprintf(file, ")");
			i++;
		     }
		     break;

		/*
		 *   Logical Toggle
		 */
		case LogicOpt:
	     	     fprintf(file, 
			     "\t\t%s->%s_flag = vget_int(fid, prompt_flag,\n",
                    	      name, variable);
	     	     fprintf(file, "\t\t\"-%s\", %s_pstr, ",variable, variable);
		     /* append _toggle, or not, as desired by user */
		     if (writer->tag)
		         fprintf(file, "&(%s->%s_toggle),\n", name, variable);
		     else
		         fprintf(file, "&(%s->%s),\n", name, variable);
	             fprintf(file, "\t\t%d, 0, 0);\n\n", 
				toggle_ptr->logicvals[toggle_ptr->def-1]);
		     /*
		      *  print lines to errorcheck input for toggle
		      */
		     if (flag == true)
		         fprintf(file, "\t\tif ((");
		     else
		         fprintf(file, "\t\tif (");
		     i = 0;
		     while (i < toggle_ptr->elem_num)
		     {
			if (writer->tag == true)
			    fprintf(file, "(%s->%s_toggle != %d )", 
				name, variable, toggle_ptr->logicvals[i]);
			else
			    fprintf(file, "(%s->%s!= %d )", 
				name, variable, toggle_ptr->logicvals[i]);
			if (i < toggle_ptr->elem_num - 1)
			    fprintf(file, " && \n\t\t     ");
			else 
			    fprintf(file, ")");
			i++;
		     }
		     break;

	     }  /* end switch */


	    /*
	     *  at this point, the code that errorchecks value of toggle
	     *  input from command line is still unfinished.  Need to finish
	     *  it up.
	     */

	     if (flag == true)
		fprintf(file, " ||\n\t\t     (!(%s->%s_flag)))\n", 
			name, variable);
	     else
		fprintf(file, "\n");

	     fprintf(file, "\t\t{\n");
	     if (flag == true)
		fprintf(file,  "\t\t    fprintf(stderr, \"%s is a required argument.\\n\");\n", variable);
	/*
	     fprintf(file, "\t\t    fprintf(stderr, \"allowed values for %s are: \");\n", variable);
	     gw_print_toggle_vals(file, toggle_ptr, VGCGetArgs);
	*/
	     fprintf(file,"\t\t    fprintf(stderr, \"\\n\");\n");
	     fprintf(file,"\t\t    gw_usage();\n");
	     fprintf(file,"\t\t    exit(0);\n");
	     fprintf(file, "\t\t}\n");
	     
	}
	toggle_count++;
    }

    fprintf(file, "\t}\n\n");

} /* end gw_print_get_toggle_commline */

/************************************************************
*
*      Routine: gw_print_vgparm_call
*
*      Purpose: This routine prints the correct call to vgparms
*		based on line_info information
*
*        Input: file      - open stream to .c file
*		line_info - pointer to line_info structure
*               name      -  name of the program being generated
*               whitespace-  white space to 2nd line of call
*
*    Called by: gw_print_get_mut_excl_prompt
*		gw_print_get_mut_excl_commline
*		gw_print_get_required_args
*		gw_print_get_optional_args
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_vgparm_call(file, line_info, name, whitespace)
FILE 	  *file;
Line_Info *line_info;
char 	  *name, *whitespace;
{
    
    switch(line_info->typeflag)
    {
	case InputFile:
	     fprintf(file, "%s->%s_flag = vget_infile(fid, prompt_flag,\n",
		     name, line_info->variable);
	     if (writer->tag)
	             fprintf(file, "%s\"-%s\", %s_pstr, &(%s->%s_file))\n",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable);
	     else
	             fprintf(file, "%s\"-%s\", %s_pstr, &(%s->%s))\n",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable);
	     break;

	case OutputFile:
	     fprintf(file, "%s->%s_flag = vget_outfile(fid, prompt_flag,\n",
		     name, line_info->variable);
	     if (writer->tag)
	             fprintf(file, "%s\"-%s\", %s_pstr, &(%s->%s_file))\n",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable);
	     else
	             fprintf(file, "%s\"-%s\", %s_pstr, &(%s->%s))\n",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable);
	     break;

        case IntegerOpt:
	     fprintf(file, "%s->%s_flag = vget_int(fid, prompt_flag,\n",
                     name, line_info->variable);
	     if (writer->tag)
	             fprintf(file, "%s\"-%s\", %s_pstr, &(%s->%s_int),\n",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable);
	     else
	             fprintf(file, "%s\"-%s\", %s_pstr, &(%s->%s),\n",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable);
	     fprintf(file, "%s%d, %d, %d)", whitespace, line_info->int_def,
		     line_info->lower_int, line_info->upper_int);
	     break;

        case FloatOpt:
	     fprintf(file, "%s->%s_flag = vget_float(fid, prompt_flag,\n",
                     name, line_info->variable);
	     if (writer->tag)
	             fprintf(file, "%s\"-%s\", %s_pstr, &(%s->%s_float),\n",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable);
	     else
	             fprintf(file, "%s\"-%s\", %s_pstr, &(%s->%s),\n",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable);
	     fprintf(file, "%s%f, %f, %f)", whitespace, line_info->float_def,
		     line_info->lower_float, line_info->upper_float);
	     break;

        case StringOpt:
	     fprintf(file, "%s->%s_flag = vget_string(fid, prompt_flag,\n",
                     name, line_info->variable);
	     if (writer->tag)
	             fprintf(file, 
		     "%s\"-%s\", %s_pstr, &(%s->%s_string),\n%s%s_def)",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable, whitespace,line_info->variable);
	     else
	             fprintf(file, 
		     "%s\"-%s\", %s_pstr, &(%s->%s),\n%s%s_def)",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable, whitespace,line_info->variable);
	     break;

        case LogicOpt:
	     fprintf(file, "%s->%s_flag = vget_logic(fid, prompt_flag,\n",
                     name, line_info->variable);
	     if (writer->tag)
	             fprintf(file, "%s\"-%s\", %s_pstr, &(%s->%s_logic), %d)",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable, line_info->logical_def);
	     else
	             fprintf(file, "%s\"-%s\", %s_pstr, &(%s->%s), %d)",
		     whitespace, line_info->variable, line_info->variable,
		     name, line_info->variable, line_info->logical_def);
	     break;

	 

    }  /* end switch */

} /* end gw_print_vgparm_call */


/************************************************************
*
*      Routine: gw_print_get_mut_excl_prompt
*
*      Purpose: This routine prints the code for getting required
*		mutually exclusive groups' values from the command 
*		line when the prompt (-P) flag is used
*
*        Input: file     - open stream to .c file
*               database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*		flag     -  tells if mut. exclusive group required or not
*		variables - array of names of choices
*
*    Called by: gw_print_getargs
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_get_mut_excl_prompt(file, database, prog_spec, name, flag, variables)
FILE *file;
char **database;
char **prog_spec;
char *name;
int   flag;
char *variables[20];
{
    mut_excl_link *mut_excl_ptr;
    int 	  i, count, mut_excl_count;
    Line_Info     line_info;

    mut_excl_count = 0;


    fprintf(file, "\tif (prompt_flag == TRUE)\n\t{\n");
    fprintf(file, 
       "\t\t   fprintf(stderr, \"\\nThe following prompts are mutually exclusive.\\n\");\n");
    fprintf(file, 
       "\t\t   fprintf(stderr, \"\\nHit <cr> until the desired prompt appears.\\n\");\n");

    while (mut_excl_count < gw_mut_excl_num)
    {
        mut_excl_ptr = gw_mut_excl[mut_excl_count];
        if (mut_excl_ptr->type == flag)  /* mut excl group required/optional */
        {
            if (flag == true)
            {
                fprintf(file, "\t   ok = FALSE;\n");
                fprintf(file, "\t   while(!ok)\n");
                fprintf(file, "\t   {\n");
            }

	    fprintf(file, "\t\t");
            mut_excl_ptr = mut_excl_ptr->next;
	    count = 0;
            while (mut_excl_ptr->next != NULL)
            {
	        xvf_gen_parse(database[mut_excl_ptr->index], 
			      &line_info);
	        variables[count++] = xvf_strcpy(line_info.variable);
                gw_print_vgparm_call(file, &line_info, name, "\t\t    ");
		fprintf(file, ";\n\n    ");

		fprintf(file, "\t\tif(!(");
		i = 0;
		while (i < count)
		{
		  fprintf(file, " (%s->%s_flag)", name, variables[i]); i++;
		  if (( count > 1) && (i <= count-1)) 
		      fprintf(file, "|| \n\t\t    ");
		}
		fprintf(file, "))\n\t\t    "); 
		
                mut_excl_ptr = mut_excl_ptr->next;
            }
	    xvf_gen_parse(database[mut_excl_ptr->index], 
			     &line_info);
	    variables[count++] = xvf_strcpy(line_info.variable);
            gw_print_vgparm_call(file, &line_info, name, "\t\t    ");
	    fprintf(file, ";\n\n");
	
	    if (flag == true)
	    {
	       fprintf(file, "\t\tif(");
	       i = 0;
	       while (i < count)
	       {
	            fprintf(file, "(%s->%s_flag)", name, variables[i]); i++;
	  	    if (( count > 1) && (i <= count-1)) 
	       	    fprintf(file, "|| \n\t\t    ");
	       }
	       fprintf(file, ")\n"); 
	       fprintf(file, "\t\t    ok = TRUE;\n");
	       fprintf(file, "\t\telse\n\t\t{\n");
	       fprintf(file, "\t\t   fprintf(stderr, \"Error: You MUST specify one of :\\n\");\n");

	       gw_fprintf_me_choices(file, variables, count);

	       fprintf(file, "\t\t}\n");
	   }
	    
        if (flag == true) fprintf(file, "\t   }\n");
        }
        mut_excl_count++;
    }
    fprintf(file, 
       "\t\t   fprintf(stderr, \"\\n ------\\n\");\n");
    fprintf(file, "\t}\n");

} /* end gw_print_get_mut_excl_prompt */

/************************************************************
*
*      Routine: gw_print_get_mut_excl_commline
*
*      Purpose: This routine prints the code for getting required
*		mutually exclusive groups' values from the command 
*		line when the prompt (-P) flag is NOT used
*
*        Input: file     - open stream to .c file
*               database -  pointer to the internal database struct
*               prog_spec-  pointer to the internal specification struct
*               name     -  name of the program being generated
*		flag     -  tells if mut. exclusive group required or not
*		variables - array of names of choices
*
*    Called by: gw_print_getargs
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_get_mut_excl_commline(file, database, prog_spec, name, flag, variables)
FILE *file;
char **database;
char **prog_spec;
char *name;
int   flag;
char *variables[20];
{
    mut_excl_link *mut_excl_ptr;
    int 	  count, mut_excl_count;
    Line_Info     line_info;

    mut_excl_count = 0;

    fprintf(file, "\telse\n\t{\n");
    count = 0;
    while (mut_excl_count < gw_mut_excl_num)
    {
        mut_excl_ptr = gw_mut_excl[mut_excl_count];

	if (mut_excl_ptr->type == flag)
	{
            mut_excl_ptr = mut_excl_ptr->next;
	    fprintf(file, "\t\tcount = 0;\n");
            while (mut_excl_ptr != NULL)
            {
    	        fprintf(file, "\t\tif(");
                xvf_gen_parse(database[mut_excl_ptr->index], 
	    	    	      &line_info);
		variables[count++] = xvf_strcpy(line_info.variable);
                gw_print_vgparm_call(file, &line_info, name, "\t\t    ");
	        fprintf(file, ")\n\t\t    count++;\n\n");
                mut_excl_ptr = mut_excl_ptr->next;
            }
	    if (flag == true)
	    {
                fprintf(file, "\t\tif (count == 0)\n\t\t{\n");
                fprintf(file, "\t\t    fprintf(stderr, \"Error: You MUST specify one of :\\n\");\n");

		gw_fprintf_me_choices(file, variables, count);

		fprintf(file, "\t\t    gw_usage();\n");
		fprintf(file, "\t\t    exit(1);\n");
                fprintf(file, "\t\t}\n\n");   

	        fprintf(file, "\t\telse if (count > 1)\n\t\t{\n");
                fprintf(file, "\t\t    fprintf(stderr, \"Error: You may specify ONLY ONE of :\\n\");\n");

		gw_fprintf_me_choices(file, variables, count);

	        fprintf(file, "\t\t    gw_usage();\n");
	        fprintf(file, "\t\t    exit(1);\n");
                fprintf(file, "\t\t}\n\n");   
	    }
	    else 
	    {
	        fprintf(file, "\t\tif (count > 1)\n\t\t{\n");
                fprintf(file, "\t\t    fprintf(stderr, \"Error: You may specify ONLY ONE of :\\n\");\n");

		gw_fprintf_me_choices(file, variables, count);

	        fprintf(file, "\t\t    gw_usage();\n");
	        fprintf(file, "\t\t    exit(1);\n");
                fprintf(file, "\t\t}\n\n");   
	    }
	}
        mut_excl_count++;
    }
    fprintf(file, "\t}\n\n");

} /* end gw_print_get_mut_excl_commline */

/************************************************************
*
*      Routine: gw_print_me_choices
*
*      Purpose: This routine prints the choices for 
*		mutually exclusive groups' values to the command line 
*
*        Input: file      - open stream to .c file
*		variables - array of names of choices
*		count     - the number of choices in the me group
*
*    Called by: gw_print_get_mut_excl_commline
*               gw_print_get_mut_excl_prompt
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_fprintf_me_choices(file, variables, count)
FILE *file;
char *variables[];
int   count;
{
    int i;

    i = 0;
    fprintf(file, "\t\t    fprintf(stderr, \"");
    while (i < count)
    {
        fprintf(file, " -%s ", variables[i]); 
 	i++;
    }
    fprintf(file, "\\n\");\n");

}  /* end gw_fprintf_me_choices */

/************************************************************
*
*      Routine: gw_print_debug_stmts
*
*      Purpose: For debugging only, this routine prints fprintf() statements
*		into the main program, to print out values of the generated
*		structure in the .h file, as extracted from the command line
*		by the generated gw_get_args() routine.
*
*        Input: file      - open stream to .c file
*		variables - array of names of choices
*		count     - the number of choices in the me group
*
*    Called by: gw_print_main
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_debug_stmts(file, database, name)
FILE  *file;
char **database;
char  *name;
{
	Line_Info line_info;
    	mut_excl_link *mut_excl_ptr;
    	gw_toggle_struct *toggle_ptr;
    	int index, mut_excl_count, toggle_count;

    	/*
      	 * print fprintf() stmts for required arguments
     	 */

	if (gw_num_req > 0)
	{
            fprintf(file,"\t/*\n\t * print values of required args for %s\n\t */\n",
		     name);
	    fprintf(file, "\tfprintf(stderr,\"\\nRequired Arguments\\n\");\n");

            index = 0;
	    while (index < gw_num_req)
	    {
		xvf_gen_parse(database[gw_req_args[index]], &line_info);
		gw_print_debug_stmt(file, line_info.typeflag, 
				    line_info.variable, name);
		index++;
	    }
	}

    	/*
      	 * print fprintf() stmts for required toggles
     	 */
	
	if (gw_num_tog_req > 0)
	{
            fprintf(file,"\n\t/*\n\t * print values of required toggles for %s\n\t */\n", name);
	    fprintf(file, "\tfprintf(stderr,\"\\nRequired Toggles\\n\");\n");
	    toggle_count = 0;
	    while(toggle_count < gw_toggle_num)
    	    {
                 toggle_ptr = gw_toggle[toggle_count];
	         if (toggle_ptr->req == true)
	             gw_print_toggle_debug_stmt(file, toggle_ptr->type, 
				           	toggle_ptr->variable, name);
	         toggle_count++;
	    }
	}

    	/*
      	 * print fprintf() stmts for required me groups
     	 */

	if (gw_num_me_req > 0)
	{
            fprintf(file,"\n\t/*\n\t * print values of required me groups for %s\n\t */\n", name);
	    fprintf(file, "\tfprintf(stderr,\"\\nRequired ME Groups\\n\");\n");
	    mut_excl_count = 0;
	    while (mut_excl_count < gw_mut_excl_num)
            {
	        mut_excl_ptr = gw_mut_excl[mut_excl_count];
                if (mut_excl_ptr->type == true)  /* mut excl group is required */
                {
	    	    mut_excl_ptr = mut_excl_ptr->next;
        	    while (mut_excl_ptr != NULL)
                    {
		        index = mut_excl_ptr->index;
                        xvf_gen_parse(database[index], &line_info);
		        gw_print_debug_stmt(file, line_info.typeflag, 
				         line_info.variable, name);
		        mut_excl_ptr = mut_excl_ptr->next;
		   }
	       }
	       mut_excl_count++;
	       fprintf(file, "\n");
	   }
	}

    	/*
      	 * print fprintf() stmts for optional arguments
     	 */

	if (gw_num_opt > 0)
	{
            fprintf(file,"\t/*\n\t * print values of optional args for %s\n\t */\n",
		     name);
	    fprintf(file, "\tfprintf(stderr,\"\\nOptional Arguments\\n\");\n");

            index = 0;
	    while (index < gw_num_opt)
	    {
		xvf_gen_parse(database[gw_opt_args[index]], &line_info);
		gw_print_debug_stmt(file, line_info.typeflag, 
				    line_info.variable, name);
		index++;
	    }
	}

    	/*
      	 * print fprintf() stmts for optional toggles
     	 */
	
	if (gw_num_tog_opt > 0)
	{
            fprintf(file,"\n\t/*\n\t * print values of optional toggles for %s\n\t */\n", name);
	    fprintf(file, "\tfprintf(stderr,\"\\nOptional Toggles\\n\");\n");
	    toggle_count = 0;
	    while(toggle_count < gw_toggle_num)
    	    {
                 toggle_ptr = gw_toggle[toggle_count];
	         if (toggle_ptr->req == false)
                 {
		    gw_print_toggle_debug_stmt(file, toggle_ptr->type, 
				           toggle_ptr->variable, name);
	         }
	         toggle_count++;
	    }
	}

	
    	/*
      	 * print fprintf() stmts for optional me groups
     	 */

	if (gw_num_me_opt > 0)
	{
            fprintf(file,"\n\t/*\n\t * print values of optional me groups for %s\n\t */\n", name);
	    fprintf(file, "\tfprintf(stderr,\"\\nOptional ME Groups\\n\");\n");
	    mut_excl_count = 0;
	    while (mut_excl_count < gw_mut_excl_num)
            {
	        mut_excl_ptr = gw_mut_excl[mut_excl_count];
                if (mut_excl_ptr->type == false)  /* mut excl group is optional */
                {
	    	    mut_excl_ptr = mut_excl_ptr->next;
        	    while (mut_excl_ptr != NULL)
                    {
		         index = mut_excl_ptr->index;
                         xvf_gen_parse(database[index], &line_info);
		         gw_print_debug_stmt(file, line_info.typeflag, 
			    	         line_info.variable, name);
		         mut_excl_ptr = mut_excl_ptr->next;
		    }
	        }
	        mut_excl_count++;
	        fprintf(file, "\n");
	    }
	}

	fprintf(file, "\n");
    	
}  /* end gw_print_debug_stmts */

/************************************************************
*
*      Routine: gw_print_debug_stmt
*
*      Purpose: prints a single debug statement
*
*        Input: flag      - typeflag of database line
*
*    Called by: gw_print_debug_stmts
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_debug_stmt(file, flag, variable, name)
FILE *file;
int  flag;
char *variable;
char *name;
{
        if ((flag == InputFile)||(flag == OutputFile)||(flag == StringOpt))
	     fprintf(file, "\tfprintf(stderr, \"%s = %%s\\n\", %s->%s", 
			   variable, name, variable);
	else if ((flag == LogicOpt) || (flag == IntegerOpt))
	     fprintf(file, "\tfprintf(stderr, \"%s = %%d\\n\", %s->%s", 
			   variable, name, variable);
	else if (flag == FloatOpt)
	     fprintf(file, "\tfprintf(stderr, \"%s = %%f\\n\", %s->%s", 
	                   variable, name, variable);
	if (writer->tag) gw_print_variable_tag(file, flag);
	fprintf(file, ");\n");

}  /* end gw_print_debug_stmt */


/************************************************************
*
*      Routine: gw_print_toggle_debug_stmt
*
*      Purpose: prints a single debug statement (for toggles only)
*
*        Input: flag      - typeflag of toggle group
*
*    Called by: gw_print_debug_stmts
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_toggle_debug_stmt(file, flag, variable, name)
FILE *file;
int  flag;
char *variable;
char *name;
{
        if ((flag == InputFile)||(flag == OutputFile)||(flag == StringOpt))
	{
	   if (writer->tag == true)
	       fprintf(file,
		"\tfprintf(stderr, \"%s = %%s\\n\", %s->%s_toggle);\n",
		variable, name, variable);
	   else
	       fprintf(file,
		"\tfprintf(stderr, \"%s = %%s\\n\", %s->%s);\n",
		variable, name, variable);
	}

	else if ((flag == LogicOpt) || (flag == IntegerOpt))
	{
	   if (writer->tag == true)
	       fprintf(file,
		"\tfprintf(stderr, \"%s = %%d\\n\", %s->%s_toggle);\n",
		variable, name, variable);
	   else
	       fprintf(file,
		"\tfprintf(stderr, \"%s = %%d\\n\", %s->%s);\n",
		variable, name, variable);
	}

	else if (flag == FloatOpt)
	{
	   if (writer->tag == true)
	       fprintf(file,
		"\tfprintf(stderr, \"%s = %%f\\n\", %s->%s_toggle);\n",
		variable, name, variable);
	   else
	       fprintf(file,
		"\tfprintf(stderr, \"%s = %%f\\n\", %s->%s);\n",
		variable, name, variable);
	}

}  /* end gw_print_toggle_debug_stmt */


/************************************************************
*
*      Routine: gw_print_variable_tag
*
*      Purpose: this routine adds the tag "_file", "_int", etc.
*		to a routine.
*
*        Input: flag      - typeflag of database line
*
*    Called by: gw_print_debug_stmt
*
*   Written By: Danielle Argiro
*
*************************************************************/

gw_print_variable_tag(file, flag)
FILE  *file;
int flag;
{
	switch (flag) 
	{
	    case InputFile:
 	    case OutputFile:
		 fprintf(file, "_file");
	         break;
	    case FloatOpt:
		 fprintf(file, "_float");
	         break;
	    case IntegerOpt:
		 fprintf(file, "_int");
	         break;
	    case StringOpt:
		 fprintf(file, "_string");
	         break;
	    case LogicOpt:
		 fprintf(file, "_logic");
	         break;
	}

}  /* end gw_print_variable_tag */
