/* copyright 1990, Grace Giras, Pittsburgh Supercomputing Center */
/* all rights reserved */
/* lvr.c GPLOT laser video functions module */
/* lvr_ctrl()        - initialize lvr controller delimiter functions 
/* lvr_setup()       - B_Mf (Begin Metafile) I/O open; opts handling; lvr setup
/* lvr_noop()	     - B_Pic_Body (Begin Picture Body) no-op stub
/* lvr_record()	     - E_Pic (End Picture) record lvr framd
/* lvr_end()         - E_Mf (End Metafile) write user data; close I/O
/* current_frame()   - get current frame # - check disk inserted 
/* options_info()    - setup user data struct; dump options to audit file
/* select_blank()    - select largest blank video area or first user data rec 
/* empty_input()      - empty any garbage in lvr input buffer
/* input_bytes()      - either get bytes of input until a COMPLETEION character
/*                      of non-ACK return character OR get specified # of bytes
/* input_byte()       - input 1 byte, returning byte and putting in g_return
/* ouput_cmds()       - output a commands and parameters to lvr and get back 
                        ACK and any expected COMPLETION codes
/* open_audit()       - open AUDIT_FILE
/* au_dump()          - write to AUDIT_FILE and stderr
/* open_lvr_port()    - open lvr port for read/write
/* close_files()      - close open files
 */

#include <stdio.h>
#include <string.h>
#include <time.h>
#ifndef vms
#include <sys/time.h>
#endif
#ifdef ultrix
#include <sys/types.h> 
#endif

#include "lvr_defines.h"
#include "defs.h"

int lvr_setup(), lvr_noop(), lvr_record(), lvr_end();


static bool debug_flag = OFF;

static char tmp[MAX_STRING], buf[MAX_STRING];
static struct one_opt *popt;    /* the command line options, in only */
static struct record_data_rec user_data;
static unsigned int frames_recorded = 0;
static unsigned int user_data_rec = 0;
static bool rec_standby = OFF;	


/* ********** lvr_ctrl() ************ */
/* this is called at startup to initialize function pointers */
void lvr_ctrl(ctrl, opt)
int (*ctrl[])();	/* controller functions, out only */
struct one_opt opt[];	/* the command line options, in only */
   {
   /* the controller delimiter functions */
   ctrl[(int) B_Mf] 	      = lvr_setup;
   ctrl[(int) B_Pic_Body]     = lvr_noop;
   ctrl[(int) E_Pic]	      = lvr_record; 
   ctrl[(int) E_Mf]	      = lvr_end;

   /* store the options pointer */
   popt = opt;
   }


/* ********** lvr_setup() ************* */
/* set up for laser_video */

lvr_setup()
   {
   char dev_name[MAX_STRING];
   if ( popt[(int) debug].set) debug_flag = ON; /* debug switch */
   strcpy(dev_name, DEFAULT_PORT);
   if ( popt[(int)lvr].set ) 			/* lvr controller port */
      {
      if (strlen( popt[(int)lvr].val.str) )     /* -Vport option has port */
           strncpy(dev_name, popt[(int)lvr].val.str,MAX_STRING);
      dev_name[MAX_STRING-1] = '\0';	
      }
   setup_tty( dev_name);	      /* open lvr port for I/O */
   open_audit( AUDIT_FILE);	      /* open debug/audit file */
   empty_input();                     /* purge input port of any garbage */
   current_frame();	              /* get current frame - check disk in */
   options_info();	              /* setup user data; dump options info */

   user_data_rec = select_blank(OFF);     /* find next blank user data rec */
   disk_info();			      /* get disk init info */
   if (user_data_rec == START_DATA_FRAME) /* 90001 for disk not user info */
      {
      au_dump("Disk not Initialized - User Record 90001 blank",YES);
      exit(1);
      }         

   select_blank(ON);	              /* record standby mode for record
                                         on largest video blank area */
   }

/* ********** lvr_noop() ************* */
/* no-op at beginning of picture display */
lvr_noop()
   {
   }

/* *********** lvr_record() *********** */
lvr_record()
   {
   if (rec_standby == ON)
      {
      buf[0] = REC;
      if ( output_cmds(buf,1) != COMPLETION)
	 {
	 au_dump("Frame Not Recorded !!",YES);
         lvr_end();
	 }
      else frames_recorded++;
      }
   }

/* ********** lvr_end() ************ */
/* this is called at end of Metafile */
lvr_end()
   {
   char *pt;
   char set_data   = SET_DATA;
   char write_data = WRITE_DATA;
   char sum = 0;
   int i;

   if (rec_standby == OFF) return;  /* lvr_end already turned off rec standby */

   /* else */
   rec_standby = OFF;
   /* au_dump("Record Standby Mode Off",NO); */
   buf[0] = CLEAR;
   output_cmds(buf,1);

   sprintf(tmp,"Total frames recorded = %d", frames_recorded);
   au_dump(tmp,YES);

   if (frames_recorded) 
      {   
      user_data.end = (unsigned short)current_frame();

      /* get to user data area */
      sprintf(buf,"%c%d%c", SEARCH, user_data_rec, ENTER);
      output_cmds(buf,strlen(buf));

      sprintf(tmp,"End Frame = %d\n", user_data.end);
      au_dump(tmp,YES);

#ifdef vms
      user_data.timestamp  += GMT_ADJUST;	/* add 4 hours to adjust to
                                                  seconds GMT */
      flip_unix( &user_data);
#endif
      for( pt = (char *)&user_data, i=0; i < USER_DATA_LEN; i++) sum += pt[i];
      user_data.checksum = - sum;

      if( output_cmds(&set_data,1) != ACK) 
	 { au_dump("User Data Error: Write Set Data failed",YES); exit(1); }

      for (i =0, pt = (char *)&user_data; i < USER_DATA_LEN + 1; i++)
	      output_byte( (int)*pt++);  /* no ACK's */

      if (input_byte() != COMPLETION)
	 { au_dump("User Data Error: Write of Data failed",YES); exit(1); }

      if (  output_cmds(&write_data,1) != COMPLETION) 
	 { au_dump("User Data Error: Write failed??",YES); exit(1); }
      }
   close_io();
   close_audit();
   }
    

/* *** current_frame() *** */
/* get current frame #  */
current_frame()
   {
   int frame;
   int return_code;
   buf[0] = ADDR_INQ;
   if ((return_code=output_cmds(buf,1)) == NAK) 
      {
      au_dump("No Disk is Inserted or Not in Remote Control Mode!!", YES);
      exit(1);
      }

   input_bytes(buf,5);
   buf[5] = 0;
   sscanf(buf,"%d", &frame);
   return(frame);
   }


/* **** options_info() **** */
options_info()
   {
   register i;
   /* do initial setup of user data record */
   user_data.type[0]   = USER_RECORD_DATA;
   user_data.timestamp  = time(0);
   if ( popt[(int)user].set )
      {   
      strncpy(user_data.username, popt[(int)user].val.str,MAXUSER);
      for (i = 0 ; i < MAXUSER; i++)
	 user_data.username[i] = tolower(user_data.username[i]);
      }
   if ( popt[(int)copies].set )
	 user_data.copies = popt[(int)copies].val.i;
   else  user_data.copies = 1;    /* playback at normal speed */

   /* dump all options stuff to audit/debug file */
   au_dump("\n*********",NO);

   sprintf(tmp,"date     = %s   (%d)",
           (char *)ctime(&user_data.timestamp), user_data.timestamp);
   au_dump(tmp,NO);

   sprintf(tmp,"user     = %.*s",MAXUSER,user_data.username);
   au_dump(tmp,NO);

   sprintf(tmp,"copies   = %d",user_data.copies);
    au_dump(tmp,NO);

   if ( popt[(int)device].set )
      {
      sprintf(tmp,"device   = %s",popt[(int)device].val.str); 
      au_dump(tmp,NO);
      }

   if ( popt[(int)in_name].set )
      {
      strncpy(user_data.cgm, popt[(int)in_name].val.str,MAXCGM);
      for (i = 0 ; i < MAXCGM; i++)
	 user_data.cgm[i] = tolower(user_data.cgm[i]);
      sprintf(tmp,"cgm      = %.*s",MAXCGM, user_data.cgm);
      au_dump(tmp,NO);
      }

   if ( popt[(int)title_string].set )
      {
      sprintf(tmp,"title    = %s",popt[(int)title_string].val.str);
      au_dump(tmp,NO);
      }

   if ( popt[(int)eject].set )   /* eject laser disk and abort gplot!! */
      {
      sprintf(tmp,"eject    = YES");
      au_dump(tmp,NO);
      au_dump("Ejecting Laser Disk and Exiting", YES);
      buf[0] = EJECT;
      output_cmds(buf,1);
#ifdef vms
      exit(1);
#else
      exit(0);
#endif
      }

   /* etc !!!!!!!!!!!!!!!!!!!! */
   }


static FILE *lvr_info;

/* *** open_audit() **** */
/* Open a file for debug/ record progress */
open_audit(fil) char *fil;
{
#ifdef vms
   lvr_info = fopen( fil,"a","rfm=var", "rat=cr");
#else
   lvr_info = fopen( fil ,"a");
#endif
}
/* ********* close_audit() ********** */
close_audit()
{ 
fclose( lvr_info);
}
/* ***** au_dump() *** */
au_dump(message, output_flag)
char *message;
bool output_flag;	/* if YES output message to stderr regardless of debug_flag */
   {
   if (debug_flag == ON || output_flag == YES ) fprintf(stderr,"%s\n", message);
   if (!lvr_info) return;
   fprintf(lvr_info,"%s\n", message);
   fflush(lvr_info);
   }




/* **** select_blank() *** */
select_blank(video_flag)
bool video_flag;	/* On if for Video Area and Off if for Data Area */
   {

   int first_in, first_out;
   int biggest_area;
   int in, out = 0;

   /* Do minimal Setup and get to center */

   sprintf(buf,"%c%c%c%s%c", CLEAR, FRAME_MODE,SEARCH,
           video_flag == ON ? "1" : "90001" ,ENTER);
   output_cmds(buf,strlen(buf));

   buf[0] = CURRENT_BLANK_INQ;
   if ( output_cmds(buf,1) == NAK)   /* Need to do Blank Area Search */
      {
      if ( video_flag)
         au_dump("Doing Video Blank Area Search - Takes 60 seconds", YES);
      buf[0] = BLANK_AREA_SEARCH;
      if ( output_cmds(buf,1) == COMPLETION)
	 au_dump("Blank Area Search Complete",YES);
      else { au_dump("Blank Area Search failed",YES); exit(1); }
      /* trying again */
      buf[0] = CURRENT_BLANK_INQ;
      output_cmds(buf,1);
      }

   input_bytes(buf,10);
   buf[10] = 0;
   sscanf(buf,"%5d%5d", &first_in, &first_out);
   biggest_area = first_out - first_in;
   sprintf(tmp,"%s Blank Area : %5d - %5d",
              video_flag == ON ? "Video" : "User Data", first_in, first_out);
   au_dump(tmp,NO);

   if (video_flag == OFF)	/* get to first blank user data rec */
      {
      sprintf(tmp,"user data frame          =   %d", first_in);
      au_dump(tmp,YES);
      return(first_in);
      }

   /* video  */
   sprintf(buf,"%c%c", FRAME_REC,REC_STANDBY);
   if (output_cmds(buf,strlen(buf)) != COMPLETION)
     { au_dump("Record Setup Failure",YES); exit(1); }

   /* au_dump("Record Standby Active",NO); */

   for (;;)
      {
      buf[0] = EDGE_RIGHT;
      output_cmds(buf,1);

      buf[0] = CURRENT_BLANK_INQ;
      output_cmds(buf,1);

      input_bytes(buf,10);
      buf[10] = 0;
      sscanf(buf,"%5d%5d", &in, &out);
      if (in == first_in && out == first_out) break;
      else
	 {
	 if ( (out-in) > biggest_area) biggest_area = out -in;
	 sprintf(tmp,"Video Blank Area : %5d - %5d", in, out);
         au_dump(tmp,NO);
	 }   
      }

   while( ( out -in) != biggest_area)   
      {
      buf[0] = EDGE_RIGHT;
      output_cmds(buf,1);
      buf[0] = CURRENT_BLANK_INQ;
      output_cmds(buf,1);
      input_bytes(buf,10);
      buf[10] = 0;
      sscanf(buf,"%5d%5d", &in, &out);
      }
   user_data.start = in;
   user_data.end   = out;		/* for now  */
   sprintf(tmp,"start video record frame =   %d ( %d -%d blank )", in, in, out);
   au_dump(tmp,YES);

 
   sprintf(buf,"%c%c%c", ENTER, ENTER, ENTER);  /* use current blank area */
   output_cmds(buf,strlen(buf));
   if (input_byte() != COMPLETION)
      {
      au_dump("Record Standby IN, OUT Bad",YES);
      exit(1);
      }

   rec_standby = ON;
   /* au_dump("Ready to Record",NO); */
   return(in);
   }

/* ************* disk_info() ****************** */
disk_info()
   {
   static int disk_init_frame[2] = { START_DATA_FRAME, END_DATA_FRAME };
   char read_data = READ_DATA;
   struct disk_data_rec *rec; 
   int i,frame;
   /* get to frame */
  for (frame=0;frame < 2;frame++)
      {
      sprintf(buf,"%c%d%c",SEARCH, disk_init_frame[frame] ,ENTER);
      if (output_cmds(buf,strlen(buf)) != COMPLETION)
         { au_dump("Search for Disk Init Frame Failed??", YES); exit(1); }

      if ( output_cmds(&read_data,1) != COMPLETION)
         { au_dump("Disk Initialization Frame not written", YES); exit(1); }

      for (i = 0; i < USER_DATA_LEN +1; i++) buf[i] = input_byte();

      if ( (unsigned char)buf[0] == DISK_INIT_DATA && buf[1]  == '\0')
         {
	 au_dump("Disk Initialization :",YES);
#ifdef vms
	 flip_unix( rec);
	 rec->timestamp -= GMT_ADJUST;
#endif
	 rec = (struct disk_data_rec *)buf;
	 sprintf(tmp, "disk initialization date     = %s",
                                          (char *)ctime(&rec->timestamp));
	 au_dump(tmp,YES);
	 sprintf(tmp, "disk #                       = %X", rec->timestamp);
	 au_dump(tmp,YES);
	 return;
         }
      }	 
   }         


#ifdef vms
/* ****************** flip_unix() ******************* */
/* flip binary info in user data to/from unix */
flip_unix(rec)
char *rec;
   {
   if ( (unsigned char)rec[0] == USER_RECORD_DATA && rec[1]  == '\0')
	 flip_record(rec );
   else if ( (unsigned char)rec[0] == DISK_INIT_DATA && rec[1]  == '\0')
	 flip_disk( rec );
   }

/* ***************** flip_disk() ************** */
flip_disk(rec)
char *rec;
{
 /* timestamp */
 char tmp;
 tmp = rec[4];
 rec[4] = rec[7];
 rec[7] = tmp;
 tmp = rec[5];
 rec[5] = rec[6];
 rec[6] = tmp;

 /* side */
 tmp = rec[8];
 rec[8] = rec[9];
 rec[9] = tmp;

}

/* *************** flip_record() ************** */
/* put user data record session bianry info into unix format */
flip_record(rec)
char *rec;
 {
 /* timestamp */
 char tmp;
 tmp = rec[4];
 rec[4] = rec[7];
 rec[7] = tmp;
 tmp = rec[5];
 rec[5] = rec[6];
 rec[6] = tmp;

 /* start */
 tmp = rec[8];
 rec[8] = rec[9];
 rec[9] = tmp;

 /* end */
 tmp = rec[10];
 rec[10] = rec[11];
 rec[11] = tmp;

 /* copies */
 tmp = rec[12];
 rec[12] = rec[13];
 rec[13] = tmp;
 }

#endif


        
