#############################################################################
##
#A  Matrix Group and G-module library                   Derek Holt
#A                                                      Charles Leedham-Green
#A                                                      Eamonn O'Brien
#A                                                      Sarah Rees 
##
#A  @(#)$Id$
##
#Y  Copyright 1994 -- School of Mathematical Sciences, ANU   
##
#H  $Log$
##
#matorb.g
#
#############################################################################
##
#F  HashVal( rsg ) . . . . . value of a hash function applied to subspace
## HashVal is called by SpinSpacePerm (below) to calculate a hashed
## value for the subspace generated by rsg.
## <rsg> is assumed to be triangulized.

HashVal := function(rsg)

  local F,w,zero,one,dim,dimrs,val,inc,size,i,j,k,vec;

  inc := 3;
  size := 2^15;

  dim := Length(rsg[1]);
  dimrs := Length(rsg);
  F := Field(Flat(rsg));
  w := F.root; zero := F.zero; one :=F.one;

  vec := rsg[1];
  for i in [2..dimrs] do vec := vec+rsg[i]; od;

  val := 0;
  k := 1;
  for j in [1..dim] do
     if vec[j] <> zero then
       if vec[j] = one then
         val := val + k;
       else 
         val := (val + k*(LogFFE(vec[j],w)+1));
       fi;
       k := (k*inc) mod size;
     fi;
  od;
  val := val mod size;

  return val+1;

end;

#############################################################################
##
#F  SpinSpacePerm( rsg,matrices,max ) . . . . . perm action of matrix group
##                                   on orbit of subspace generated by rsg.
##
## SpinSpacePerm( rsg,matrices,max ) computes the orbit of the subspace 
## generated by the list of generators <rsg>
## under the action of the list <matrices>. If this orbit has size
## bigger than <max>, it aborts and returns the empty list. Otherwise, it
## returns a list giving the permutation actions of the generators on this
## orbit.
## A hashtable is used to locate the number of a subspace.
SpinSpacePerm := function ( rsg,matrices,max )
  local dimrs,orbit,orbitlength,perms,ngens,dim,lc,i,j,k,crsg,nrsg,new,
        hashtable,val,hv,lhv,rti,rt,rth;

  hashtable := [];
  TriangulizeMat(rsg);
  dim := Length(rsg[1]);
  if dim <> Length(matrices[1]) then
     Error("Rowspace and matrices have different dimensions.");
  fi;
  dimrs := Length(rsg);
  if dimrs = 0 then
    Error("Rowspace is zero.");
  fi;
  ngens := Length(matrices);

  perms := [];
  for i in [1..ngens] do perms[i]:=[]; od;
  orbit := [rsg];
  orbitlength := 1;
  val := HashVal(rsg);
  hashtable[val] := [[rsg,1]];

  i := 1;
  while i <= orbitlength do
    crsg := orbit[i];
    for j in [1..ngens] do
      nrsg := crsg * matrices[j];
      TriangulizeMat(nrsg);
# See if this is a new element of the orbit
#rt := Runtime();
      val := HashVal(nrsg);
#rth := rth + Runtime()-rt;
      if IsBound(hashtable[val]) then
        new := true;
        hv := hashtable[val];
        lhv := Length(hv);
        for k in [1..lhv] do
          if hv[k][1] = nrsg then
            new := false;
            perms[j][i] := hv[k][2];
          fi;
        od;
        if new then
          orbitlength := orbitlength+1;
          if orbitlength > max then
            return [];
          fi;
          hv[lhv+1] := [nrsg,orbitlength];
          orbit[orbitlength] := nrsg;
          perms[j][i] := orbitlength;
          if orbitlength mod 50 = 0 then
            Print("orbitlength = ",orbitlength,"\n");
          fi;
        fi;
      else
        orbitlength := orbitlength+1;
        if orbitlength > max then
          return [];
        fi;
        hashtable[val] := [[nrsg,orbitlength]];
        orbit[orbitlength] := nrsg;
        perms[j][i] := orbitlength;
        if orbitlength mod 50 = 0 then
          Print("orbitlength = ",orbitlength,"\n");
        fi;
      fi;

    od;

    i:=i+1;
  od;
  
  for i in [1..ngens] do perms[i] := PermList(perms[i]); od;

#rt := Runtime();
#Print ("Total time = ",Int((rt-rti)/1000 +1/2),"; Hash time = ",
#                       Int(rth/1000 + 1/2),"\n");
  return perms;
end;

#############################################################################
##
#F  SpinVecPerm( vec,matrices,max ) . . . . . perm action of matrix group
##                                   on orbit of vector vec.
##
## SpinVecPerm( vec,matrices,max ) computes the orbit of the vector vec
## under the action of the list <matrices>. If this orbit has size
## bigger than <max>, it aborts and returns the empty list. Otherwise, it
## returns a list giving the permutation actions of the generators on this
## orbit.
## A hashtable is used to locate the number of a subspace.
SpinVecPerm := function ( vec,matrices,max )
  local dimrs,orbit,orbitlength,perms,ngens,dim,lc,i,j,k,cvec,nvec,new,
        hashtable,val,hv,lhv,rti,rt,rth;

  hashtable := [];
  dim := Length(vec);
  if dim <> Length(matrices[1]) then
     Error("Rowspace and matrices have different dimensions.");
  fi;
  ngens := Length(matrices);

  perms := [];
  for i in [1..ngens] do perms[i]:=[]; od;
  orbit := [vec];
  orbitlength := 1;
  val := HashVal([vec]);
  hashtable[val] := [[vec,1]];

  i := 1;
  while i <= orbitlength do
    cvec := orbit[i];
    for j in [1..ngens] do
       nvec := cvec * matrices[j];
# See if this is a new element of the orbit
      val := HashVal([nvec]);
      if IsBound(hashtable[val]) then
        new := true;
        hv := hashtable[val];
        lhv := Length(hv);
        for k in [1..lhv] do
          if hv[k][1] = nvec then
            new := false;
            perms[j][i] := hv[k][2];
          fi;
        od;
        if new then
          orbitlength := orbitlength+1;
          if orbitlength > max then
            return [];
          fi;
          hv[lhv+1] := [nvec,orbitlength];
          orbit[orbitlength] := nvec;
          perms[j][i] := orbitlength;
          if orbitlength mod 50 = 0 then
            Print("orbitlength = ",orbitlength,"\n");
          fi;
        fi;
      else
        orbitlength := orbitlength+1;
        if orbitlength > max then
          return [];
        fi;
        hashtable[val] := [[nvec,orbitlength]];
        orbit[orbitlength] := nvec;
        perms[j][i] := orbitlength;
        if orbitlength mod 50 = 0 then
          Print("orbitlength = ",orbitlength,"\n");
        fi;
      fi;

    od;

    i:=i+1;
  od;
  
  for i in [1..ngens] do perms[i] := PermList(perms[i]); od;

  return perms;
end;
