/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/* matrix.c */

/* matrix routines, from Numerical Recipes */

/* Note: matrix allocation routine assumes whole matrix can be 
   allocated in one call to calloc.  If not true (say for large
   matrices on an IBM PC), dmatrix() and free_matrix() will
   have to be modified */

#include "include.h"

/* error routine */
void nrerror(error_text)
char *error_text;
{ 
  error(error_text,RECOVERABLE);
}

/* copies matrix b to matrix a */
/* for zero-base indexing only */
void matcopy(a,b,rows,cols)
REAL **a,**b;
int rows,cols;
{
  int i;

  for ( i = 0 ; i < rows ; i++ )
    memcpy((char *)a[i],(char *)b[i],cols*sizeof(REAL));
}

/* allocation */
REAL **dmatrix(rlo,rhi,clo,chi)
int rlo,rhi,clo,chi;
/* allocates a REAL matrix with range [rlo..rhi][clo..chi] */
{
  int i;
  REAL **m;

  if ( rhi-rlo+1 == 0 ) return NULL;
  m = (REAL **)mycalloc((unsigned)(rhi-rlo+1),sizeof(REAL *));
  if ( !m ) nrerror("dmatrix allocation error.");
  m -= rlo;

  m[rlo] = (REAL *) mycalloc((unsigned)(chi-clo+1),(rhi-rlo+1)*sizeof(REAL));
  if ( !m[rlo] ) nrerror("dmatrix allocation error.");
  for ( i = rlo+1 ; i <= rhi ; i++ )
      m[i] = m[rlo] + (i - rlo)*(chi - clo + 1) - clo;

  return m;
}

REAL ***dmatrix3(n1,n2,n3)
int n1,n2,n3;
/* allocates a REAL matrix with range [0..n1-1][0..n2-1][0..n3-1] */
{
  int i,j;
  REAL ***m;

  /* assumes all pointers same machine size and alignment */
  m = (REAL ***)mycalloc((n2+1)*n1,sizeof(REAL *));
  if ( !m ) nrerror("dmatrix3 allocation error.");

  for ( i = 0 ; i < n1 ; i++ )
    m[i] = (REAL **)(m + n1 + i*n2);
  m[0][0] = (REAL *) mycalloc(n1*n2*n3,sizeof(REAL));
  if ( !m[0][0] ) nrerror("dmatrix3 allocation error.");
  for ( i = 0 ; i < n1 ; i++ )
    for ( j = 0 ; j < n2 ; j++ )
      m[i][j] = m[0][0] + i*n2*n3 + j*n3;

  return m;
}

int *ivector(lo,hi)
int lo,hi;
/* allocates a int vector with range [lo..hi] */
{
  int *v;

  v = (int *)mycalloc((unsigned)(hi-lo+1),sizeof(int));
  if ( !v ) nrerror("allocation failure in ivector().");
  return v-lo;
}

REAL *vector(lo,hi)
int lo,hi;
/* allocates a REAL vector with range [lo..hi] */
{
  REAL *v;

  v = (REAL *)mycalloc((unsigned)(hi-lo+1),sizeof(REAL));
  if ( !v ) nrerror("allocation failure in vector().");
  return v-lo;
}

void free_ivector(v,lo,hi)
int *v,lo,hi;
{
  free((char *)(v+lo));
}

void free_vector(v,lo,hi)
REAL *v;
int lo,hi;
{
  free((char *)(v+lo));
}

void free_matrix(m)
REAL **m;
{
  if ( !m ) return;
  free((char *)m[0]);
  free((char *)m);
}

void free_matrix3(m)
REAL ***m;
{
  if ( !m ) return;
  free((char *)m[0][0]);
  free((char *)m);
}

/* given 3 points, find cross product of sides */
void vnormal(a,b,c,n)
REAL *a,*b,*c,*n;
{
  REAL aa[MAXCOORD],bb[MAXCOORD];
  int i;

  for ( i = 0 ; i < web.sdim ; i++ )
    {
      aa[i] = a[i] - c[i];
      bb[i] = b[i] - c[i];
    }
  cross_prod(aa,bb,n);
}
  
         

void cross_prod(a,b,c)
REAL *a,*b,*c;
{
  c[0] = a[1]*b[2] - a[2]*b[1];
  c[1] = a[2]*b[0] - a[0]*b[2];
  c[2] = a[0]*b[1] - a[1]*b[0];
} 

REAL triple_prod(a,b,c)
REAL *a,*b,*c;
{
  return  a[0]*(b[1]*c[2] - b[2]*c[1]) - a[1]*(b[0]*c[2] - b[2]*c[0])
          + a[2]*(b[0]*c[1] - b[1]*c[0]);
}

#ifndef MATASM
/* dot product */
REAL dot(a,b,n)
REAL *a,*b;
int n;  /* number of items */
{
  REAL x = 0.0;
  for (  ; --n >= 0 ;  ) x += (*(a++))*(*(b++));
  return x;
}
#endif

/* dot product for floats */
REAL dotf(a,b,n)
float *a,*b;
int n;  /* number of items */
{
  REAL x = 0.0;
  for (  ; --n >= 0 ;  ) x += (*(a++))*(*(b++));
  return x;
}


#ifndef MATASM
/* matrix times vector multiplication */
void matvec_mul(a,b,c,rows,cols)
REAL **a,*b,*c;
int rows,cols;
{
  int i,j;

  for ( i = 0 ; i < rows ; i++ )
    { c[i] = 0.0;
      for ( j = 0 ; j < cols ; j++ )
        c[i] += a[i][j]*b[j];
    }
}
#endif


#ifndef MATASM
/* vector times matrix multiplication */
void vec_mat_mul(a,b,c,rows,cols)
REAL *a,**b,*c;
int rows,cols;
{
  int i,j;

  for ( i = 0 ; i < cols ; i++ )
    { c[i] = 0.0;
      for ( j = 0 ; j < rows ; j++ )
        c[i] += a[j]*b[j][i];
    }
}
#endif

#ifndef MATASM
/* matrix by matrix multiplication */
/* output: c = a*b */
void mat_mult(a,b,c,imax,jmax,kmax)
REAL **a,**b,**c;  /* not assumed distinct */
int imax,jmax,kmax;
{
  REAL **temp;  /* temporary storage, if needed */
  int i,j,k;

  if ( (a == c) || (b == c) )
    {
      temp = dmatrix(0,imax-1,0,kmax-1);  /* temporary storage */
      for ( i = 0 ; i < imax ; i++ )
        for ( j = 0 ; j < jmax ; j++ )
          for ( k = 0 ; k < kmax ; k++ )
            temp[i][k] += a[i][j]*b[j][k];
      matcopy(c,temp,imax,kmax);
      free_matrix(temp);
    }
  else
    {
      for ( i = 0 ; i < imax ; i++ )
        for ( k = 0 ; k < kmax ; k++ )
          {
            c[i][k] = 0.0;
            for ( j = 0 ; j < jmax ; j++ )
              c[i][k] += a[i][j]*b[j][k];
          }
    }
}
#endif

/* matrix transpose by matrix multiplication */
/* output: c = aT*b */
void tr_mat_mul(a,b,c,imax,jmax,kmax)
REAL **a,**b,**c;  /* not assumed distinct */
int imax,jmax,kmax; /* a is imax x jmax, b is imax x kmax, c is jmax x kmax */
{
  REAL **temp;  /* temporary storage, if needed */
  int i,j,k;

  if ( (a == c) || (b == c) )
    {
      temp = dmatrix(0,jmax-1,0,kmax-1);  /* temporary storage */
      for ( j = 0 ; j < jmax ; j++ )
        for ( k = 0 ; k < kmax ; k++ )
          for ( i = 0 ; i < imax ; i++ )
            temp[j][k] += a[i][j]*b[i][k];
      matcopy(c,temp,jmax,kmax);
      free_matrix(temp);
    }
  else
    { REAL *s,*t;
      for ( j = 0 ; j < jmax ; j++ )
        for ( k = 0, s = c[j] ; k < kmax ; k++,s++ )
          {
            *s = 0.0;
            for ( i = 0, t = a[j] ; i < imax ; i++ )
              *s += a[i][j]*b[i][k];
          }
    }
}

/* matrix by matrix transpose multiplication */
/* output: c = a*bT */
void mat_mul_tr(a,b,c,imax,jmax,kmax)
REAL **a,**b,**c;  /* not assumed distinct */
int imax,jmax,kmax;
{
  REAL **temp;  /* temporary storage, if needed */
  int i,j,k;

  if ( (a == c) || (b == c) )
    {
      temp = dmatrix(0,imax-1,0,kmax-1);  /* temporary storage */
      for ( i = 0 ; i < imax ; i++ )
        for ( j = 0 ; j < jmax ; j++ )
          for ( k = 0 ; k < kmax ; k++ )
            temp[i][k] += a[i][j]*b[k][j];
      matcopy(c,temp,imax,kmax);
      free_matrix(temp);
    }
  else
    {
      for ( i = 0 ; i < imax ; i++ )
        for ( k = 0 ; k < kmax ; k++ )
          {
            c[i][k] = 0.0;
            for ( j = 0 ; j < jmax ; j++ )
              c[i][k] += a[i][j]*b[k][j];
          }
    }
}

#ifndef MATASM
/* quadratic form evaluation; only uses lower triangle */
double quadratic_form(a,b,c,n)
REAL *a,**b,*c;
int n;
{ int i,j;
  double sum = 0.0;
  double temp;

  for ( i = 0 ; i < n ; i++ )
    { temp = b[i][0]*c[0];
      for ( j = 1 ; j <= i ; j++ )
        temp += b[i][j]*c[j];
      for (  ; j < n ; j++ )
        temp += b[j][i]*c[j];
      sum += a[i]*temp;
    }

  return sum;
}
#endif


/* inverse by gauss-jordan */
/* returns -1 for singular matrix */

#define SWAP(a,b) {REAL temp = (a); (a) = (b); (b) = temp; }
#define SMALL 10

int mat_inv(a,n)
REAL **a;    /* matrix to invert in place */
int n;         /* size of matrix */
{
  int *indxc,*indxr,*ipiv;
  int i,icol=0,irow=0,j,k,l,ll;
  REAL big,dum,pivinv;
  int retval = 1;  /* default return value is success */
  int temp1[SMALL],temp2[SMALL],temp3[SMALL]; /* avoid alloc for small sizes */

  if ( n <= SMALL )
    { indxc = temp1; indxr = temp2; ipiv = temp3; }
  else
    { /* large size */
      indxc = ivector(0,n-1);
      indxr = ivector(0,n-1);
      ipiv  = ivector(0,n-1);
    }
  for ( j = 0 ; j < n ; j++ ) ipiv[j] = -1;
  for ( i = 0 ; i < n ; i++ )
    { big = 0.0;
      for ( j = 0 ; j < n ; j++ )
        if ( ipiv[j] != 0 )
          for ( k = 0 ; k < n ; k++ )
            { if ( ipiv[k] == -1 )
                { if ( fabs(a[j][k]) >= big )
                   { big = fabs(a[j][k]);
                     irow = j;
                     icol = k;
                   }
                }
              else if ( ipiv[k] > 0 ) { retval = -1; goto mat_inv_exit; }
            }
       ++(ipiv[icol]);

       if ( irow != icol )
          for ( l = 0 ; l < n ; l++ ) SWAP(a[irow][l],a[icol][l])
       indxr[i] = irow;
       indxc[i] = icol;
       if ( a[icol][icol] == 0.0 ) { retval = -1; goto mat_inv_exit; }
       pivinv = 1/a[icol][icol];
       a[icol][icol] = 1.0;
       for ( l = 0 ; l < n ; l++ ) a[icol][l] *= pivinv;
       for ( ll = 0  ; ll < n ; ll++ )
         if ( ll != icol )
           { dum = a[ll][icol];
             a[ll][icol] = 0.0;
             for ( l = 0 ; l < n ; l++ ) a[ll][l] -= a[icol][l]*dum;
           }
    }
  for ( l = n-1 ; l >= 0 ; l-- )
    { if ( indxr[l] != indxc[l] )
        for ( k = 0 ; k < n ; k++ )
          SWAP(a[k][indxr[l]],a[k][indxc[l]])
    }

mat_inv_exit:
  if ( n > SMALL )
    {
      free_ivector(ipiv,0,n-1);
      free_ivector(indxr,0,n-1);
      free_ivector(indxc,0,n-1);
    }
  return retval;
}

/* calculates determinant in place and leaves adjoint transpose */
double  det_adjoint(a,n)
REAL **a;    /* matrix to change in place */
int n;         /* size of matrix */
{
  int *indxc,*indxr,*ipiv;
  int i,icol=0,irow=0,j,k,l,ll;
  REAL big,dum,pivinv,piv;
  int temp1[SMALL],temp2[SMALL],temp3[SMALL]; /* avoid alloc for small sizes */
  double det = 1.0;  /* will multiply by pivots */

  if ( n <= 0 ) error("Matrix size not positive.",RECOVERABLE);
  if ( n == 1 ) { det = a[0][0]; a[0][0] = 1.0; return det; }
  if ( n == 2 )
    { REAL temp;
      det = a[0][0]*a[1][1] - a[0][1]*a[1][0];
      temp = a[0][0]; a[0][0] = a[1][1]; a[1][1] = temp;
      a[0][1] = -a[0][1]; a[1][0] = -a[1][0];
      return det;
    }

  if ( n <= SMALL )
    { indxc = temp1; indxr = temp2; ipiv = temp3; }
  else
    { /* large size */
      indxc = ivector(0,n-1);
      indxr = ivector(0,n-1);
      ipiv  = ivector(0,n-1);
    }
  for ( j = 0 ; j < n ; j++ ) ipiv[j] = -1;
  for ( i = 0 ; i < n-1 ; i++ )
    { big = 0.0;
      for ( j = 0 ; j < n ; j++ )
        if ( ipiv[j] != 0 )
          for ( k = 0 ; k < n ; k++ )
            { if ( ipiv[k] == -1 )
                { if ( fabs(a[j][k]) >= big )
                   { big = fabs(a[j][k]);
                     irow = j;
                     icol = k;
                   }
                }
              else if ( ipiv[k] > 0 )
               { error("ipiv > 0.\n",WARNING); det = 0.0; goto det_exit; }
            }
       ++(ipiv[icol]);

       if ( irow != icol )
         { for ( l = 0 ; l < n ; l++ ) SWAP(a[irow][l],a[icol][l])
	   det = -det;
	 }
       indxr[i] = irow;
       indxc[i] = icol;
       det *= a[icol][icol];  /* build determinant */
       if ( a[icol][icol] == 0.0 ) { goto det_lowrank; }
       pivinv = 1/a[icol][icol];
       a[icol][icol] = 1.0;
       for ( l = 0 ; l < n ; l++ ) a[icol][l] *= pivinv;
       for ( ll = 0  ; ll < n ; ll++ )
         if ( ll != icol )
           { dum = a[ll][icol];
             a[ll][icol] = 0.0;
             for ( l = 0 ; l < n ; l++ ) a[ll][l] -= a[icol][l]*dum;
           }
    }
  /* special treatment for last pivot; works even if zero */
  for ( j = 0 ; j < n ; j++ )
    if ( ipiv[j] != 0 ) { irow = icol = j; break; }
  indxr[n-1] = irow;
  indxc[n-1] = icol;
  piv = a[icol][icol];
  a[icol][icol] = 1.0;
  for ( l = 0 ; l < n ; l++ ) a[icol][l] *= det;
  for ( ll = 0  ; ll < n ; ll++ )
    if ( ll != icol )
      { dum = a[ll][icol];
        a[ll][icol] = 0.0;
        for ( l = 0 ; l < n ; l++ ) 
           a[ll][l] = a[ll][l]*piv*det - a[icol][l]*dum;
       }
  det *= piv;

  for ( l = n-1 ; l >= 0 ; l-- )
    { if ( indxr[l] != indxc[l] )
        for ( k = 0 ; k < n ; k++ )
          SWAP(a[k][indxr[l]],a[k][indxc[l]])
    }

det_exit:
  if ( n > SMALL )
    {
      free_ivector(ipiv,0,n-1);
      free_ivector(indxr,0,n-1);
      free_ivector(indxc,0,n-1);
    }
  return det;

det_lowrank: /* rank less than n-1, so adjoint = 0 */
  for ( i = 0 ; i < n ; i++ )
    for ( j = 0 ; j < n ; j++ )
      a[i][j] = 0.0;
  det = 0.0;
  goto det_exit;
  
}


/* calculates determinant; no change in matrix for 3x3 or smaller */
double  determinant(a,n)
REAL **a;    /* matrix to change in place */
int n;         /* size of matrix */
{
  if ( n == 1 ) { return a[0][0];  }
  if ( n == 2 )
    { 
      return  a[0][0]*a[1][1] - a[0][1]*a[1][0];
    }
  if ( n == 3 )
    { return a[0][0]*(a[1][1]*a[2][2] - a[1][2]*a[2][1])
	   - a[0][1]*(a[1][0]*a[2][2] - a[1][2]*a[2][0])
	   + a[0][2]*(a[1][0]*a[2][1] - a[1][1]*a[2][0]);
    }
  return det_adjoint(a,n);  /* other cases */
}

void print_matrix(a,rows,cols)
REAL **a;
int rows,cols;
{
  int i,j;

  for ( i = 0 ; i < rows ; i++ )
   { for ( j = 0 ; j < cols ; j++ )
      fprintf(outfd,"%10.6f ",a[i][j]);
     fprintf(outfd,"\n");
   }
}

/* conversion of k vectors to a k-vector */
/* components in index lexicographic order */
void exterior_product(v,w,k,n)
REAL **v;  /* list of k vectors */
REAL *w;   /* returned k-vector */
int k;     /* number of vectors */
int n;     /* space dimension */
{
  /* anticipate only small k, so just brute force */
  int i1,i2,i3;

  switch ( k )
   {
     case 1:  for ( i1 = 0 ; i1 < n ; i1++ ) *(w++) = v[0][i1];
	      break;

     case 2:  for ( i1 = 0 ; i1 < n ; i1++ )
		for ( i2 = i1+1 ; i2 < n ; i2++ )
		  *(w++) = v[0][i1]*v[1][i2] - v[0][i2]*v[1][i1];
              break;

     case 3:  for ( i1 = 0 ; i1 < n ; i1++ )
		for ( i2 = i1+1 ; i2 < n ; i2++ )
		  for ( i3 = i2+1 ; i3 < n ; i3++ )
		    *(w++) = v[0][i1]*v[1][i2]*v[2][i3]
		           + v[0][i2]*v[1][i3]*v[2][i1] 
		           + v[0][i3]*v[1][i1]*v[2][i2] 
		           - v[0][i1]*v[1][i3]*v[2][i2] 
		           - v[0][i3]*v[1][i2]*v[2][i1] 
		           - v[0][i2]*v[1][i1]*v[2][i3] ;
              break;

     default: sprintf(errmsg,"Exterior product of %d vectors.\n",k);
	      error(errmsg,RECOVERABLE);
	      break;
   }
}

/**********************************************************************
*
*  function: kernel_basis()
*
*  purpose:  Find basis for kernel of matrix
*/

int kernel_basis(a,ker,imax,jmax)
REAL **a;  /* the matrix, will be altered */
REAL **ker; /* for basis vectors in columns */
int imax,jmax;  /* rows and columns of a */
{
  int i,j,k;
  int pivrow[20];   /* pivot row in column */
  int n; /* nullity */

  for ( j = 0 ; j < jmax ; j++ ) pivrow[j] = -1;  /* mark as no pivot in col */

  /* get row echelon form, pivot largest in each row */
  for ( i = 0 ; i < imax ; i++ )
    { int piv = -1;
      double b,big;

      /* find largest element in row */
      big = 0.0;
      for ( j = 0 ; j < jmax ; j++ )
        if ( fabs(a[i][j]) > big )
	   { big = fabs(a[i][j]);
	     piv = j;
           }
      if ( piv == -1 ) continue; /* row of zeros */
      pivrow[piv] = i;

      /* pivot step */
      for ( j = 0 ; j < jmax ; j++ )
	a[i][j] /= a[i][piv];
      for ( k = 0 ; k < imax ; k++ )
        for ( j = 0 ; j < jmax ; j++ )
	    { if ( k == i ) continue;
	      b = a[k][piv];
	      a[k][j] -= a[k][piv]*a[i][j];
            }
    }       

  /* now find kernel basis */
  for ( j = 0, n = 0 ; j < jmax ; j++ )
    { if ( pivrow[j] >= 0 ) continue;  /* column has leading 1 */
      /* column j is parameter column */
      for ( k = 0 ; k < jmax ; k++ )
	{ if ( pivrow[k] >= 0 )
	    ker[k][n] = -a[pivrow[k]][j];
          else if ( k == j )
	    ker[k][n] = 1.0;
	  else ker[k][n] = 0.0;
        }
      n++;
    }
  return n; /* nullity */
}


