/*
 * ifconfig	This file contains an implementation of the command
 *		that either displays or sets the characteristics of
 *		one or more of the system's networking interfaces.
 *
 * Usage:	ifconfig [-a] [-v] interface [address | up | down |
 *			metric NN | mtu NN | arp | -arp | trailers |
 *			-trailers | broadcast aa.bb.cc.dd |
 *			dstaddr aa.bb.cc.dd | netmask aa.bb.cc.dd |
 *			pointopoint aa.bb.cc.dd .. ]
 *
 * Version:	@(#)ifconfig.c	1.12	05/25/93
 *
 * Author:	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#ifdef LINUX
#   include <linux/ddi.h>
#endif
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <netinet/in.h>
#include <net/if.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <resolv.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>


struct interface {
  char			name[IFNAMSIZ];		/* interface name	*/
  short			flags;			/* various flags	*/
  short			family;			/* address family	*/
  int			metric;			/* routing metric	*/
  int			mtu;			/* MTU value		*/
  struct sockaddr	addr;			/* IP address		*/
  struct sockaddr	dstaddr;		/* P-P IP address	*/
  struct sockaddr	broadaddr;		/* IP broadcast address	*/
  struct sockaddr	netmask;		/* IP network mask	*/
};

  
char *Version = "@(#) ifconfig 1.12 (05/25/93)";


int opt_a = 0;				/* show all interfaces		*/
int opt_v = 0;				/* debugging output flag	*/
int sock = -1;				/* AF_INET raw socket desc.	*/
struct addrtyp {
  char		*name;
  int		af;
  char		*(*ntoa)();		/* no specified args!!!!	*/
} addrtyp[] = {				/* the address families		*/
  { "(none)",	AF_UNSPEC,	NULL		},
#ifdef AF_UNIX
  { "UNIX",	AF_UNIX,	NULL		},
#endif
#ifdef AF_INET
  { "IP",	AF_INET,	inet_ntoa	},
#endif
  { NULL,	-1,		NULL		}
};


/* Find the correct address family handler. */
static struct addrtyp *get_af(int af)
{
  register int i;

  i = 0;
  while (addrtyp[i].af != -1) {
	if (addrtyp[i].af == af) return(&addrtyp[i]);
	i++;
  }
  return((struct addrtyp *)NULL);
}


static int resolve(char *name, struct in_addr *in)
{
  struct hostent *hp;

  /* Hack- this address is refused by the resolver routines! */
  if (!strcmp(name, "255.255.255.255")) {
	in->s_addr = INADDR_BROADCAST;
	return(0);
  }
  if ((hp = gethostbyname(name)) == (struct hostent *)NULL) return(-1);
  memcpy((char *) in, (char *) hp->h_addr_list[0], hp->h_length);
  return(0);
}


static void ife_print(struct interface *ptr)
{
  struct addrtyp *ap;

  ap = get_af(ptr->family);

  printf("%-8.8s  %s ADDR %s  ", ptr->name, ap->name,
	ap->ntoa((*(struct sockaddr_in *) &ptr->addr).sin_addr));
  printf("BCAST %s  ",
	ap->ntoa((*(struct sockaddr_in *) &ptr->broadaddr).sin_addr));
  printf("NETMASK %s\n",
	ap->ntoa((*(struct sockaddr_in *) &ptr->netmask).sin_addr));
  printf("          MTU %d  METRIC %d", ptr->mtu, ptr->metric);
  printf("  POINT-TO-POINT ADDR %s",
	ap->ntoa((*(struct sockaddr_in *) &ptr->dstaddr).sin_addr));
  printf("\n          FLAGS: 0x%04X ( ", ptr->flags);
  if (ptr->flags & IFF_UP) printf("UP ");
  if (ptr->flags & IFF_BROADCAST) printf("BROADCAST ");
  if (ptr->flags & IFF_DEBUG) printf("DEBUG ");
  if (ptr->flags & IFF_LOOPBACK) printf("LOOPBACK ");
  if (ptr->flags & IFF_POINTOPOINT) printf("POINTOPOINT ");
  if (ptr->flags & IFF_NOTRAILERS) printf("NOTRAILERS ");
  if (ptr->flags & IFF_RUNNING) printf("RUNNING ");
  if (ptr->flags & IFF_NOARP) printf("NOARP ");
  if (ptr->flags & IFF_PROMISC) printf("PROMISC ");
  if (ptr->flags & IFF_ALLMULTI) printf("ALLMULTI ");
  printf(")\n\n");
}


/* Fetch the inteface configuration from the kernel. */
static int if_fetch(char *ifname, struct interface *ife)
{
  struct ifreq ifr;

  memset((char *) ife, 0, sizeof(struct interface));
  strcpy(ife->name, ifname);

  strcpy(ifr.ifr_name, ifname);
  if (ioctl(sock, SIOCGIFADDR, &ifr) < 0) return(-1);
  ife->addr = ifr.ifr_addr;
  ife->family = ifr.ifr_addr.sa_family;

  strcpy(ifr.ifr_name, ifname);
  if (ioctl(sock, SIOCGIFFLAGS, &ifr) < 0) return(-1);
  ife->flags = ifr.ifr_flags;

  strcpy(ifr.ifr_name, ifname);
  if (ioctl(sock, SIOCGIFMETRIC, &ifr) < 0) return(-1);
  ife->metric = ifr.ifr_metric;

#ifdef SIOCGIFMTU
  strcpy(ifr.ifr_name, ifname);
  if (ioctl(sock, SIOCGIFMTU, &ifr) < 0) return(-1);
  ife->mtu = ifr.ifr_mtu;
#endif

  strcpy(ifr.ifr_name, ifname);
  if (ioctl(sock, SIOCGIFDSTADDR, &ifr) < 0) return(-1);
  ife->dstaddr = ifr.ifr_dstaddr;

  strcpy(ifr.ifr_name, ifname);
  if (ioctl(sock, SIOCGIFBRDADDR, &ifr) < 0) return(-1);
  ife->broadaddr = ifr.ifr_broadaddr;

#ifdef SIOCGIFNETMASK
  strcpy(ifr.ifr_name, ifname);
  if (ioctl(sock, SIOCGIFNETMASK, &ifr) < 0) return(-1);
#ifdef LINUX
  ife->netmask = ifr.ifr_netmask;
#else
  memcpy(ife->netmask.sa_data, &ifr.ifr_data, sizeof(struct sockaddr));
#endif
#endif

  return(0);
}


static void if_print(char *ifname)
{
  char buff[1024];
  struct interface ife;
  struct ifconf ifc;
  struct ifreq *ifr;
  int i;

  if (ifname == (char *)NULL) {
	ifc.ifc_len = sizeof(buff);
	ifc.ifc_buf = buff;
	if (ioctl(sock, SIOCGIFCONF, &ifc) < 0) {
		fprintf(stderr, "SIOCGIFCONF: %s\n", strerror(errno));
		return;
	}

	ifr = ifc.ifc_req;
	for (i = ifc.ifc_len / sizeof(struct ifreq); --i >= 0; ifr++) {
		if (if_fetch(ifr->ifr_name, &ife) < 0) {
			fprintf(stderr, "%s: unknown interface.\n",
							ifr->ifr_name);
			continue;
		}

		if (((ife.flags & IFF_UP) == 0) && !opt_a) continue;
		ife_print(&ife);
	}
  } else {
	if (if_fetch(ifname, &ife) < 0)
		fprintf(stderr, "%s: unknown interface.\n", ifname);
	  else ife_print(&ife);
  }
}


/* Set a certain interface flag. */
static void set_flag(char *ifname, short flag)
{
  struct ifreq ifr;

  strcpy(ifr.ifr_name, ifname);
  if (ioctl(sock, SIOCGIFFLAGS, &ifr) < 0) {
	fprintf(stderr, "SIOCGIFFLAGS: %s\n", strerror(errno));
  }
  ifr.ifr_flags |= flag;
  if (ioctl(sock, SIOCSIFFLAGS, &ifr) < 0) {
	fprintf(stderr, "SIOCSIFFLAGS: %s\n", strerror(errno));
  }
}


/* Clear a certain interface flag. */
static void clr_flag(char *ifname, short flag)
{
  struct ifreq ifr;

  strcpy(ifr.ifr_name, ifname);
  if (ioctl(sock, SIOCGIFFLAGS, &ifr) < 0) {
	fprintf(stderr, "SIOCGIFFLAGS: %s\n", strerror(errno));
  }
  ifr.ifr_flags &= ~flag;
  if (ioctl(sock, SIOCSIFFLAGS, &ifr) < 0) {
	fprintf(stderr, "SIOCSIFFLAGS: %s\n", strerror(errno));
  }
}


static void usage()
{
  fprintf(stderr, "Usage: ifconfig [-av] interface [address | up | down |\n");
  fprintf(stderr, "       metric NN | mtu NN | arp | -arp | trailers |\n");
  fprintf(stderr, "       -trailers | broadcast aa.bb.cc.dd |\n");
  fprintf(stderr, "       dstaddr aa.bb.cc.dd | netmask aa.bb.cc.dd |\n");
  fprintf(stderr, "       pointopoint aa.bb.cc.dd ... ]\n");
  exit(-1);
}


void main(argc, argv)
int argc;
char *argv[];
{
  struct ifreq ifr;
  struct in_addr in;
  char **spp;
  char *sp;

  /* Create a socket to the INET kernel. */
  if ((sock = socket(AF_INET, SOCK_RAW, PF_INET)) < 0) {
	perror("socket");
	exit(-1);
  }

  /* Find any options. */
  argc--;
  argv++;
  while (*argv[0] == '-') {
	if (!strcmp(*argv, "-a")) opt_a = 1;
	if (!strcmp(*argv, "-v")) opt_v = 1;
	argv++;
	argc--;
  }

  /* Do we have to show the current setup? */
  if (argc == 0) {
	if_print((char *)NULL);
	(void) close(sock);
	exit(0);
  }

  /* No. Fetch the interface name. */
  spp = argv;
  strncpy(ifr.ifr_name, *spp++, IFNAMSIZ);
  if (*spp == (char *)NULL) {
	if_print(ifr.ifr_name);
	(void) close(sock);
	exit(0);
  }

  /* Process the remaining arguments. */
  while (*spp != (char *)NULL) {
#ifdef SIOCSIFMETRIC
	if (!strcmp(*spp, "metric")) {
		sp = *++spp;
		if (sp == NULL) usage();
		spp++;
		ifr.ifr_metric = atoi(sp);
		if (ioctl(sock, SIOCSIFMETRIC, &ifr) < 0) {
			fprintf(stderr, "SIOCSIFMETRIC: %s\n", strerror(errno));
		}
		continue;
	}
#endif

#ifdef SIOCSIFMTU
	if (!strcmp(*spp, "mtu")) {
		sp = *++spp;
		if (sp == NULL) usage();
		spp++;
		ifr.ifr_mtu = atoi(sp);
		if (ioctl(sock, SIOCSIFMTU, &ifr) < 0) {
			fprintf(stderr, "SIOCSIFMTU: %s\n", strerror(errno));
		}
		continue;
	}
#endif

#ifdef SIOCSIFBRDADDR
	if (!strcmp(*spp, "broadcast")) {
		sp = *++spp;
		if (sp == NULL) usage();
		spp++;
		if (resolve(sp, &in) < 0) {
			herror(sp);
			continue;
		}
		(*(struct sockaddr_in *) &ifr.ifr_broadaddr).sin_addr = in;
		if (ioctl(sock, SIOCSIFBRDADDR, &ifr) < 0) {
			fprintf(stderr, "SIOCSIFBRDADDR: %s\n",
						strerror(errno));
		}
		set_flag(ifr.ifr_name, IFF_BROADCAST);
		continue;
	}
#endif

#ifdef SIOCSIFDSTADDR
	if (!strcmp(*spp, "dstaddr")) {
		sp = *++spp;
		if (sp == NULL) usage();
		spp++;
		if (resolve(sp, &in) < 0) {
			herror(sp);
			continue;
		}
		(*(struct sockaddr_in *) &ifr.ifr_dstaddr).sin_addr = in;
		if (ioctl(sock, SIOCSIFDSTADDR, &ifr) < 0) {
			fprintf(stderr, "SIOCSIFDSTADDR: %s\n",
							strerror(errno));
		}
		set_flag(ifr.ifr_name, IFF_BROADCAST);
		continue;
	}
#endif

#ifdef SIOCSIFNETMASK
	if (!strcmp(*spp, "netmask")) {
		sp = *++spp;
		if (sp == NULL) usage();
		spp++;
		if (resolve(sp, &in) < 0) {
			herror(sp);
			continue;
		}
#ifdef LINUX
		(*(struct sockaddr_in *) &ifr.ifr_netmask).sin_addr = in;
#else
	(*(struct in_addr *) &ifr.ifr_data) = in;
#endif
		if (ioctl(sock, SIOCSIFNETMASK, &ifr) < 0) {
			fprintf(stderr, "SIOCSIFNETMASK: %s\n",
						strerror(errno));
		}
		continue;
	}
#endif

#ifdef IFF_NOARP
	if (!strcmp(*spp, "arp")) {
		spp++;
		clr_flag(ifr.ifr_name, IFF_NOARP);
		continue;
	}
	if (!strcmp(*spp, "-arp")) {
		spp++;
		set_flag(ifr.ifr_name, IFF_NOARP);
		continue;
	}
#endif

#ifdef IFF_NOTRAILERS
	if (!strcmp(*spp, "trailers")) {
		spp++;
		clr_flag(ifr.ifr_name, IFF_NOTRAILERS);
		continue;
	}

	if (!strcmp(*spp, "-trailers")) {
		spp++;
		set_flag(ifr.ifr_name, IFF_NOTRAILERS);
		continue;
	}
#endif

#ifdef IFF_PROMISC
	if (!strcmp(*spp, "promisc")) {
		spp++;
		set_flag(ifr.ifr_name, IFF_PROMISC);
		continue;
	}

	if (!strcmp(*spp, "-promisc")) {
		spp++;
		clr_flag(ifr.ifr_name, IFF_PROMISC);
		continue;
	}
#endif

#ifdef IFF_ALLMULTI
	if (!strcmp(*spp, "allmulti")) {
		spp++;
		set_flag(ifr.ifr_name, IFF_ALLMULTI);
		continue;
	}

	if (!strcmp(*spp, "-allmulti")) {
		spp++;
		clr_flag(ifr.ifr_name, IFF_ALLMULTI);
		continue;
	}
#endif

#ifdef IFF_UP
	if (!strcmp(*spp, "up")) {
		spp++;
		set_flag(ifr.ifr_name, (IFF_UP | IFF_RUNNING));
		continue;
	}

	if (!strcmp(*spp, "down")) {
		spp++;
		clr_flag(ifr.ifr_name, IFF_UP);
		continue;
	}
#endif

#ifdef IFF_POINTOPOINT
	if (!strcmp(*spp, "pointopoint")) {
		spp++;
		if (resolve(*spp, &in)) {
			herror(*spp);
			spp--;
			continue;
		};

		(*(struct sockaddr_in *) &ifr.ifr_dstaddr).sin_addr = in;
		ifr.ifr_dstaddr.sa_family = AF_INET;
		if (ioctl(sock, SIOCSIFDSTADDR, &ifr) < 0) {
			fprintf(stderr, "SIOCSIFDSTADDR: %s\n",
							strerror(errno));
		}
		set_flag(ifr.ifr_name, IFF_POINTOPOINT);
		spp++;
		continue;
	};
#endif

	/* If the next argument is a valid hostname, assume OK. */
	if (resolve(*spp, &in) < 0) usage();
	spp++;

	(*(struct sockaddr_in *) &ifr.ifr_addr).sin_addr = in;
	ifr.ifr_addr.sa_family = AF_INET;
	if (ioctl(sock, SIOCSIFADDR, &ifr) < 0) {
		fprintf(stderr, "SIOCSIFADDR: %s\n", strerror(errno));
	}
	set_flag(ifr.ifr_name, (IFF_UP | IFF_RUNNING));
	continue;
  }

  /* Close the socket. */
  (void) close(sock);

  exit(0);
}
