/*
 * @(#)signal.c	1.5 91/09/05
 *
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/signal.h>

#include "defs.h"

char *signals[] = {
#ifdef linux
	"SIGNULL",
	"SIGHUP",
	"SIGINT",
	"SIGQUIT",
	"SIGILL",
	"SIGTRAP",
	"SIGABRT",
	"SIGUNUSED",
	"SIGFPE",
	"SIGKILL",
	"SIGUSR1",
	"SIGSEGV",
	"SIGUSR2",
	"SIGPIPE",
	"SIGALRM",
	"SIGTERM",
	"SIGSTKFLT",
	"SIGCHLD",
	"SIGCONT",
	"SIGSTOP",
	"SIGTSTP",
	"SIGTTIN",
	"SIGTTOU",
	"SIGIO",
	"SIGXCPU",
	"SIGXFSZ",
	"SIGVTALRM",
	"SIGPROF",
	"SIGWINCH",
	"SIGLOST",
	"SIG30",
	"SIG31",
	"SIG32",
#else
     "SIGNULL",
     "SIGHUP",   /*    1    hangup */
     "SIGINT",   /*    2    interrupt */
     "SIGQUIT",  /*    3*   quit */
     "SIGILL",   /*    4*   illegal instruction */
     "SIGTRAP",  /*    5*   trace trap */
     "SIGIOT",   /*    6*   IOT trap (not generated on Suns) */
     "SIGEMT",   /*    7*   EMT trap (A-line or F-line op code) */
     "SIGFPE",   /*    8*   arithmetic exception */
     "SIGKILL",  /*    9    kill (cannot be caught, blocked, or ignored) */
     "SIGBUS",   /*    10*  bus error */
     "SIGSEGV",  /*    11*  segmentation violation */
     "SIGSYS",   /*    12*  bad argument to system call */
     "SIGPIPE",  /*    13   write on a pipe with no one to read it */
     "SIGALRM",  /*    14   alarm clock */
     "SIGTERM",  /*    15   software termination signal */
     "SIGURG",   /*    16@  urgent condition present on socket */
     "SIGSTOP",  /*    17+  stop (cannot be caught, blocked, or ignored) */
     "SIGTSTP",  /*    18+  stop signal generated from keyboard */
     "SIGCONT",  /*    19@  continue after stop (cannot be blocked) */
     "SIGCHLD",  /*    20@  child status has changed */
     "SIGTTIN",  /*    21+  background read attempted from control terminal */
     "SIGTTOU",  /*    22+  background write attempted to control terminal */
     "SIGIO",    /*    23@  I/O is possible on a descriptor (see fcntl(2)) */
     "SIGXCPU",  /*    24   cpu time limit exceeded (see setrlimit(2)) */
     "SIGXFSZ",  /*    25   file size limit exceeded (see setrlimit(2)) */
     "SIGVTALRM", /*   26   virtual time alarm (see setitimer(2)) */
     "SIGPROF",  /*    27   profiling timer alarm (see setitimer(2)) */
     "SIGWINCH", /*    28@  window changed (see win(4S)) */
     "SIGLOST",  /*    29*  resource lost (see lockd(8C)) */
     "SIGUSR1",  /*    30   user-defined signal 1 */
     "SIGUSR2"   /*    31   user-defined signal 2 */
#endif
};

static Xlat sigvec_flags[] = {
#ifndef linux
	SV_ONSTACK,	"ONSTACK",
	SV_INTERRUPT,	"INTERRUPT",
	SV_RESETHAND,	"RESETHAND",
#else
	SA_STACK,	"STACK",
	SA_RESTART,	"RESTART",
	SA_INTERRUPT,	"INTERRUPT",
	SA_NOMASK,	"NOMASK",
	SA_ONESHOT,	"ONESHOT",
#endif
	SA_NOCLDSTOP,	"NOCLDSTOP",
	0,		NULL,
};

static void
sprintsigmask(s, mask)
char *s;
sigset_t mask;
{
	int i;
	char *format;

	format = "%s";
	*s++ = '[';
	for (i = 1; i <= NSIG; i++) {
		if (mask & sigmask(i)) {
			sprintf(s, format, signals[i] + 3); s += strlen(s);
			format = " %s";
		}
	}
	*s++ = ']';
	*s = '\0';
}

static void
printsigmask(mask)
sigset_t mask;
{
	char outstr[1024];

	sprintsigmask(outstr, mask);
	fprintf(outf, "%s", outstr);
}

void
printsignal(nr)
{
	if (nr > 0 && nr <= NSIG)
		fprintf(outf, "%s", signals[nr]);
	else
		fprintf(outf, "%d", nr);
}

#ifndef linux
int
sys_sigvec(tcp)
struct tcb *tcp;
{
	struct sigvec sv;
	int addr;

	if (entering(tcp)) {
		printsignal(tcp->u_args[0]);
		addr = tcp->u_args[1];
	} else {
		addr = tcp->u_args[2];
	}
	if (addr == 0 || umove(tcp->pid, addr, sizeof sv, (char *)&sv) < 0) {
		fprintf(outf, "(struct sigvec *)0");
	} else {
		switch (sv.sv_handler) {
		case SIG_ERR:
			fprintf(outf, "SIG_ERR");
			break;
		case SIG_DFL:
			fprintf(outf, "SIG_DFL");
			break;
		case SIG_IGN:
			fprintf(outf, "SIG_IGN");
			break;
		default:
			fprintf(outf, "{%#x, ", sv.sv_handler);
			printsigmask(sv.sv_mask);
			fprintf(outf, ", ");
			if (!printflags(sigvec_flags, sv.sv_flags))
				putc('0', outf);
			putc('}', outf);
		}
	}
	if (entering(tcp))
		fprintf(outf, ", ");
	return 0;
}

int
sys_sigblock(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printsigmask(tcp->u_args[0]);
#if 0
		if ((tcp->u_args[0] & sigmask(SIGTRAP)) && brutal) {
			pokearg(tcp->pid, 0, tcp->u_args[0] & ~sigmask(SIGTRAP));
		}
#endif
	}
	return 0;
}

int
sys_sigsetmask(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printsigmask(tcp->u_args[0]);
	}
	return 0;
}

int
sys_sigpause(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printsigmask(tcp->u_args[0]);
	}
	return 0;
}

int
sys_sigstack(tcp)
struct tcb *tcp;
{
	struct sigstack ss;
	int addr;

	if (entering(tcp)) {
		addr = tcp->u_args[0];
	} else {
		addr = tcp->u_args[1];
	}
	if (addr == 0 || umove(tcp->pid, addr, sizeof ss, (char *)&ss)) {
		fprintf(outf, "(struct sigstack *)0");
	} else {
		fprintf(outf, "{sp: %#x,", ss.ss_sp);
		fprintf(outf, "%s}", ss.ss_onstack?"ON":"OFF");
	}
	if (entering(tcp))
		fprintf(outf, ", ");
	return 0;
}

int
sys_sigcleanup(tcp)
struct tcb *tcp;
{
	return 0;
}
#else

int
sys_sigaction(tcp)
struct tcb *tcp;
{
	struct sigaction sv;
	int addr;

	if (entering(tcp)) {
		printsignal(tcp->u_args[0]);
		fprintf(outf, ", ");
		addr = tcp->u_args[1];
	} else {
		addr = tcp->u_args[2];
	}
	if (addr == 0 || umove(tcp->pid, addr, sizeof sv, (char *)&sv) < 0) {
		fprintf(outf, "NULL");
	} else {
		switch ((int) sv.sa_handler) {
		case SIG_ERR:
			fprintf(outf, "SIG_ERR");
			break;
		case SIG_DFL:
			fprintf(outf, "SIG_DFL");
			break;
		case SIG_IGN:
			fprintf(outf, "SIG_IGN");
			break;
		default:
			fprintf(outf, "{%#x, ", sv.sa_handler);
			printsigmask(sv.sa_mask);
			fprintf(outf, ", ");
			if (!printflags(sigvec_flags, sv.sa_flags))
				putc('0', outf);
			putc('}', outf);
		}
	}
	if (entering(tcp))
		fprintf(outf, ", ");
	return 0;
}

int
sys_signal(tcp)
struct tcb *tcp;
{
	struct sigaction sv;
	int addr;

	if (entering(tcp)) {
		printsignal(tcp->u_args[0]);
		switch (tcp->u_args[1]) {
		case SIG_ERR:
			fprintf(outf, "SIG_ERR");
			break;
		case SIG_DFL:
			fprintf(outf, "SIG_DFL");
			break;
		case SIG_IGN:
			fprintf(outf, "SIG_IGN");
			break;
		default:
			fprintf(outf, "%#x", tcp->u_args[1]);
		}
	}
	return 0;
}

int
sys_sigreturn(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printsigmask(tcp->u_args[1]);
		fprintf(outf, ", %#x, %#x", tcp->u_args[1], tcp->u_args[2]);
	}
	return RVAL_NONE;
}

int
sys_ssetmask(tcp)
struct tcb *tcp;
{
	static char outstr[1024];

	if (entering(tcp)) {
		printsigmask(tcp->u_args[0]);
	}
	else {
		sprintf(outstr, "omask:");
		sprintsigmask(outstr + 6, tcp->u_rval);
		tcp->auxstr = outstr;
	}
	return RVAL_HEX | RVAL_STR;
}

int
sys_sgetmask(tcp)
struct tcb *tcp;
{
	static char outstr[1024];

	if (exiting(tcp)) {
		sprintf(outstr, "mask:");
		sprintsigmask(outstr + 5, tcp->u_rval);
		tcp->auxstr = outstr;
	}
	return RVAL_HEX | RVAL_STR;
}

int
sys_sigsuspend(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		/* first two are not really arguments, but print them anyway */
		fprintf(outf, "%d, %#x, ", tcp->u_args[0], tcp->u_args[1]);
		printsigmask(tcp->u_args[2]);
	}
	return 0;
}

#endif

int
sys_kill(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%d, %s",
				tcp->u_args[0], signals[tcp->u_args[1]]);
	}
	return 0;
}

int
sys_killpg(tcp)
struct tcb *tcp;
{
	return sys_kill(tcp);
}

int
sys_sigpending(tcp)
struct tcb *tcp;
{
	sigset_t sigset;

	if (exiting(tcp)) {
		if (syserror(tcp) ||
			umove(tcp->pid, tcp->u_args[0],
				sizeof sigset, (char *)&sigset) < 0)
			fprintf(outf, "%#x", tcp->u_args[0]);
		else
			printsigmask(sigset);
	}
	return 0;
}
