#include <OI/oi.H>



/*
 *	translations - using a C language function as callbacks.
 *
 *	This program demonstrates the use of the translation mechanism
 *	to add callbacks to Mouse events.  This program should be
 *	executed as:
 *		./transC -config trans.ad
 *	Otherwise the appropriate mouse actions will not be set.
 *
 *	The reader should refer back to the README file for
 *	other translations examples.
 *
 *	The reader should refer to the OI documentation for
 *	information on the following member functions.
 *		- OIIntro
 *			OI_init()
 *			OI_begin_interaction()
 *			OI_fini()
 *			OI_add_actions()
 *		- OI_d_tech
 *			layout_associated_object()
 *			set_associated_object()
 *		- OI_app_window
 *			oi_create_app_window()
 *		- OI_static_text
 *			oi_create_static_text()
 *			disallow_cut_paste()
 */


/*
 *	The OI_actions_rec structure allows the user to specify
 *	either C-functions, or C++ member functions, to be called
 *	when the specified mouse event occurs.
 */
static void press(OI_d_tech*, const XEvent*, const char **, unsigned int*);
static void drag(OI_d_tech*, const XEvent*, const char **, unsigned int*);
static void release(OI_d_tech*, const XEvent*, const char **, unsigned int*);

static OI_actions_rec myactions[] = {
	{"press",	(OI_translation_fnp)press,	NULL, NULL_PMF },
	{"drag",	(OI_translation_fnp)drag,	NULL, NULL_PMF },
	{"release",	(OI_translation_fnp)release,	NULL, NULL_PMF },
};


void
main (int argc, char **argv)
{
	OI_connection	*conp;				/* the connection to the server */
	OI_app_window	*appWindow;			/* the enclosing app window */
	OI_static_text	*staticText;			/* the text label */

	
	/*
	 *	Open a connection.
	 */
	if ((conp = OI_init(&argc, argv, "actions", "Actions"))) {
		
		/*
		 *	OI_add_actions makes the connection between the user specified
		 *	mouse events (in trans.ad), with the actual callbacks.
		 */
	        OI_add_actions(myactions, OI_count(myactions));

		/*
		 *	create application window.
		 */
		appWindow = oi_create_app_window("myapp", 200, 200, "C Translations");

		/*
		 *	create static text object.
		 *	disallow cut and paste, to allow the user to select the object with
		 *		the mouse.
		 *	set the static text object into the application window.
		 */
		staticText = oi_create_static_text("label", " Drag Me ");
		staticText->disallow_cut_paste();
		staticText->set_associated_object(appWindow, 5, 5, OI_ACTIVE);

		/*
		 *	Display application window.
		 *	Begin interaction.
		 */
		appWindow->set_associated_object(conp->root(), OI_DEF_LOC, OI_DEF_LOC, OI_ACTIVE);
		OI_begin_interaction();
	}

	/*
	 *	Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini();
}


/*
 *	global variable to hold original button press offset.
 */
int deltaX, deltaY;


/*
 *	on ButtonPress
 *		- initialize deltaX and deltaY
 *		- if this was a ButtonPress Event,
 *			capture the actual event location and store as deltaX and deltaY.
 *		- refresh the object color.
 */
static void
press(OI_d_tech *objp, const XEvent *eventp, const char **, unsigned int *)
{
	deltaX = deltaY = 0;
	if (eventp->type == ButtonPress) {
		deltaX = eventp->xbutton.x;
		deltaY = eventp->xbutton.y;
	}
	objp->set_pixels(objp->fg_pixel(), objp->bkg_pixel(), objp->bdr_pixel());
}

/*
 *	on MotionNotify (drag event)
 *		- capture the event that caused this procedure to be called.
 *		- process all MotionNotify events in the queue (less choppy movement)
 *		- change the object's location from the old location to the new location
 */
static void
drag(OI_d_tech *objp, const XEvent *eventp, const char **, unsigned int *)
{
	XEvent event;
	if (eventp->type == MotionNotify) {
		event = *eventp;
		while (XCheckTypedEvent(objp->display(), MotionNotify, &event));
		objp->set_loc(event.xmotion.x+objp->loc_x()-deltaX, event.xmotion.y+objp->loc_y()-deltaY);
	}
}

/*
 *	on ButtonRelease
 *	refresh the object color.
 */
static void
release(OI_d_tech *objp, const XEvent *, const char **, unsigned int *)
{
	objp->set_pixels(objp->fg_pixel(), objp->bkg_pixel(), objp->bdr_pixel());
}
