#include <math.h>
#include <OI/oi.H>				/* ScrollBar.C */

/*
 *	Xlib calls
 *
 *	This program was created to introduce the programmer to working
 *	with both the OI library and raw Xlib calls.
 *
 *	This program demonstrates the basic form of all OI programs:
 *		OI_init			- initialize OI and open a connection
 *		OI_create_app_window	- create the main application window
 *		>> 
 *		>>	build forest of objects within the application window
 *		>>
 *		OI_begin_interaction	- begin interaction with the user
 *		OI_fini			- clean up
 *
 *	The reader should refer to the OI documentation for
 *	information on the following member functions.
 *		- OIIntro
 *			OI_init()
 *			OI_begin_interaction()
 *			OI_fini()
 *		- OI_d_tech
 *			set_associated_object()
 *			set_layout()
 *			layout_associated_object()
 *		- OI_app_window
 *			oi_create_app_window()
 *		- OI_box
 *			oi_create_box()
 *			set_expose()
 *		- OI_scroll_bar
 *			oi_create_scroll_bar
 *			set_span()
 *			set_view()
 */

static	GC	draw_gc = 0 ;

int
main( int argc, char **argv )
{
	int		mov_curve (OI_ctlr_1d*,void*,OI_scroll_event,long) ;
	void		expose(OI_d_tech*,void*,const XEvent*);
	OI_connection	*conp ;
	OI_app_window	*wp ;
	OI_scroll_bar	*sbp ;
	OI_box		*bp;
	

	/*
	 *	Open a connection.
	 */
	if (conp = OI_init(&argc,argv,"ScrollBar")) {
		/*
		 *	create the application window.
		 *	set layout to row major.
		 */
		wp = oi_create_app_window("main",200,100,"ScrollBar") ;
		wp->set_layout(OI_layout_column);

		/*
		 *	create the curve box.
		 *	place the box into the application window.
		 */
		bp = oi_create_box("curve_box",300,120);
		bp->layout_associated_object(wp,0,1,OI_ACTIVE);

		/*
		 *	create the scroll bar, setting the motion
		 *	callback to function `mov_curve', setting
		 *	the span and view, then placing the scrollbar
		 *	into the application window.
		 */
		sbp = oi_create_scroll_bar(
			"sc_bar",300,OI_HORIZONTAL,(OI_ctlr_1d_fnp) &mov_curve,(void*)bp);
		sbp->set_span(1000);
		sbp->set_view(100,3);
		sbp->layout_associated_object(wp,0,2,OI_ACTIVE) ;

		/*
		 *	set the expose callback for the box, to
		 *	routine expose, using the scrollbar as the
		 *	argument.
		 */
		bp->set_expose(expose,sbp) ;

		/*
		 *	display main window.
		 *	begin interaction.
		 */
		wp->set_associated_object(
			wp->root(),OI_DEF_LOC,OI_DEF_LOC,OI_ACTIVE) ;
		OI_begin_interaction();
	}

	/*
	 *	Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini();
	return( 0 );
}


/*
 *	create and draw the curve
 */
void
disp_curve( OI_scroll_bar *sbp, long start_unit, long end_unit )
{
	long		i;
	int		x1,y1,x2,y2;
	OI_box		*bp;
	XGCValues	gcv;
	double		damping;

	end_unit += sbp->handle_loc();
	start_unit += sbp->handle_loc();

	bp = (OI_box*)sbp->app_window()->subobject("curve_box");
	if (!draw_gc) {
		gcv.foreground = bp->fg_pixel();
		gcv.background = bp->bkg_pixel();
		draw_gc = XCreateGC(
			bp->display(), bp->X_window(),
			GCForeground|GCBackground,&gcv);
	}
	x1 = (int)(bp->size_x()/sbp->view_span()*(start_unit - sbp->handle_loc()));
	damping = (sbp->span() - 0.1*(9*start_unit+1))/(sbp->span() - 1);
	y1 = (int)(damping*sin(double(start_unit)/10.0)*bp->size_y()/2.0 + bp->size_y()/2.0);
	for (i = start_unit + 1 ; i <= end_unit ; i++) {
		x2 = (int)(bp->size_x()/sbp->view_span()*(i - sbp->handle_loc()));
		damping = (sbp->span() - 0.1*(9*i+1))/(sbp->span() - 1);
		y2 = (int)(damping*sin(double(i)/10.0)*bp->size_y()/2.0 + bp->size_y()/2.0);
		XDrawLine(bp->display(),bp->X_window(),draw_gc,x1,y1,x2,y2);
		x1 = x2 ;
		y1 = y2 ;
	}
	return;
}



/*
 *	display full curve
 *	
 *	If the app window exists, and the curve box exists within the
 *	app window, then Clear the window, and display the curve.
 */
void
disp_full_curve( OI_scroll_bar *sbp )
{
	OI_box	*bp;

	if (sbp->app_window()) {
		bp = (OI_box*)sbp->app_window()->subobject("curve_box");
		XClearWindow(bp->display(),bp->X_window());
		disp_curve( sbp, 0, sbp->view_span());
	}
	return;
}


/*
 *	display partial curve
 *	
 *	This routine is only called if the scroll bar moved in either
 *	direction by a few units.
 *
 */
void
disp_partial_curve( OI_scroll_bar *sbp, long num_u )
{
	int	num_pix;
	OI_box	*bp;
	Display	*d ;
	Window	xw;

	/*
	 *	get a pointer to the curve box.
	 *	then determine if we moved to the right or the left.
	 *	On closer examination, you will see that both halves of
	 *	this algorithm take into account which direction the scroll
	 *	bar moved, copies the displayed curve the specified number
	 *	of units in the opposite direction, clears the new area,
	 *	then computes and displays the necessary section of the
	 *	curve.
	 */
	bp = (OI_box*)sbp->app_window()->subobject("curve_box");
	d = bp->display();
	xw = bp->X_window();
	if (num_u > 0) {
		num_pix = (int)(bp->size_x()/sbp->view_span()*num_u);
		XCopyArea( d, xw, xw, draw_gc,num_pix,0,bp->size_x()-num_pix,bp->size_y(),0,0);
		XClearArea( d, xw, (int)(bp->size_x() - num_pix),0,num_pix,bp->size_y(),False);
		disp_curve(sbp, sbp->view_span() - num_u, sbp->view_span());
	}
	else {
		num_pix = (int)(bp->size_x()/sbp->view_span()*(-num_u));
		XCopyArea( d, xw, xw, draw_gc, 0, 0, bp->size_x()-num_pix, bp->size_y(),num_pix,0);
		XClearArea( d, xw, 0, 0, num_pix, bp->size_y(), False);
		disp_curve( sbp, 0, -num_u );
	}
	return;
}


/*
 *	expose callback function
 *
 *	The pointer to the scroll bar is sent in as the argument when
 *	this expose function was registered for the box.  Note that
 *	we wait until the X expose count reaches 0 before displaying
 *	the complete curve.
 */
void
expose( OI_d_tech *, void * argp, const XEvent *ep )
{
	OI_scroll_bar   *sbp;

	sbp = (OI_scroll_bar*)argp;
	if ((ep->type == Expose) && (ep->xexpose.count==0))
		disp_full_curve(sbp);
	else if ((ep->type == GraphicsExpose) && (ep->xgraphicsexpose.count==0))
		disp_full_curve(sbp);
}

/*
 *	scroll bar callback function.
 *
 *	move the display the specified number of units.
 */
int
mov_curve( OI_ctlr_1d *sbp, void*, OI_scroll_event typ, long n )
{
	switch (typ) {
	 case OI_SCROLL_UNIT:
		disp_partial_curve( (OI_scroll_bar*) sbp,n);
		break;
	 case OI_SCROLL_VIEWPORT:
	 case OI_SCROLL_EXTREME:
	 case OI_SCROLL_POSITION:
		disp_full_curve( (OI_scroll_bar*) sbp);
		break;
	}
	return(1) ;
}
