/*
 * Copyright (c) 2001-2005 Servertec. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS NOTICE MUST NOT BE ALTERED NOR REMOVED.
 *
 * CopyrightVersion 1.0
 */

import java.util.Vector;

import java.io.IOException;

import stec.pos.ObjectStore;
import stec.pos.ObjectStoreReference;
import stec.pos.ObjectStoreEnumeration;
import stec.pos.ObjectStoreEnumerationFilter;

public class Registry
{
	public static final char DELIMITER_CHAR = 0xff;
	private static final char[] delimiter = {DELIMITER_CHAR};
	public static final String DELIMITER = new String(delimiter);

	ObjectStoreReference reference;

	public Registry(String filename) throws IOException
	{
		if(!ObjectStore.exists(filename))
		{
			ObjectStore.create(filename, 100, 1003, -1, -1, false);
		}

		reference = ObjectStore.open(filename, true);
	}

	public final synchronized boolean containsKey(String key) throws IOException
	{
		ObjectStoreEnumeration e = reference.keys(new RegistryFilter(key));

		while(e.hasMoreElements())
		{
			return true;
		}

		return false;
	}

	public final synchronized boolean containsValue(String key) throws IOException
	{
		return reference.containsKey(key);
	}

	public final synchronized String get(String key) throws IOException
	{
		return (String)reference.get(key);
	}

	public final synchronized void put(String key, String value) throws IOException
	{
		reference.put(key, value);
	}

	public final synchronized void removeKey(String key) throws IOException
	{
		ObjectStoreEnumeration e = reference.keys(new RegistryFilter(key));

		while(e.hasMoreElements())
		{
			reference.remove(e.nextElement());
		}
	}

	public final synchronized void removeValue(String key) throws IOException
	{
		reference.remove(key);
	}

	public final synchronized void renameKey(String old_key, String new_key) throws IOException
	{
		if(old_key.endsWith(DELIMITER))
		{
			old_key.substring(0, old_key.length() - 1);
		}

		ObjectStoreEnumeration e = reference.keys(new RegistryFilter(old_key));

		String tkey;
		String key;
		String value;

		while(e.hasMoreElements())
		{
			key = (String)e.nextElement();
			value = (String)reference.get(key);

			tkey = new_key + key.substring(old_key.length());

			reference.put(tkey, value);
			reference.remove(key);
		}
	}

	public final synchronized void renameValue(String old_key, String new_key) throws IOException
	{
		Object value = reference.get(old_key);
		reference.put(new_key, value);
		reference.remove(old_key);
	}

	public final boolean isValue(String key) throws IOException
	{
		if(reference.get(key) == null)
		{
			return false;
		}
		else
		{
			return true;
		}
	}

	public final synchronized void clear() throws IOException
	{
		reference.clear();
	}

	public final synchronized Vector keys(String key) throws IOException
	{
		Vector keys = new Vector();

		String path;

		int soffset = key.length();

		// skip /
		if(!key.endsWith(DELIMITER))
		{
			soffset++;
		}

		int eoffset;

		ObjectStoreEnumeration e = reference.keys(new RegistryFilter(key));

		while(e.hasMoreElements())
		{
			path = (String)e.nextElement();

			if(reference.get(path) == null)
			{
				eoffset = path.indexOf(DELIMITER, soffset);

				if(eoffset == -1)
				{
					path = path.substring(soffset);
				}
				else
				{
					path = path.substring(soffset, eoffset);
				}

				if(path.length() > 0)
				{
					if(!keys.contains(path))
					{
						keys.addElement(path);
					}
				}
			}
		}

		return keys;
	}

	public final synchronized Vector values(String key) throws IOException
	{
		Vector keys = new Vector();

		String path;

		int soffset = key.length();

		// skip /
		if(!key.endsWith(DELIMITER))
		{
			soffset++;
		}

		int eoffset;

		ObjectStoreEnumeration e = reference.keys(new RegistryFilter(key));

		while(e.hasMoreElements())
		{
			path = (String)e.nextElement();

			eoffset = path.indexOf(DELIMITER, soffset);

			if(eoffset != -1)
			{
				continue;
			}

			if(reference.get(path) != null)
			{
				if(eoffset == -1)
				{
					path = path.substring(soffset);
				}
				else
				{
					path = path.substring(soffset, eoffset);
				}

				if(path.length() > 0)
				{
					if(!keys.contains(path))
					{
						keys.addElement(path);
					}
				}
			}
		}

		return keys;
	}

	public final synchronized void close() throws IOException
	{
		try
		{
			reference.close();
		}
		finally
		{
			reference = null;
		}
	}

	protected final synchronized void finalize() throws IOException
	{
		if(reference != null)
		{
			close();
		}
	}

	class RegistryFilter implements ObjectStoreEnumerationFilter
	{
		String key;

		RegistryFilter(String key)
		{
			if(!key.endsWith(DELIMITER))
			{
				this.key = key + DELIMITER;
			}
			else
			{
				this.key = key;
			}
		}

		public boolean matches(Object path, Object value)
		{
			String spath = (String)path;

			if(spath.startsWith(key))
			{
				return true;
			}
			else
			{
				return false;
			}
		}
	}
}