////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: EventGenerator 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _EVENT_GENERATOR_H_ 
#define _EVENT_GENERATOR_H_

#pragma C+

/*
 *  EventGenerator
 *
 *  Performs a list of events in sequence, allowing for randomized 
 *  execution of sub-elements in either singular or self-reducing
 *  fashion. 
 *
 *  For example, the following eventSequenceList:
 *  
 *      [[&event1],[&event2,&event3,&event4,&event5],[&event6],[&event7]]
 *
 *  has fixed execution elements &event1, &event6, and &event7; and
 *  random execution elements &event2, &event3, &event4, and &event5.
 *
 *  In self-reducing random list mode each element of sublist[2] is
 *  processed and removed from the sub-list, thus processing &event2, 
 *  &event3, &event4 and &event5 in a random order.
 *
 *  in non self-reducing random list mode only one element of sublist[2] 
 *  is randomly selected and processed.
 */
class EventGenerator: object
    sequenceList            = []
    selfReducingRandomList  = nil
    
    /*
     *  initialize( s1, ... )
     *
     *  initialize the sequenceList. If a second argument is passed then
     *  it indicates whether the list is to be a self-reducing random
     *  list or not.
     */
    initialize( sl, ... ) = {
        self.sequenceList      = sl;
        if ( argcount == 2 ) 
            selfReducingRandomList  = getarg( 2 );
    }
    
    /*
     *  The sequence list is processed beginning with sublist[1] thru
     *  the length of the sequence list. 
     */
    start = {
        local i, len = length( self.sequenceList );
        local eList = self.sequenceList;
        local elem;
        
        for ( i = 1; i <= len; i++ )
        {
            if ( selfReducingRandomList )
            {
                do {
                    elem = self.randomElement( eList[ i ] );
                    self.processElement( elem );
                    eList[ i ] -= [elem];
                }
                while( length( eList[ i ] ) > 0 );
            }
            else 
            {
                elem = self.randomElement( eList[ i ] ); 
                self.processElement( elem );
            }
        }
    }
    
    /*
     *  randomEvent( list )
     *
     *  Method returns a member from the list that is randomly selected.
     *  The range is from 1 to the length of the list. An empty list 
     */
    randomElement( list ) = {
        local len = length( list );
    
        if ( len == 0 )
            return nil;
        else return list[ _rand( len ) ];
    }
    
    /*
     *  processElement( elem )
     *
     *  This method can be overridden to indicate how the element is to
     *  be processed. The default is to treat the element as a function
     *  pointer and to indirectly call the function with no argument
     *  list.
     */
    processElement( elem ) = {
        return ((elem) ());
    }
;

#pragma C-

#endif  /* _EVENT_GENERATOR_H_ */
