/*
** Masta'mind, a Mastermind clone.
**
** Written by Stephen Granade for the TextFire project.
*/

#include <adv.t>
#include "bugs.t"
#include "plurals.t"
#include "consist.t"
#include "footnote.t"
#include "detect.t"

#pragma C+

die: function
{
    "\b*** Masta'mind is done ***\b";
    scoreRank();
    "\bYou may restore a saved game, start over, quit, or undo
    the current command.\n";
    while ( 1 )
    {
        local resp;

	"\nPlease enter RESTORE, RESTART, QUIT, or UNDO: >";
        resp = upper(input());
        if ( resp == 'RESTORE' )
	{
	    resp = askfile( 'File to restore' );
	    if ( resp == nil ) "Restore failed. ";
	    else if ( restore( resp )) "Restore failed. ";
	    else
	    {
	        Me.location.lookAround(true);
	        scoreStatus( global.wongames, global.numgames );
		abort;
	    }
	}
        else if ( resp == 'RESTART' )
	{
	    scoreStatus( 0, 0 );
            restart();
	}
	else if ( resp == 'QUIT' || resp == 'Q' )
        {
	    terminate();
            quit();
	    abort;
        }
	else if (resp == 'UNDO')
	{
	    if (undo())
	    {
		"(Undoing one command)\b";
		Me.location.lookAround(true);
	        scoreStatus(global.wongames, global.numgames);
		abort;
	    }
	    else
		"Sorry, no undo information is available. ";
	}
    }
}

incGames: function(won)
{
    global.numgames++;
    if (won)
        global.wongames++;
    setscore('Wins: '+cvtstr(global.wongames)+'  Attempts: '+
        cvtstr(global.numgames));
}

/*
 *   The scoreRank() function displays how well the player is doing.
 *   This default definition doesn't do anything aside from displaying
 *   the current and maximum scores.  Some game designers like to
 *   provide a ranking that goes with various scores ("Novice Adventurer,"
 *   "Expert," and so forth); this is the place to do so if desired.
 *
 *   Note that "global.maxscore" defines the maximum number of points
 *   possible in the game; change the property in the "global" object
 *   if necessary.
 */
scoreRank: function
{
    local i;

    if (global.numgames == 0)
        "You haven't played any games yet. Geez. ";
    else {
        "You have played <<global.numgames>> game<<global.numgames > 1 ?
            "s" : "">> and won <<global.wongames>>. This means you are ";
        i = (global.wongames * 100) / global.numgames;
        if (i < 20) "barely able to play Masta'mind without moving your lips
            while you read. ";
        else if (i < 50) "the Masta'mind equivalent of Vanilla Ice. ";
        else if (i < 75) "fairly decent at Masta'mind. ";
        else if (i < 90) "gettin' good at Masta'mind. ";
        else if (i < 95) "nearly able to read minds. At least, computer ones. ";
        else "someone with entirely too much free time on their hands. ";
    }
}

replace gameinit: function
{
    // put introductory text here
    
    version.sdesc;                // display the game's name and version number

    Me.location = startroom;                // move player to initial location
    startroom.lookAround(true);                    // show player where he is
    startroom.isseen = true;                  // note that we've seen the room
    setscore('Wins: '+cvtstr(global.wongames)+'  Attempts: '+
        cvtstr(global.numgames));
    randomize();
}

mastaInit: initialization
    preinit_phase = {
        local i, len;
        
        len = length(mastamind.slotList);
        for (i = 1; i <= len; i++)
            mastamind.authorList += (mastamind.slotList[i]).author;
    }
;

/*
 *   The terminate() function is called just before the game ends.  It
 *   generally displays a good-bye message.  The default version does
 *   nothing.  Note that this function is called only when the game is
 *   about to exit, NOT after dying, before a restart, or anywhere else.
 */
terminate: function
{
    if (global.wongames == 0)
"\bThank you for playing the demo version of \(Masta'mind\), a TextFire
release. ";
    else "Congratulations on puzzling out who wrote what. We ask that you
        \(not post this list to either of the newsgroups\) for some time,
        in order that the joke not be ruined for people prematurely. Thanks. ";
}

/*
 *   The pardon() function is called any time the player enters a blank
 *   line.  The function generally just prints a message ("Speak up" or
 *   some such).  This default version just says "I beg your pardon?"
 */
pardon: function
{
    "Say what? ";
}

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj  // use default definition from adv.t
;

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */
strObj: basicStrObj     // use default definition from adv.t
;

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.  The properties of this
 *   object that are particularly important to the objects and functions
 *   are defined here; if you replace this object, but keep other parts
 *   of this file, be sure to include the properties defined here.
 *
 *   Note that awakeTime is set to zero; if you wish the player to start
 *   out tired, just move it up around the sleepTime value (which specifies
 *   the interval between sleeping).  The same goes for lastMealTime; move
 *   it up to around eatTime if you want the player to start out hungry.
 *   With both of these values, the player only starts getting warnings
 *   when the elapsed time (awakeTime, lastMealTime) reaches the interval
 *   (sleepTime, eatTime); the player isn't actually required to eat or
 *   sleep until several warnings have been issued.  Look at the eatDaemon
 *   and sleepDaemon functions for details of the timing.
 */
global: object
    turnsofar = 0                        // no turns have transpired so far
    score = 0                            // no points have been accumulated yet
    maxscore = 100                       // maximum possible score
    verbose = true                       // we are currently in TERSE mode
    numgames = 0
    wongames = 0
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */
version: object
    sdesc = "\b\(Masta'mind\)\n
\nAn interactive adaptation of the popular game
\nA TextFire Production
\n(Type \"about\" for more information; \"instructions\" for help)
\nCopyright (c) 1998 by TextFire, Inc. All rights reserved.
\nDeveloped with TADS, the Text Adventure Development System.\b"
;

/*
 *   "Me" is the player's actor.  Pick up the default definition, basicMe,
 *   from "adv.t".
 */
Me: basicMe
    ldesc = "You look the same as always. "
;

/*
 *   darkTravel() is called whenever the player attempts to move from a dark
 *   location into another dark location.  By default, it just says "You
 *   stumble around in the dark," but it could certainly cast the player into
 *   the jaws of a grue, whatever that is...
 */
darkTravel: function
{
    "You stumble around in the dark, and don't get anywhere. ";
}

/*
 *   goToSleep - carries out the task of falling asleep.  We just display
 *   a message to this effect.
 */
goToSleep: function
{
    "***\bYou wake up some time later, feeling refreshed. ";
}

replace scoreStatus: function(points, turns)
{
}

startroom: room
    sdesc = "The Masta'mind Challenge Chamber"
    ldesc = {
        "Curving around you, the chamber is capped at the north by a metal
            railing. Large doors to the south open into sunshine, a dramatic
            change from the metal-tinged cool air of inside. A panel and large
            board hug the railing near its middle. All are overshadowed by
            a <<alcove.closed ? "curtained" : "deep">> alcove. East is a
            darkened booth. ";
        if (!alcove.closed) "\bDeep in the alcove, floating, a disembodied
            brain hovers. ";
    }
    north = { "The railing prevents you. "; return nil; }
    south = {
        "You walk through the exit and into the wide world beyond. Behind you,
            the Masta'mind doors slam shut.\b";
        die();
    }
    east = author_booth
    out = {return self.south;}
;

fake_doors: fixeditem
    isThem = true
    noun = 'door' 'doors'
    adjective = 'large'
    location = startroom
    sdesc = "large doors"
    ldesc = "They allow egress from the chamber. "
    verDoOpen(actor) = { "They are already open. "; }
    verDoClose(actor) = { "You push, but they do not close. "; }
    verDoEnter(actor) = {}
    doEnter(actor) = { startroom.south; }
;

fake_booth: fixeditem
    noun = 'booth'
    adjective = 'darkened' 'author'
    location = startroom
    sdesc = "darkened booth"
    ldesc = "The booth lies to the east. It is labeled \"Author Booth.\" "
    verDoRead(actor) = {}
    doRead(actor) = { self.ldesc; }
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(author_booth);
    }
;

sunshine: distantItem
    noun = 'light' 'sunshine'
    location = startroom
    sdesc = "sunshine"
    ldesc = "Warm, inviting. "
;

railing: fixeditem
    noun = 'railing' 'rail'
    adjective = 'metal'
    location = startroom
    sdesc = "metal railing"
    ldesc = "It curves near the north end of the chamber. In its middle, directly
        beneath the alcove, are a panel and a large board. "
    verDoClimb(actor) = {
        if (alcove.closed)
            "As soon as you begin to go over the railing, you receive a mild
                electrical shock. You let go, hands shaking. Five-foot high
                letters appear in front of you, glowing white: \"PLEASE
                REFRAIN.\" The letters then vanish. ";
        else "\(NO,\) says Masta'mind, and a gentle force pushes you off of
            the railing. ";
    }
    doSynonym('Climb') = 'Jump'
;

scoreboard: fixeditem
    noun = 'board' 'scoreboard'
    adjective = 'large' 'green'
    location = startroom
    sdesc = "large board"
    ldesc = {
        local i = global.numgames - global.wongames;
        
        "A green scoreboard, bolted onto the railing. Its top is feet above
            you. On it are two numbers: the leftmost, labeled \"Failures,\"
            is <<i>>; the rightmost, labeled \"Attempts,\" ";
        if (i == global.numgames) "also ";
        "is <<global.numgames>>. ";
    }
;

leftnum: fixeditem
    noun = 'number'
    adjective = 'left' 'leftmost' 'failures'
    plural = 'numbers'
    location = startroom
    sdesc = "leftmost number"
    ldesc = "The leftmost number, labeled \"Failures,\" is <<global.numgames -
        global.wongames>>. "
;

rightnum: fixeditem
    noun = 'number'
    adjective = 'right' 'rightmost' 'attempts'
    plural = 'numbers'
    location = startroom
    sdesc = "rightmost number"
    ldesc = "The rightmost number, labeled \"Attempts,\" is <<
        global.numgames>>. "
;

panel: fixeditem
    noun = 'panel'
    adjective = 'plastic' 'square' 'squarish' 'shiny' 'translucent'
    location = startroom
    sdesc = "panel"
    ldesc = "A squarish panel, attached to the railing at just over
        waist-height, made of shiny translucent plastic. Mounted on its surface
        is a tiny key, a tremendous number of slots and buttons, and a large
        white lever. "
    verDoOpen(actor) = { "You see no way of doing that. "; }
    doTurn -> tinykey
    doTurnon -> tinykey
    doTurnoff -> tinykey
;

tinykey: fixeditem
    ison = nil
    frozen = nil
    triedOnce = nil
    noun = 'key'
    adjective = 'tiny'
    location = startroom
    sdesc = "tiny key"
    ldesc = {
        "The tiny key is, from the looks of things, a permanent part of the
            panel. It has two positions to which it can be turned:\ on and
            off. Currently it points to <<self.ison ? "on" : "off">>. ";
        if (self.frozen)
            "Small burn marks mar its surface. ";
    }
    verDoTurnon(actor) = {
        if (self.ison) "The key is already turned to 'on.' ";
    }
    doTurnon(actor) = { self.doTurn(actor); }
    verDoTurnoff(actor) = {
        if (!self.ison) "The key is already turned to 'off.' ";
    }
    doTurnoff(actor) = { self.doTurn(actor); }
    verDoTurn(actor) = {}
    doTurn(actor) = {
        if (self.frozen) {
            "It has been spot-welded in place. You can no longer turn it. ";
            if (!self.triedOnce) {
                self.triedOnce = true;
                "From behind the curtain, a muffled Masta'mind says, \(NUH-UH.
                    YOU DIS THE BEST, YOU WAIT WITH THE REST. RUN ALONG NOW.\) ";
            }
            return;
        }
        "You turn the key until it points to '<<self.ison ? "off" : "on">>.' ";
        self.ison = !self.ison;
        if (self.ison) {
            if (global.numgames > 0)
                "The curtains part, once again revealing Masta'mind. \(BACK
                    FOR MORE?\)\ he asks. \(THEN LET THE GAME BEGIN.\) ";
            else {
                "The curtains slowly pull back from the alcove, revealing a
                    giant disembodied brain. From somewhere, faintly, you swear
                    you hear a short trumpet fanfare. The brain slowly turns,
                    orienting
                    so that its frontal lobe points directly at you.
                    \(WELCOME!\)\ he";
                makeNote('Granted, Masta\'mind has no definite sex. But to make
                    things easier for all concerned, it will be henceforth
                    referred to as a he.');
                " booms. \(WELCOME! I AM MASTA'MIND!\) He then whispers, \(the
                    demo version, anyway.\) He continues,
                    \(THE GAME HAS BEGUN. IF YOU ARE UNFAMILIAR WITH THE RULES,
                    YOU MAY TYPE instructions TO LEARN.\) ";
            }
            alcove.closed = nil;
            mastamind.moveInto(startroom);
            mastamind.startGame;
        }
        else {
            "The curtains swing closed over the alcove. ";
            if (mastamind.inGame) {
                "\(HEY! HEY! WE'RE IN THE MIDDLE OF A GAME!\) When you don't
                    answer, Masta'mind bellows, \(FINE. THEN NO MORE.\) A puff
                    of smoke curls from the small key. ";
                self.frozen = true;
            }
            else if (global.numgames - global.wongames > global.wongames)
                "\(NO DOUBT YOU WISH SOME TIME TO LICK YOUR WOUNDS,\) says
                    Masta'mind as he is covered. ";
            else "\(HAD YOU CONTINUED, NO DOUBT I WOULD HAVE BEGUN TO WIN,\)
                Masta'mind grumbles as he vanishes. ";
            alcove.closed = true;
            mastamind.moveInto(nil);
        }
    }
;

class game_slot: fixeditem
    title = ''
    noun = 'slot'
    location = startroom
    sdesc = "'<<self.title>>'\ slot"
    ldesc = {
        "A small slot, above which is written '<<self.title>>";
        if (!self.punctuated) ".";
        "'\ ";
        if (self.contents != [])
            " Sticking out of the slot is <<self.contents[1].adesc>>.";
        " A small white button is directly below the slot. ";
    }
    verIoPutIn(actor) = {
        if (self.contents != [])
            "There is already <<self.contents[1].adesc>> in the slot. ";
    }
    ioPutIn(actor, dobj) = {
        if (!dobj.isTile) {
            "\^<<dobj.thedesc>> won't fit in the slot. ";
            return;
        }
        "You slide <<dobj.thedesc>> into the slot. It fits perfectly. ";
        dobj.moveInto(self);
    }
;

fake_game_slots: fixeditem
    noun = 'slots'
    location = startroom
    sdesc = "slots"
    ldesc = {
        local i, len, slot;
        
        "There are ";
        if (mastamind.noFar) "seventeen";
        else "eighteen";
        " slots, each labeled with one of the TextFire demos. Below each slot
            is a white button. The demos are:";
        len = length(mastamind.slotList);
        for (i = 1; i <= len; i++) {
            slot = mastamind.slotList[i];
            "\n\t<<slot.title>>";
            if (slot.contents != [])
                " (which contains <<(slot.contents[1]).adesc>>)";
        }
/*Revenge of the Killer Surf Nazi Robot Babes from Hell\n\t
Bad Guys\n\t
An Exploration of Colour\n\t
Coma!\n\t
Flowers for Algernon\n\t
George\n\t
The Inanimator\n\t
Insomnia\n\t
Jack\'s Adventures\n\t
Will The Real Marjorie Hopkirk Please Stand Up?\n\t
Once\n\t
Operate!\n\t
Pumping!\n\t";
        if (!mastamind.noFar)
            "So Far\n\t";
"A Tenuous Hold\n\t
The U.S.\ Men\'s Hockey Team Olympic Challenge\n\t
Verb!\n\t
Zugzwang";*/
    }
;

babes_slot: game_slot
    title = 'Revenge of the Killer Surf Nazi Robot Babes from Hell'
    adjective = 'revenge' 'killer' 'surf' 'nazi' 'robot' 'babes' 'from' 'hell'
    author = 6
;
badguys_slot: game_slot
    title = 'Bad Guys'
    adjective = 'bad' 'guys'
    author = 7
;
colours_slot: game_slot
    title = 'An Exploration of Colour'
    adjective = 'exploration' 'colour' 'color' 'colours' 'colors'
    author = 2
;
coma_slot: game_slot
    title = 'Coma!'
    punctuated = true
    adjective = 'coma' 'coma!'
    author = 4
;
flowers_slot: game_slot
    title = 'Flowers for Algernon'
    adjective = 'flowers' 'algernon'
    author = 3
;
george_slot: game_slot
    title = 'George'
    adjective = 'george'
    author = 12
;
inanimate_slot: game_slot
    title = 'The Inanimator'
    adjective = 'inanimate' 'inanimator'
    author = 1
;
insomnia_slot: game_slot
    title = 'Insomnia'
    adjective = 'insomnia'
    author = 14
;
jack_slot: game_slot
    title = 'Jack\'s Adventures'
    adjective = 'jack' 'jacks' 'jack\'s' 'adventures'
    author = 13
;
marjorie_slot: game_slot
    title = 'Will The Real Marjorie Hopkirk Please Stand Up?'
    punctuated = true
    adjective = 'will' 'real' 'marjorie' 'hopkirk' 'please' 'stand' 'up' 'up?'
    author = 2
;
once_slot: game_slot
    title = 'Once'
    adjective = 'once'
    author = 9
;
operate_slot: game_slot
    title = 'Operate!'
    punctuated = true
    adjective = 'operate' 'operate!'
    author = 12
;
pumping_slot: game_slot
    title = 'Pumping!'
    punctuated = true
    adjective = 'pumping' 'pumping!'
    author = 8
;
so_far_slot: game_slot
    title = 'So Far'
    adjective = 'so' 'far'
    author = 0
;
tenuous_slot: game_slot
    title = 'A Tenuous Hold'
    adjective = 'tenuous' 'hold'
    author = 8
;
hockey_slot: game_slot
    title = 'The U.S.\ Men\'s Hockey Team Olympic Challenge'
    adjective = 'hockey' 'team' 'olympic' 'challenge'
    author = 10
;
verb_slot: game_slot
    title = 'Verb!'
    punctuated = true
    adjective = 'verb' 'verb!'
    author = 5
;
zugzwang_slot: game_slot
    title = 'Zugzwang'
    adjective = 'zugzwang'
    author = 11
;

class small_button: fixeditem
    title = ''
    noun = 'button'
    adjective = 'small' 'white' 'foam'
    location = startroom
    sdesc = "'<<self.title>>\' button"
    ldesc = "The button is small and white, and evidently made of foam. "
    verDoPush(actor) = {}
    doPush(actor) = {
        "The button deforms slightly under pressure, then springs back.
            White letters spell out the
            following:\b-----\n<<self.printInfo>>\n-----";
    }
    printInfo = {}
;

fake_buttons: fixeditem
    noun = 'buttons'
    location = startroom
    sdesc = "buttons"
    ldesc = {
        "There are ";
        if (mastamind.noFar) "seventeen";
        else "eighteen";
        " white buttons, evidently made of foam. ";
    }
;

babes_button: small_button
    title = 'Revenge of the Killer Surf Nazi Robot Babes from Hell'
    adjective = 'revenge' 'killer' 'surf' 'nazi' 'robot' 'babes' 'from' 'hell'
    author = 6
    printInfo = {
"\(Title\): Revenge of the Killer Surf Nazi Robot Babes from Hell\n
(a way cool surfin' dude saves the world)\b

\(Author\): Alex Crowley hails originally from the small town of Mooroopna in
Victoria,
Australia. He completed a degree in Cinema Studies at Latrobe University,
Melbourne in 1996 and is presently pursuing a Master of Arts at Stanford. It's
a long way from a town of only 3000 people, but Alex keeps himself sane by
playing
cricket in the Northern California League and writing games with his net
friends
at TextFire, having first met Riley Hawkins on IRC (neither of them will say
which channel) in 1995.\b

\(Opening\): Hangin' on the beach for six months has been an awesome way to forget
your brush
with those robot babes back east. Best hang out another six to be real sure
tho...\b

Gary's Surf Shack\n
Boards, wax, wetsuits, it's all here at the Shack. Best hangout before hittin'
the waves, too.\b

West to the beach, when your buds arrive. Seems you're the first one here
today.\b

Goofy Foot Gary (he owns the place) is here, checkin' the stuff for sale. ";
    }
;
badguys_button: small_button
    title = 'Bad Guys'
    adjective = 'bad' 'guys'
    author = 7
    printInfo = {
"\(Title\): Bad Guys\n
An Interactive Evil\b

\(Author\): Ed Dallas was once a successful programmer at Microsoft. However,
he was laid
off in late 1996.  He made a few shareware action games for Win95, but his
reputation was further tarnished after one of them had accidentally been
infected by a virus when his supposedly \"internet-savvy\" friend uploaded
them to
an archive.  This earned him the nickname of \"Virus Games\".\b

He was saved from all this by his neighbor from his home town of Piedmont,
California: Deborah Keyes, who introduced him to IF and TextFire.  He learned
Inform in a matter of weeks and started on Bad Guys.\b

He does not wish to make a gratuitous joke about speaking in the third
person.\b

\(Opening\): Life is easy.\b

Wake up. Eat breakfast. Conquer weak countries. Torture prisoners. Go back
to sleep.\b

Today seems more difficult. Sure, you've fought off random elves in your palace
before.  Sure, you've adapted artifacts to your evil doings before.  Sure,
you've won wars before.  You've even put up with goblin strikes.\b

But never all in the same day.\b

And at this rate, you'll never find time to become immortal.\b

Watch Tower\n
From here, nearly a mile up, you can see your whole domain.  Through the large
glass window in front of you, a battle is fighting for dominance of a nearby
\"peaceful\" kingdom.  Your laboratory, where you do all of your magical evil,
is directly below you.  Your Magic Labyrinth Manipulator is affixed to the
wall.";
    }
;
colours_button: small_button
    title = 'An Exploration of Colour'
    adjective = 'exploration' 'colour' 'color' 'colours' 'colors'
    author = 2
    printInfo = {
"\(Title\): An Exploration of Colour\n
An Interactive Journey\b

\(Author\): Mal Jackson was born and raised in the county of Essex in the
United Kingdom,
and attained a BSc (Hons) in Software Engineering at Warwick University.
Deciding that computing was more enjoyable as a hobby than an occupation, Mal
relocated to Bournemouth and accepted a job as barman at a local nightclub,
where he continues to work to this day. He sees writing interactive fiction as
his 'part time day job', and argues that it doesn't really count as real work
anyway.\b

\(Opening\): Blues, reds, greens, yellows; so many different colours, and so
difficult to find.\b

=======\n
| Sky |\n
=======\n
A blanket of grey-white looming over you like a great depression, hiding the
true beauty of above.";
    }
;
coma_button: small_button
    title = 'Coma!'
    adjective = 'coma' 'coma!'
    author = 4
    printInfo = {
"\(Title\): Coma!\n
An interactive action thriller\b

\(Author\): Rachael Croft was a native of Perth, Australia, until moving to 
London in 1995 to pursue a rewarding part-time career in travel and food 
service. After backpacking and waitressing her way through some of the 
wilder parts of Europe and the Americas, she stumbled into Piedmont, 
California, where she met Deborah Keyes and Ed Dallas. Two years later, back 
in Perth, the friendship returned to haunt her when Keyes contacted her on 
IRC and the TextFire group was born.\b

Strangely enough - at least to her American friends, who _always_ 
ask - Rachael has never yet met fellow Australian Alex Crowley face to face. 
She has also never personally been in a coma, but has often been mistaken 
for dead, especially at 7:30 in the morning. Her driving ambition is to 
write an IF novel \"that would be as good as the Zork Trilogy is bad.\"\b

\(Opening\): Whispers, swirling at the edge of life. Whispers, deepening, 
widening into voices as the holes fade from your memory. Voices - familiar 
voices. You reach for a reply, but the words are lost. Somewhere. A gap. 
Between. Thought, action. You struggle to wake up, but it's almost too hard...\b

Darkness\n
Everything is murky, shrouded with distant echoes. You seem to be floating 
in clouds of darkness that swim through your mind. ";
    }
;
flowers_button: small_button
    title = 'Flowers for Algernon'
    adjective = 'flowers' 'algernon'
    author = 3
    printInfo = {
"\(Title\): Flowers for Algernon\n
An interactive adaptation of Daniel Keyes's novel\b

\(Author\): The niece of novelist Daniel Keyes -- and no relation to one-time
presidential
candidate Alan Keyes -- Deborah Keyes attended Carondelet High School in
Concord, California and Mills College in Oakland, where she majored in
education
and wrote for the Mills College Weekly.  She is currently a student teacher at
Piedmont High School in Piedmont, California, where she does *not* do any bird
calls.\b

\(Opening\): Yu ar Charly Gordon. Yu werk at a bakury in the day and in the
nite yu go to
miss kinnians clas to lern to reed and rite and do al kinds of things that
miss kinnian ses that ar very impresiv for sumwon with an I Q of 68. This 
morning yor going to hav an opurashun to mak yu smart. But furst yu hav to 
get to the hospitul!\b

Yor rume (on yor bed)\n
Heer yu ar in yor rume. The dor is to the sowth.\b

Yu can see a blakbord here. ";
    }
;
george_button: small_button
    title = 'George'
    adjective = 'george'
    author = 12
    printInfo = {
"\(Title\): George\b

\(Author\): After graduating with a degree in journalism from an obscure 
university in North Dakota, Shea Davis \"settled\" in the midwest, where 
he briefly made a living writing sentiments for a well-known greeting 
card company. Once disgruntlement set in, Shea unsettled himself, quit 
the wordbyte business, and relocated to a shinier, more suitable 
environment:\ Piedmont, California. There, once he passed through 
false starts in film and newspaper, he co-founded TextFire, Inc. with 
a group of like-minded, equally unsettling individuals. And now he does 
what he should have done all along:\ write interactive fiction full-time.\b

\(Opening\): George is a friend of yours. A good friend.\b

Zoo Entrance\n\t
   The crowd follows the littered walkway as it splits apart and continues
through the surrounding eucalyptus groves: to the primates, the cats, the
pachyderms, and the aviary.\b

Bright animal tracks lead westward.\n\t
   George is here.\n\t
   The swarming crowd presses in against you.
";
    }
;
inanimate_button: small_button
    title = 'The Inanimator'
    adjective = 'inanimate' 'inanimator'
    author = 1
    printInfo = {
"\(Title\): The Inanimator\n
A new dimension of human understanding\b

\(Author\): Thomas \"Flip\" Winkler, TextFire accountant and junior 
programmer, prepped at Hotchkiss and received his Harvard MBA in 1996. He 
was introduced to the world of interactive fiction at age 15 when his 
governess gave him a copy of Trinity for his birthday. The game touched not 
only his mind, but also his heart, and Winkler went to several 
anti-nuclear demonstrations on the Hotchkiss campus. Now 25 and still very 
much against nuclear war, Winkler plans to retire from the daily IF grind 
in five years, when he comes into his money.\b

\(Opening\): \"Woo! Spare!\" you yell, nailing a tricky 8-10 split. \"Drinks 
are on me!\" A couple of high-fives, and you head over to the vending 
machines. Funny...they seem to have replaced the Coke machine with some sort 
of...gambling device? With no coin slot?\b

Jim's Big 27 Lanes\n
\tYou've been bowling here since '89, but you never noticed this odd 
machine before. Behind you, the reassuring sound of toppling pins fills 
the air. ";
    }
;
insomnia_button: small_button
    title = 'Insomnia'
    adjective = 'insomnia'
    author = 14
    printInfo = {
"\(Title\): Insomnia\n
The Up-All-Night Conspiracy\b

\(Author\): August Pelter is a grad student at a major midwestern 
university, studying the effects of radiation on mutogenic biophages. 
However, his real loves are his wife and daughter (ages 28 and 1, 
respectively) and games. Though this is his first work of IF, he's been a 
huge fan of the genre for over ten years now.\b

\(Opening\): Two-thirty in the morning, and yet not a wink of sleep. You feel 
as if the whole world is against your sleeping:\ your nagging project 
tomorrow, your stomach in knots from stress, and not to forget, the 
neighbors having their noisome sexual rendez-vous next door.\n
    \tIt's as if you were not destined to sleep tonight, and perhaps bigger 
things are at hand.\b

Your Bedroom, on your bed\n\t
   Your bedroom is a modest room in a more modest dwelling, probably not the 
best efficiency apartment that one could rent for $750 per month, but not 
the worst, either. Moonlight and streetlight shine through the window on 
your autographed and framed Bill Gates glossy.\n\t
   Out the only exit to the south would be what the floor plan called the 
\"Family Room,\" assuming improperly you have some sort of extended 
relationship, and if you did, they would fit into that room.\b

On the nearby night stand you see an analog alarm clock (providing light) and 
a Bill the Cat tote bag.\n\t
   You can still hear your next-door neighbors through the cardboard-thin 
apartment walls, likely performing acts that would be deemed a misdemeanor 
in some of the more conservative states. ";
    }
;
jack_button: small_button
    title = 'Jack\'s Adventures'
    adjective = 'jack' 'jacks' 'jack\'s' 'adventures'
    author = 13
    printInfo = {
"\(Title\): Jack's Adventures\n
or\n
On The Run in Fairyland with a Golden Goose and a Magic Guitar\n
\b

\(Author\): The demo version of \(Jack's Adventures\) was written by Ben
Scagels, a 21-year-old marine biology major. He codes TADS with his
right hand, Inform with his left, and has unmatched socks and a fondness
for platypi. Besides wanting to write great text adventures, his ambition
is to form a band, \"Ben and the Dancing Platypi\", but has found no
drummer who is both competent and willing to be called a platypus.\b

\(Opening\): \"I'm all out of magic lamps, sorry.\"\n
\"Magic rings?\"\n
\"Er..\ none of those, either. I've got one of those magic swords of slaying,
barely used, except by a little old lady who used to take it out of its 
scabbard once a week to prune the begonias...\"\n
\"Great! I'll take it!\"\n
\"..But it's going to cost you two pigs, a sheep, a goat, three chickens and
a donkey.\"\n
\"Oh. All I have is this cow.\"\n
\"A cow, huh? I'm afraid there's rather a glut of cows on the market right
now. Well, I tell you what. I got these magic beans.\"\n
\"Beans? Like magic kidney beans?\"\n
\"No, no. The kind you plant.\"\n
\"Gee, I dunno. My mom never said anything about magic vegetables.\"\n
\"Look, I tell you what. Give me the cow, and I'll throw in a shovel *and*
a magic watering can!\"\n
\"I'll take it!\"\b

Garden\n\t
   Well, as gardens go, it's not much. In fact, what with the drought and
the unicorns getting into the carrots, it's not so much a garden as a bit
of dirt with slightly less rocks than usual. The cottage is back west. ";
    }
;
marjorie_button: small_button
    title = 'Will The Real Marjorie Hopkirk Please Stand Up?'
    adjective = 'will' 'real' 'marjorie' 'hopkirk' 'please' 'stand' 'up' 'up?'
    author = 2
    printInfo = {
"\(Title\): Will The Real Marjorie Hopkirk Please Stand Up?\n
An Assassin's Nightmare!\b

\(Author\): Steve's parents tell him that he was the result of a power cut 
in 1970, and his
conception can certainly be blamed for the shotgun wedding that soon followed.
Growing into a deceitful and cruel young man, it seemed only logical that he
would end up joining the police, and join he did, but only so that he could
send
photos of himself in uniform to desperate young women over the internet, and
make them sweat with anticipation before mysteriously vanishing from their
lives. For Steve is evil, and delights in hurting people.\b

This is not Steve Parsons, however. For he is someone quite different.\b

\(Opening\): Sometimes your job goes like a dream, and everything is
hunkydory. At other times, things fall apart miserably and you're left 
with a horrible mess to sort out. Like today.\n\t
  It seemed easy enough; some fat balding guy wanted his missus bumped off 
so he could claim the insurance and spend it all on the blonde nubile
young thing he's been shagging on the quiet. Fair enough - it's good 
money. Trouble is, when you
followed your target Marjorie Hopkirk into this conference hall, your eyes 
were
out on stalks. Seems old Marj isn't as stupid as she looks - she's gone and
cloned herself 99 times to reduce the chances of getting bumped off, and 
now you can't tell the real one among the hundred Marjories in the room. 
There's nothing
else for it - you're going to have to kill them all to make sure the real Mrs
Hopkirk dies!\b

Conference Hall\n\t
  The main area of this grand hall is covered with rows upon rows of wooden
seats, sitting on which are exactly one hundred Marjorie Hopkirks. They watch
you attentively but without fear, although with all conferences a few at the
back simply will not shut up no matter how serious the occasion. The main door
out is firmly closed, but a small passage leads backstage to the south. ";
    }
;
once_button: small_button
    title = 'Once'
    adjective = 'once'
    author = 9
    printInfo = {
"\(Title\): Once:\ a fable for the lost\b

\(Author\): Robin Andrews is a former journalist whose book of children's 
stories, \(Rockets and Shooting Stars\), will be published next year. He 
lives on the Eastern Shore of Maryland with his wife, Sarah, and their two 
daughters, Lauren and Leslie.\b

\(Opening\): You were once a little boy. Your eyes once were green, and your 
words were like wildflowers.\b

In the Garden\b

You stand in the middle of Grandmother's garden. You still have many weeds left
to find. The woods are so far away.\b

Grandmother always insists on an immaculate garden. \"The daises will not allow
anyone to crowd their soil. They forgot how to live with the unknowable.\" ";
    }
;
operate_button: small_button
    title = 'Operate!'
    adjective = 'operate' 'operate!'
    author = 12
    printInfo = {
"\(Title\): \(Operate!\)\n
An interactive adaptation of the popular parlor game\b

\(Author\): After graduating with a degree in journalism from an obscure 
university in North Dakota, Shea Davis \"settled\" in the midwest, where 
he briefly made a living writing sentiments for a well-known greeting 
card company. Once disgruntlement set in, Shea unsettled himself, quit 
the wordbyte business, and relocated to a shinier, more suitable 
environment:\ Piedmont, California. There, once he passed through 
false starts in film and newspaper, he co-founded TextFire, Inc. with 
a group of like-minded, equally unsettling individuals. And now he does 
what he should have done all along:\ write interactive fiction full-time.\b

\(Opening\): As Chief Surgeon, the honor is yours:\ a public display of 
skill meant to inform and educate.\b

An intercom clicks on. \"PROCEED.\"\b

Operating Room\n\t
   Billowing lights bear down upon the clean, white body and supporting 
table. In a similar pattern of sterility, a single blank wall forms a 
perfect circle around the arena. Above and beyond, through sheets of 
brightness, a full gallery of onlookers peers in from behind a plastic 
shield.\b

There is a bone vat here. To the side sits your instrument tray, neatly
centered on a short push-cart.\n\t
   Sitting on the instrument tray, bone tongs and a scalpel can be seen. ";
    }
;
pumping_button: small_button
    title = 'Pumping!'
    adjective = 'pumping' 'pumping!'
    author = 8
    printInfo = {
"\(Title\): Pumping!\b

\(Author\): After receiving a degree in library science from Texas 
Woman's University, Riley Hawkins was directionless. Eventually, like 
so many other Okies, Riley went west in search of opportunity, finally 
stopping in Piedmont, CA. Unfortunately, Piedmont has a noticeable dearth 
of libraries; fortunately, one of the numerous branches of the Oakland 
Public Library system had a job opening.\b

A few months later, Riley met Shea Davis when Shea was doing research for 
a game. In an odd series of coincidences, Deborah Keyes attended the 
same PFLAG meeting Riley did, and Riley and Flip Winkler crossed paths 
at a show at Slim's. Though still working at the library, Riley is now 
spending more and more time writing interactive fiction.\b

\(Opening\): You've never minded the stretching, the pulling, the 
pumping. It's all been part of a day's work. And a night's work, come to 
think of it.\b

But now...now you're under more and more stress. Tension's building, 
and something's gotta give.\b

Doesn't matter. You've got a job to do, and you're going to keep doing 
it, come plaque or high pressure. After all, you are Joe's heart.\b

Pericardium\n\t
   You are floating in your fibrous home, attached to Joe's circulatory 
system. Occasional vibrations send you jiggling as Joe shifts about. ";
    }
;
so_far_button: small_button
    title = 'So Far'
    adjective = 'so' 'far'
    author = 0
    printInfo = {
"\(Title\): SO FAR\n
An Interactive Catharsis\b

\(Author\): Andrew Plotkin is a graduate of Carnegie Mellon University and
the House of Everything. At last count, he had 71 shelf-feet of science
fiction and fantasy paperbacks. He is not and has never been a sentinent
rutabaga, or indeed a rutabaga of any sort.\b

\(Opening\): Hot, foul, and dark. How did indoor theater become so
fashionable? Well enough
in spring rain or winter, but not in the thick, dead afternoon of high summer.
And though Rito and Imita looks very fine, shining with electric moonslight in
the enclosed gloom, you're much more aware of being crammed in neck-by-neck
with your sweaty fellow citizens.\b

Damn the crowd, in truth: your mood was hot, foul, and dark when you sat down.
Aessa was supposed to meet you here. She's made excuses before, and you don't
think about what it might mean. Try not to think, rather. Just watch the story.
One of your favorites. But it's miserably hot, and you just aren't caught up in
the play...\b

Lower Theater (on the bench)\n
If nothing else, you have a good seat. The stage is only a few rows to the
north. People are jammed on benches in every direction; you think the aisle is
somewhere to the west.\b

The play is into the final act. Rito has finally found out about Imita's
affair, and he stalks the stage, whipping voice and hands about himself. A
footfall behind him; he turns, and sees Imita. She backs a step at the force
of his gaze. ";
    }
;
tenuous_button: small_button
    title = 'A Tenuous Hold'
    adjective = 'tenuous' 'hold'
    author = 8
    printInfo = {
"\(Title\): A Tenuous Hold\b

\(Author\): After receiving a degree in library science from Texas 
Woman's University, Riley Hawkins was directionless. Eventually, like 
so many other Okies, Riley went west in search of opportunity, finally 
stopping in Piedmont, CA. Unfortunately, Piedmont has a noticeable dearth 
of libraries; fortunately, one of the numerous branches of the Oakland 
Public Library system had a job opening.\b

A few months later, Riley met Shea Davis when Shea was doing research for 
a game. In an odd series of coincidences, Deborah Keyes attended the 
same PFLAG meeting Riley did, and Riley and Flip Winkler crossed paths 
at a show at Slim's. Though still working at the library, Riley is now 
spending more and more time writing interactive fiction.\b

\(Opening\): An endless river of steel and plastic; the red glow of
taillights; a cacophany of growls and bleating honks.\b

You look down at your hands, unsurprised to find that you have been
massaging the steering wheel, molding and kneading it. The veins on the
back of your hands stand out, pulsing with the uneven flow of your blood.\b

God, you hate traffic.\b

Inside Your Car\n\t
   The inside of your car is an all-too-familiar sight, thanks in part to
the hours you have spent travelling to and from work. Its cracked dash and
malformed steering wheel glow dully in the evening light. ";
    }
;
hockey_button: small_button
    title = 'The U.S.\ Men\'s Hockey Team Olympic Challenge'
    adjective = 'hockey' 'team' 'olympic' 'challenge'
    author = 10
    printInfo = {
"\(Title\): The U.S. Men's Hockey Team Olympic Challenge!\n
An Interactive Trashing\b

\(Author\): Shishi M. Hayashi has been fascinated with interactive
fiction since the 5th grade, when he accidentally melted his brother's
Magick Stone of Dreams from Infocom's \"Wishbringer.\" After using the
bubble gum cards in \"Spellbreaker\" as coasters and the flying saucer
package of \"Starcross\" as a frisbee, Shishi learned there were actual
computer games involved.\b

After graduating from a famous California university in 1993 (whose name
he presently forgets), Shishi dabbled in various pursuits, the most
successful of which was a brief career starring in karaoke videos. You
may remember him as a clown in \"Send in the Clowns\" and the pie in
\"American Pie.\"\b

Shishi met Shea Davis on the set of the video for \"Red Rubber Ball,\" in
which Shishi played the ball. The two found a common interest in
interactive fiction. This friendship would lead to Shishi's invitation to
join TextFire.\b

Shishi thinks TADS is an improvement over BASIC, and likes to spend his
spare time playing with \"Uni-chan.\" His blood type is O and his sign is
Aquarius.\b

\(Opening\): Dammit! Just because of some crummy Eastern European team,
you won't get to be on a fricking Wheaties box! But you'll show them!
You still have a few hours to kill in your Olympic Village apartment, and
baby, it's gonna to get trashed good, because this time, violence is
the answer to this one!\b

Olympic Village Apartment\n\t
   This is your apartment in the Olympic village. It's pretty simple in
design, surrounded by the usual four walls (north, south, east, west) and
sporting a door in the north wall.\n\t
   There is a bed here, which is sitting in the corner. Next to it is
a dresser, which is made of mahogany.\n\t
   You see a fire extinguisher, a reading table, a pine chair, a delicate
vase, a television, and a hockey stick here. ";
    }
;
verb_button: small_button
    title = 'Verb!'
    adjective = 'verb' 'verb!'
    author = 5
    printInfo = {
"\(Title\): \(Verb!\)\b

\(Author\): D.L.\ Schneck was born in 1972 on the streets of Queens,
N.Y., where he won acclaim as Citywide Stickball Champion, and
later worked for the city organizing summer youth leagues. His first
love, though, was computer games, and he jumped at the chance to relocate
to California and join TextFire's team of programmers. He now lives near
the Piedmont-Berkeley border with his childhood sweetheart, Dee Hahn.\b

\(Opening\): \(This Place\)\n\t
   A roomful of furniture. Papers scattered around the room. A portable
TV set, left on. Doors to the left and right. Stairs ahead. ";
    }
;
zugzwang_button: small_button
    title = 'Zugzwang'
    adjective = 'zugzwang'
    author = 11
    printInfo = {
"\(Title\): Zugzwang\n
The Interactive Life of a Chess Piece\b

\(Author\): Alfred Timpson was born in Dundee, Scotland, but spent most of his
childhood in Piedmont, California. He later moved back to Britain, got a
degree in geography from Durham University, and is now living in Edinburgh,
working as a freelance writer. A recent high school reunion brought him back
into contact with the founders of TextFire. Apart from computer games, his
hobbies include chess, ornithology, hang gliding and trainspotting.\b

\(Opening\): It has been a fierce battle, full of bold moves, daring
attacks, surprising gambits and heroic sacrifices. Throughout this, you've
remained at your starting point, patiently waiting for orders. But now that
the metaphorical dust has settled and most pieces have been exchanged, it
finally looks as if it is your turn. The White King and one of your Pawn
colleagues have managed to nail the Black King with his back to the edge
of the board, but they can't finish him off without help.\b

The White King calls you to action. It is time to make your move.\b

Chessboard (on f2)\n
The chessboard is almost empty, now that most pieces have been exchanged. Its
surface is perfectly smooth, its 64 squares alternatingly deep black and pearly
white. ";
    }
;

large_white_lever: fixeditem
    noun = 'lever'
    adjective = 'large' 'white'
    location = startroom
    sdesc = "white lever"
    ldesc = "A metal stick jutting from the surface of the panel. "
    verDoPush(actor) = { "This particular lever can only be pulled. "; }
    verDoPull(actor) = {
        if (alcove.closed)
            "It won't move. ";
    }
    doPull(actor) = {
        local i, correct, guess;
        
        "You yank on the lever. ";
        if (!mastamind.inGame) {
            "\(AND SO IT BEGINS,\) Masta'mind says. ";
            mastamind.startGame;
            return;
        }
        mastamind.checkGuesses;
    }
;

alcove: distantItem
    closed = true
    noun = 'alcove'
    adjective = 'curtained' 'deep'
    location = startroom
    sdesc = {
        if (self.closed) "curtained";
        else "deep";
        "alcove";
    }
    ldesc = {
        if (self.closed) "Deep purple curtains mask its contents. ";
        else "Its depths contain a massive disembodied brain. ";
    }
    verDoLookin(actor) = {
        if (self.closed) "You can't see past the curtains. ";
    }
    doLookin(actor) = {
        "Masta'mind's brain fills the alcove. ";
    }
    verDoOpen(actor) = { "The curtains are too far above you. "; }
;

curtains: distantItem
    isThem = true
    noun = 'curtains'
    adjective = 'purple' 'satin' 'satiny'
    location = startroom
    sdesc = "purple curtains"
    ldesc = {
        if (alcove.closed)
            "Satiny purple curtains hang in front of the alcove, covering it. ";
        else "Satiny purple curtains are drawn back to either side of the 
            alcove. ";
    }
    verDoOpen(actor) = { "The curtains are too far above you. "; }
;

mastamind: distantItem
    slotList = [ babes_slot, badguys_slot, colours_slot, coma_slot,
        flowers_slot, george_slot, inanimate_slot, insomnia_slot, jack_slot,
        marjorie_slot, once_slot, operate_slot, pumping_slot,
        so_far_slot, tenuous_slot, hockey_slot, verb_slot, zugzwang_slot ]
    authorList = []
    guesses = 0
    firstGuess = 0
    inGame = nil
    noFar = nil            // True if I've removed the _So Far_ entry
    noTile = nil
    blankTile = nil
    noun = 'brain' 'masta\'mind' 'mastamind'
    adjective = 'disembodied' 'huge' 'massive'
    sdesc = "disembodied brain"
    ldesc = "Big. No, huge. The body to which this brain belongs must be
        nearly fifty feet tall. Its gray surface is highly wrinkled, and the
        odd nerve dangles behind it. "
    startGame = {
        self.inGame = true;
        self.guesses = 0;
    }
    endGame = {
        self.color = 0;
        self.inGame = nil;
    }
    checkGuesses = {
        local i, alist = self.authorList, tlist = [],
            len, slot, tile,
            mentionedBlank = nil, mentionedNoTile = nil, correct = 0,
            misplaced = 0, guessesLeft;
        
        if (!self.noFar && so_far_slot.contents != []) {
            tile = so_far_slot.contents[1];
            if (tile.author == so_far_slot.author) {
                self.removeSoFar;
            }
        }
        len = length(self.slotList);
        for (i = 1; i <= len; i++) {
            slot = self.slotList[i];
            if (slot.contents == []) {
                if (!mentionedNoTile) {
                    mentionedNoTile = true;
                    if (self.noTile)
                        "Masta'mind sighs heavily. \(OBVIOUSLY I AM DEALING
                            WITH A ROCKET SCIENTIST. YOU HAVE TO PUT TILES
                            WITH YOUR GUESSES IN all THE SLOTS BEFORE YOU
                            HAVE A CHANCE OF WINNING.\)\b";
                    else {
                        self.noTile = true;
                        "\(A MISSING TILE?\) chortles Masta'mind. \(THAT
                            WILL IMPACT YOUR CHANCES OF WINNING.\)\b";
                    }
                }
                continue;
            }
            tile = slot.contents[1];
            if (tile.isBlank) {
                if (!mentionedBlank) {
                    mentionedBlank = true;
                    if (self.blankTile)
                        "You hear what sounds surprisingly like Masta'mind
                            grinding his molars together, if he had molars.
                            \(THIS IS NOT A DIFFICULT POINT TO MASTER (NO
                            PUN INTENDED):\ PUT EACH BLANK TILE IN AN AUTHOR'S
                            SLOT TO MARK IT WITH THAT AUTHOR'S NAME, THEN PUT
                            THAT TILE IN ONE OF THE DEMO SLOTS.\)\b";
                    else {
                        self.blankTile = true;
                        "\(YOU HAVE A BETTER SHOT AT WINNING IF YOU MARK THE
                            BLANK TILES WITH AUTHORS' NAMES.\)\b";
                    }
                }
                continue;
            }
            if (tile.author == slot.author) {
                correct++;
                alist -= tile.author;
            }
            else tlist += tile;
        }
        if (correct == len) {
            "\(YOU ARE CORRECT, SIR.\) Masta'mind frowns. \(LUCK, NO DOUBT.\) 
                The numbers on the large board flicker, then change. \(YOU
                HAVE NOW MATCHED ALL AUTHORS WITH THEIR DEMOS. IF YOU WOULD
                LIKE TO KNOW MORE ABOUT TEXTFIRE, PLEASE TYPE textfire.
                SHOULD YOU BE STRANGE ENOUGH TO WISH TO PLAY ONCE MORE, PULL
                THE LEVER.\) ";
            incGames(true);
            mastamind.endGame;
            return;
        }
        len = length(tlist);
        for (i = 1; i <= len; i++) {
            if (find(alist, tlist[i].author) != nil) {
                misplaced++;
                alist -= tlist[i].author;
            }
        }
        "\(YOU HAVE <<correct>> AUTHOR<<correct != 1 ? "S" : "">> PLACED
            CORRECTLY AND <<misplaced>> AUTHOR<<misplaced != 1 ? "S" :
            "">> CORRECT BUT IN THE WRONG SLOT<<misplaced != 1 ? "S" :
            "">>\). ";
        mastamind.guesses++;
        guessesLeft = 10 - mastamind.guesses;
        if (guessesLeft != 0) {
            "Before the echoes from his previous announcement can die,
                Masta'mind is talking again. \(YOU HAVE <<
                guessesLeft>> MORE GUESS<<guessesLeft > 1 ? "ES" : "">>.\) ";
            return;
        }
        "He laughs, louder and louder. \(YOU HAVE LOST. The rightmost number
            on the board flickers, then changes. \(SHOULD YOU CARE TO LOSE
            ONCE MORE, PULL THE LEVER AGAIN FOR ANOTHER GAME.\) ";
        incGames(nil);
        mastamind.endGame;
    }
    removeSoFar = {
        "Masta'mind pauses. \(WHOOPS!\)\ he says. \(HOWEVER DID THAT GET ON
            THE LIST?\) With a negligent wave of his spinal cord, he makes
            the 'So Far' slot vanish. \(MUCH BETTER.\)\b";
        self.authorList -= 0;
        self.slotList -= so_far_slot;
        self.noFar = true;
        so_far_slot.moveInto(nil);
        so_far_button.moveInto(nil);
    }
;

mastamind_detector: detector
    mentionedOnce = nil
    location = startroom
    undoMove(pre, post) = {
        if (!pre || !post || !mastamind.inGame || self.mentionedOnce)
            return;
        self.mentionedOnce = true;
        "\bYou hear Masta'mind muttering to himself:\ \(...SHOULDA THOUGHT
            TO HANDLE undo...\) ";
    }
;

author_booth: room
    sdesc = "Author Booth"
    ldesc = "A claustrophobic booth, made more so by the rows of slots which
        line one wall. A wicker basket huddles in one corner. "
    west = startroom
    out = startroom
;

class author_slot: fixeditem
    contentsVisible = nil
    contentsReachable = nil
    isAuthorSlot = true
    title = ''
    noun = 'slot'
    location = author_booth
    sdesc = "<<self.title>> slot"
    ldesc = "A small slot, above which is written '<<self.title>>.' "
    verIoPutIn(actor) = {}
    ioPutIn(actor, dobj) = {
        if (!dobj.isTile) {
            "\^<<dobj.thedesc>> won't fit in the slot. ";
            return;
        }
        "You slide the tile into the slot, until it is sticking out like a
            blackened tongue. ";
        if (self.contents == []) {
            "A tiny flag pops out from above the slot. It unfurls, allowing
                you to read it:\ \"No more than four of the same author
                allowed.\" The flag then re-furls and vanishes. You remove
                the tile. ";
            return;
        }
        "The wall shudders. ";
        if (!dobj.isBlank) {
            "Squeaky erasing noises float from the slot for a moment. You then";
            dobj.moveInto(dobj.home);
        }
        else {
            "You";
            dobj.moveInto(nil);
        }
        " hear a loud KA-CHUNG!\ and the tile shivers. When you withdraw
            it, you see that the words \"<<self.title>>\" have been stamped
            on the tile. ";
        (self.contents[1]).moveInto(actor);
    }
;

fake_author_slots: fixeditem
    noun = 'slots'
    location = author_booth
    sdesc = "slots"
    ldesc = "The wall of the booth proudly sports seventeen slots, each with
        the name of an author of interactive fiction printed above. The
        following authors are featured:\n\t
Matthew Amster-Burton\n\tNeil Brown\n\tAdam Cadre\n\tRybread Celsius\n\tNate
Cull\n\tNeil deMause\n\tDavid Dyte\n\tDavid Glasser\n\tStephen
Granade\n\tAngela M.\ Horns\n\tChris Klimas\n\tLeon Lin\n\tMagnus
Olsson\n\tAndrew Plotkin\n\tCody Sandifer\n\tDan Shiovitz\n\tScott Starkey"
;

mamster_slot: author_slot
    title = 'Matthew Amster-Burton'
    author = 1
    adjective = 'matthew' 'matt' 'amster' 'burton' 'amster-burton' 'mamster'
;
neilb_slot: author_slot
    title = 'Neil Brown'
    author = 2
    adjective = 'neil' 'brown' 'neilb'
;
adam_slot: author_slot
    title = 'Adam Cadre'
    adesc = "an Adam Cadre slot"
    author = 3
    adjective = 'adam' 'cadre'
;
rybread_slot: author_slot
    title = 'Rybread Celsius'
    author = -1
    adjective = 'rybread' 'celsius'
;
nate_slot: author_slot
    title = 'Nate Cull'
    author = 4
    adjective = 'nate' 'cull'
;
neild_slot: author_slot
    title = 'Neil deMause'
    author = 5
    adjective = 'neil' 'demause' 'neild'
;
ddyte_slot: author_slot
    title = 'David Dyte'
    author = 6
    adjective = 'david' 'dyte' 'ddyte'
;
dglasser_slot: author_slot
    title = 'David Glasser'
    author = 7
    adjective = 'david' 'glasser' 'dglasser'
;
sargent_slot: author_slot
    title = 'Stephen Granade'
    author = 8
    adjective = 'stephen' 'granade' 'sargent'
;
graham_slot: author_slot
    title = 'Angela M.\ Horns'
    author = -1
    adesc = "an Angela M.\ Horns slot"
    adjective = 'angela' 'm' 'horns' 'm.' 'graham' 'nelson'
;
chris_slot: author_slot
    title = 'Chris Klimas'
    author = 9
    adjective = 'chris' 'klimas'
;
leon_slot: author_slot
    title = 'Leon Lin'
    author = 10
    adjective = 'leon' 'lin'
;
magnus_slot: author_slot
    title = 'Magnus Olsson'
    author = 11
    adjective = 'magnus' 'olsson'
;
zarf_slot: author_slot
    title = 'Andrew Plotkin'
    author = 0
    adesc = "an Andrew Plotkin slot"
    adjective = 'andrew' 'plotkin' 'zarf'
;
cody_slot: author_slot
    title = 'Cody Sandifer'
    author = 12
    adjective = 'cody' 'sandifer'
;
inky_slot: author_slot
    title = 'Dan Shiovitz'
    author = 13
    adjective = 'dan' 'shiovitz' 'daniel' 'inky'
;
scott_slot: author_slot
    title = 'Scott Starkey'
    author = 14
    adjective = 'scott' 'starkey'
;

tile_basket: fixeditem, container
    noun = 'basket'
    adjective = 'wicker' 'woven'
    location = author_booth
    sdesc = "wicker basket"
    ldesc = {
        "Lying in one corner is a woven wicker basket. ";
        if (self.contents != [])
            "Inside it you see <<listcont(self)>>. ";
    }
    verDoTake(actor) = {
        "It must be much heavier than it looks--you are unable to lift it. ";
    }
    ioPutIn(actor, dobj) = {
        if (!dobj.isTile) {
            "There is no need to put <<dobj.thedesc>> in the basket. ";
            return;
        }
        pass ioPutIn;
    }
;

class tile: item
    isEquivalent = true
    isTile = true
    bulk = 0
    noun = 'tile'
    plural = 'tiles'
    sdesc = "<<self.title>> tile"
;

class blank_tile: tile
    isBlank = true
    title = 'blank'
    author = nil
    adjective = 'blank'
    location = tile_basket
    ldesc = "It is a deep, rich black color and resembles a Mah Jongg tile. "
;
bt1: blank_tile;
bt2: blank_tile;
bt3: blank_tile;
bt4: blank_tile;
bt5: blank_tile;
bt6: blank_tile;
bt7: blank_tile;
bt8: blank_tile;
bt9: blank_tile;
bt10: blank_tile;
bt11: blank_tile;
bt12: blank_tile;
bt13: blank_tile;
bt14: blank_tile;
bt15: blank_tile;
bt16: blank_tile;
bt17: blank_tile;
bt18: blank_tile;

class author_tile: tile
    notakeall = { return (self.location.isAuthorSlot); }
    sdesc = "<<self.title>> tile"
    ldesc = "It is a deep, rich black color and resembles a Mah Jongg tile.
        Across one side, the words \"<<self.title>>\" are written in flowing
        gold script. "
;
class mamster_tile: author_tile
    title = 'Matthew Amster-Burton'
    author = 1
    home = mamster_slot
    location = mamster_slot
    adjective = 'matthew' 'matt' 'amster' 'burton' 'amster-burton' 'mamster'
;
mt1: mamster_tile;
mt2: mamster_tile;
mt3: mamster_tile;
mt4: mamster_tile;
class neilb_tile: author_tile
    title = 'Neil Brown'
    author = 2
    home = neilb_slot
    location = neilb_slot
    adjective = 'neil' 'brown' 'neilb'
;
nbt1: neilb_tile;
nbt2: neilb_tile;
nbt3: neilb_tile;
nbt4: neilb_tile;
class adam_tile: author_tile
    title = 'Adam Cadre'
    author = 3
    home = adam_slot
    location = adam_slot
    adesc = "an Adam Cadre tile"
    adjective = 'adam' 'cadre'
;
at1: adam_tile;
at2: adam_tile;
at3: adam_tile;
at4: adam_tile;
class rybread_tile: author_tile
    title = 'Rybread Celsius'
    author = -1
    home = rybread_slot
    location = rybread_slot
    adjective = 'rybread' 'celsius'
;
rt1: rybread_tile;
rt2: rybread_tile;
rt3: rybread_tile;
rt4: rybread_tile;
class nate_tile: author_tile
    title = 'Nate Cull'
    author = 4
    home = nate_slot
    location = nate_slot
    adjective = 'nate' 'cull'
;
nat1: nate_tile;
nat2: nate_tile;
nat3: nate_tile;
nat4: nate_tile;
class neild_tile: author_tile
    title = 'Neil deMause'
    author = 5
    home = neild_slot
    location = neild_slot
    adjective = 'neil' 'demause' 'neild'
;
ndt1: neild_tile;
ndt2: neild_tile;
ndt3: neild_tile;
ndt4: neild_tile;
class ddyte_tile: author_tile
    title = 'David Dyte'
    author = 6
    home = ddyte_slot
    location = ddyte_slot
    adjective = 'david' 'dyte' 'ddyte'
;
ddt1: ddyte_tile;
ddt2: ddyte_tile;
ddt3: ddyte_tile;
ddt4: ddyte_tile;
class dglasser_tile: author_tile
    title = 'David Glasser'
    author = 7
    home = dglasser_slot
    location = dglasser_slot
    adjective = 'david' 'glasser' 'dglasser'
;
dgt1: dglasser_tile;
dgt2: dglasser_tile;
dgt3: dglasser_tile;
dgt4: dglasser_tile;
class sargent_tile: author_tile
    title = 'Stephen Granade'
    author = 8
    home = sargent_slot
    location = sargent_slot
    adjective = 'stephen' 'granade' 'sargent'
;
st1: sargent_tile;
st2: sargent_tile;
st3: sargent_tile;
st4: sargent_tile;
class graham_tile: author_tile
    title = 'Angela M.\ Horns'
    author = -1
    home = graham_slot
    location = graham_slot
    adesc = "an Angela M.\ Horns tile"
    adjective = 'angela' 'm' 'horns' 'm.' 'graham' 'nelson'
;
gt1: graham_tile;
gt2: graham_tile;
gt3: graham_tile;
gt4: graham_tile;
class chris_tile: author_tile
    title = 'Chris Klimas'
    author = 9
    home = chris_slot
    location = chris_slot
    adjective = 'chris' 'klimas'
;
ckt1: chris_tile;
ckt2: chris_tile;
ckt3: chris_tile;
ckt4: chris_tile;
class leon_tile: author_tile
    title = 'Leon Lin'
    author = 10
    home = leon_slot
    location = leon_slot
    adjective = 'leon' 'lin'
;
lt1: leon_tile;
lt2: leon_tile;
lt3: leon_tile;
lt4: leon_tile;
class magnus_tile: author_tile
    title = 'Magnus Olsson'
    author = 11
    home = magnus_slot
    location = magnus_slot
    adjective = 'magnus' 'olsson'
;
mot1: magnus_tile;
mot2: magnus_tile;
mot3: magnus_tile;
mot4: magnus_tile;
class zarf_tile: author_tile
    title = 'Andrew Plotkin'
    author = 0
    home = zarf_slot
    location = zarf_slot
    adesc = "an Andrew Plotkin tile"
    adjective = 'andrew' 'plotkin' 'zarf'
;
zt1: zarf_tile;
zt2: zarf_tile;
zt3: zarf_tile;
zt4: zarf_tile;
class cody_tile: author_tile
    title = 'Cody Sandifer'
    author = 12
    home = cody_slot
    location = cody_slot
    adjective = 'cody' 'sandifer'
;
cst1: cody_tile;
cst2: cody_tile;
cst3: cody_tile;
cst4: cody_tile;
class inky_tile: author_tile
    title = 'Dan Shiovitz'
    author = 13
    home = inky_slot
    location = inky_slot
    adjective = 'dan' 'shiovitz' 'daniel' 'inky'
;
it1: inky_tile;
it2: inky_tile;
it3: inky_tile;
it4: inky_tile;
class scott_tile: author_tile
    title = 'Scott Starkey'
    author = 14
    home = scott_slot
    location = scott_slot
    adjective = 'scott' 'starkey'
;
sst1: scott_tile;
sst2: scott_tile;
sst3: scott_tile;
sst4: scott_tile;

instructionsVerb: sysverb
    toldOnce = nil
    verb = 'instructions'
    sdesc = "instructions"
    action(actor) = {
        if (alcove.closed) {
            "For now, all you need do is turn the key in the main chamber. ";
            abort;
        }
        if (actor.location != startroom) {
            "From outside the booth, you hear Masta'mind shout, \(HEY! COME
                OUT HERE, SO THAT I MAY PROPERLY INSTRUCT YOU!\) ";
            return;
        }
        if (!self.toldOnce)
            self.toldOnce = true;
        else "\(HAVE I NOT TOLD YOU ALREADY?\)\ Masta'mind sighs.\b";
        "\(THE CONCEPT IS SIMPLE, THOUGH FIENDISH IN EXECUTION,\) Masta'mind
            says. \(BELOW ME IS A SERIES OF SLOTS, EACH BEARING THE
            NAME OF A TEXTFIRE DEMONSTRATION GAME. TO THE EAST IS THE
            BOOTH OF AUTHORS, WHEREIN YOUR DECISIONS BEGIN.\b
            
            INSIDE THE BOOTH IS ANOTHER SERIES OF SLOTS AND SOME TILES. THE
            SLOTS IN THE BOOTH BEAR THE NAMES OF AUTHORS OF TEXT ADVENTURES.
            PLACE EACH TILE IN A SLOT; THE TILE WILL BE IMPRINTED WITH
            THE NAME OF THE AUTHOR SHOWN ABOVE THE SLOT.\)\b
            
            The brain chortles nastily. \(YOUR TASK IS NOT AN
            EASY ONE. NOT EVERY AUTHOR IN THE BOOTH HAS WRITTEN FOR TEXTFIRE,
            AND SOME AUTHORS HAVE WRITTEN MORE THAN ONE GAME. TO EASE YOUR
            HERCULEAN TASK, I WILL LET YOU KNOW THAT NO AUTHOR HAS WRITTEN
            MORE THAN FOUR GAMES.\b
            
            ONCE YOU HAVE IMPARTED A NAME TO EVERY TILE, BRING THE TILES HERE.
            MATCH EACH AUTHOR TILE WITH THAT OF THE DEMO YOU BELIEVE HE OR SHE
            WROTE BY PLACING THE TILE IN THE APPROPRIATE SLOT.
            WHEN YOU ARE DONE, PULL THE WHITE LEVER. I WILL TELL
            YOU THE NUMBER OF CORRECT GUESSES YOU HAVE MADE, AND THE NUMBER
            OF CORRECT AUTHORS YOU HAVE CHOSEN WHO ARE MATCHED WITH THE
            INCORRECT GAME.\b
            
            YOU HAVE 10 GUESSES. GUESS CORRECTLY AND WIN. GUESS
            INCORRECTLY TEN TIMES AND FAIL.\)\b
            
            Pause. \(I SUSPECT YOU WILL FAIL OFTEN.\) ";
        abort;
    }
;

aboutVerb : sysverb
    verb = 'info' 'about'
    sdesc = "about"
    action(actor) = {
        if (global.wongames == 0) "\bThe demo version of \(Masta'mind\) was
            written by Riley Hawkins for TextFire, Inc.
        \bAbout the author:
\bAfter receiving a degree in library science from Texas Woman's
University, Riley Hawkins was directionless. Eventually, like so many
other Okies, Riley went west in search of opportunity, finally stopping in
Piedmont, CA. Unfortunately, Piedmont has a noticeable dearth of
libraries; fortunately, one of the numerous branches of the Oakland
Public Library system had a job opening.

\bA few months later, Riley met Shea Davis when Shea was doing research for
a game. In an odd series of coincidences, Deborah Keyes attended the same
PFLAG meeting Riley did, and Riley and Flip Winkler crossed paths at a
show at Slim's. Though still working at the library, Riley is now spending
more and more time writing interactive fiction.
        \bAbout TextFire:\b
TextFire, Inc.\ is a partnership of authors formed for the
purpose of providing quality works of interactive fiction to a worldwide
community of players, hobbyists, and enthusiasts.  Help support our efforts
to keep interactive fiction alive! ";
        else "\bThe demo version of \(Masta'mind\) was written by Stephen
            Granade in order to wring one last joke out of \"TextFire,
            Inc.\"\ before revealing the perpetrators.
        \bAbout the author:
\bA while back, Cody wrote Stephen, asking him to participate in an elaborate
April Fools joke. Stephen claims that Cody begged and pleaded with him to
join in the fun; Cody, however, maintains that Stephen was a replacement for
Rybread Celsius, and a poor one at that. No matter the details, Stephen
did join in, and has since enjoyed the reception which readers of
rec.arts.int-fiction and rec.games.int-fiction have given TextFire, Inc.

\bHe never received a degree from Texas Woman's University, nor has he
ever visited Slim's.

        \bAbout \"TextFire\":\b
        \"TextFire, Inc.\"\ was an elaborate April Fools joke, conceived by
            Cody Sandifer, and executed by Cody, Matthew Amster-Burton, Neil
Brown, Adam Cadre, Nate Cull, Neil deMause, David Dyte, David Glasser,
Stephen Granade, Chris Klimas, Leon Lin, Magnus Olsson, Dan Shiovitz, and
Scott Starkey. ";
 
        abort;
    }
;

textfireVerb : sysverb
    verb = 'textfire'
    sdesc = "TextFire"
    action(actor) = {
        if (global.wongames == 0) {
            "Cheaters never prosper, neener-neener-neener. ";
            abort;
        }
        "\bTextFire, Inc.\ was the brain-child of Cody Sandifer, who claims
he got the idea when, while fixing a leaky pipe, he slipped from his precarious
perch atop a toilet and hit his head. TextFire was to be an elaborate
shaggy-dog story involving a passel of IF authors. He made his pitch in
early March, and soon had fourteen people contributing demos. On April 1,
1998, Cody uploaded the TextFire \"12-pack,\" with its sixteen demos,
to GMD.\b

\(Demos and Authors\)\b
\tRevenge of the Killer Surf Nazi Robot Babes from Hell, by David Dyte\n\t
Bad Guys, by David Glasser\n\t
An Exploration of Colour, by Neil Brown\n\t
Coma!, by Nate Cull\n\t
Flowers for Algernon, by Adam Cadre\n\t
George, by Cody Sandifer (added later to the demos)\n\t
The Inanimator, by Matthew Amster-Burton\n\t
Insomnia, by Scott Starkey\n\t
Jack\'s Adventures, by Dan Shiovitz\n\t
Masta\'mind, by Stephen Granade\n\t
Will The Real Marjorie Hopkirk Please Stand Up?, by Neil Brown\n\t
Once, by Chris Klimas\n\t
Operate!, by Cody Sandifer\n\t
Pumping!, by Stephen Granade\n\t
A Tenuous Hold, by Stephen Granade\n\t
The U.S.\ Men\'s Hockey Team Olympic Challenge, by Leon Lin\n\t
Verb!, by Neil deMause\n\t
Zugzwang, by Magnus Olsson\b

Special thanks:\ Volker Blasius, whose complicity made all this possible,
and Adam Thornton, who single-handedly (and without any suggestions from us)
did so much to stir discussion about TextFire on the newsgroups.\b

Special note for conspiracy theorists:\ the two README files were written
by different people. Cody Sandifer wrote the first; Adam Cadre wrote the
second.\b

The number of bullets which hit John F.\ Kennedy will not be speculated on. ";
        abort;
    }
;

hintVerb : sysverb
    verb = 'hint' 'hints' 'help'
    sdesc = "help"
    action(actor) = {
        "There are no hints for this game. There are, however, INSTRUCTIONS. ";
        abort;
    }
;

registerVerb : sysverb
    verb = 'register' 'registration'
    sdesc = "register"
    action(actor) = {
        if (global.wongames == 0)
            "On-line registration for TextFire products will be available at
http://www.textfire.com on or before June 30, 1998.  Thanks for your
patience!";
        else "You must be joking. ";
        abort;
    }
;

modify quitVerb
    verb = 'q'
    quitGame(actor) =
    {
        local yesno;

        if (global.numgames > 0)
            scoreRank();
        "\bDo you really want to quit? (YES or NO) > ";
        yesno = yorn();
        "\b";
        if ( yesno == 1 )
        {
            terminate();    // allow user good-bye message
	    quit();
        }
        else
        {
            "Okay. ";
        }
    }
;

modify creditsVerb
    action(actor) = { aboutVerb.action(actor); }
;
