/*
	HEQUICKC.C

	Non-portable functions specific to QuickC/MS-DOS:

		hugo_blockalloc         hugo_clearfullscreen
		hugo_blockfree          hugo_clearwindow
					hugo_settextmode
		hugo_splitpath          hugo_settextwindow
		hugo_makepath           hugo_settextpos
		hugo_getfilename        hugo_scrollwindowup
		hugo_overwrite          hugo_font
		hugo_closefiles         hugo_settextcolor
					hugo_setbackcolor
		hugo_getkey             hugo_color
		hugo_getline
		hugo_waitforkey         hugo_print
					hugo_charwidth
		hugo_addcommand         hugo_textwidth
		hugo_restorecommand     hugo_strlen

		hugo_init_screen
		hugo_hasgraphics
		hugo_setgametitle
		hugo_cleanup_screen

	For the QuickC graphical build:

		main
		ClearGRwindow
		PrintGRchar
		PrintPortInformation

	for the Hugo Engine

	Copyright (c) 1995-2006 by Kent Tessman

	NOTE:  Generally, any functions with names such as _function()
	are QuickC library calls.  In such cases it may be preferable to
	use the appropriate call from the particular compiler library;
	if none is available, it may be necessary to write a brief routine
	that accomplishes the same result.

	These QuickC-interface routines are used to build both graphical
	and text-only versions of the file.  The flag GRAPHICS_SUPPORTED
	indicates sections included only for the graphical build.  Since
	graphics use in QuickC is more tweak-intensive, the "!defined
	(GRAPHICS_SUPPORTED)" sections are probably clearer to follow.
*/


#include "heheader.h"

#include <graph.h>      /* for QuickC screen and memory management */
#include <malloc.h>

#if defined (GRAPHICS_SUPPORTED)
#include <dos.h>
#if !defined (USE_BIOS_FONT)
#include "textfont.h"			/* generic 8x8 font */
#endif
#endif


/* Function prototypes: */

/* These should be provided by all ports: */
int hugo_color(int c);
int hugo_hasgraphics(void);

/* The following should be provided by all ports unless the port/system
   in question provides something better:
*/
void hugo_addcommand(void);
void hugo_restorecommand(int n);
#if defined (NO_LATIN1_CHARSET)
void hugo_stripaccents(char *a);
#endif

#if defined (GRAPHICS_SUPPORTED)
void ClearGRwindow(int left, int top, int right, int bottom);
void ConstrainCursor(void);
void PrintGRchar(int c);
void PrintPortInformation(void);
#endif

void hugo_initpalette(void);            /* from HEJPEG.C */

#if defined (DEBUGGER)
void *AllocMemory(size_t size);         /* from HDMISC.C */

extern int fake_graphics_mode;
#endif

/* If we're replacing main() in he.c */
#if defined (DEBUGGER)
#undef FRONT_END
#endif
#if defined (FRONT_END)
int he_main(int argc, char **argv);     /* from HE.C */
void Banner(void);
#endif


/* For the stdio version, the ARBITRARY_SCREEN_WIDTH must be equal
   to the width of the target text display in order for proper line-
   wrapping (since printf() wraps the line automatically at the
   right edge)
*/
#define ARBITRARY_SCREEN_WIDTH 80
#define ARBITRARY_SCREEN_HEIGHT 24


/* Definitions and variables: */

/* Defined Hugo colors: */
#define HUGO_BLACK         0
#define HUGO_BLUE          1
#define HUGO_GREEN         2
#define HUGO_CYAN          3
#define HUGO_RED           4
#define HUGO_MAGENTA       5
#define HUGO_BROWN         6
#define HUGO_WHITE         7
#define HUGO_DARK_GRAY     8
#define HUGO_LIGHT_BLUE    9
#define HUGO_LIGHT_GREEN   10
#define HUGO_LIGHT_CYAN    11
#define HUGO_LIGHT_RED     12
#define HUGO_LIGHT_MAGENTA 13
#define HUGO_YELLOW        14
#define HUGO_BRIGHT_WHITE  15

#define HOME_KEY        0x200
#define END_KEY         0x201
#define DELETE_KEY      0x202
#define INSERT_KEY      0x203
#define BACKSPACE_KEY   0x204
#define CTRL_LEFT_KEY   0x205
#define CTRL_RIGHT_KEY  0x206

int insert_mode = true;

/* Since HEQUICKC.C provides its own command history: */
#define HISTORY_SIZE    16              /* for command-line editing */
int hcount = 0;
char *history[HISTORY_SIZE];


/* Now some variables and constants to manage text/graphics display--
   all specific to HEQUICKC.C
*/
int text_windowleft, text_windowtop, text_windowright, text_windowbottom,
	text_windowwidth;
int current_text_color = DEF_FCOLOR, current_back_color = DEF_BGCOLOR;

/* QuickC/MS-DOS "style" colors: */
void hugo_fakefontcolors(void);
int current_font_color;
#define ITALIC_COLOR       HUGO_YELLOW
#define UNDERLINE_COLOR    HUGO_LIGHT_BLUE

char print_cursor = 0;   /* necessary when mixing text/graphics */
char override_latin1_conversion = false;

#define DEFAULT_TEXT_ROWS 25
int text_rows = DEFAULT_TEXT_ROWS;

#if defined (GRAPHICS_SUPPORTED)
const VGAHIGH_DT = _VRES16COLOR;        /* constants from <graph.h> */
const VGALOW_DT = _MRES256COLOR;
const NONE_DT = _TEXTC80;

/* From hejpeg.c: */
extern int SCREENCOLORS;
extern int display_type, color_space;
extern int HUGO_PALETTE[];      /* for Hugo text colors */

/* Copied from jpeglib.h: */
typedef enum
{
	JCS_UNKNOWN,            /* error/unspecified */
	JCS_GRAYSCALE,          /* monochrome */
	JCS_RGB,                /* red/green/blue */
	JCS_YCbCr,              /* Y/Cb/Cr (also known as YUV) */
	JCS_CMYK,               /* C/M/Y/K */
	JCS_YCCK                /* Y/Cb/Cr/K */
} J_COLOR_SPACE;

/* Various variables to help mimic a text display: */
int current_text_font = 0;
int small_font = false;
#if !defined (USE_BIOS_FONT)
extern unsigned char text_font[][FONT_HEIGHT];  /* from "textfont.h" */
#else
int FONT_WIDTH = 8, FONT_HEIGHT;
unsigned long text_font_addr;
#endif

/* current_text_col/row are SCREEN positions, not window */
int current_text_col = 1, current_text_row = 1;

#endif  /* if defined (GRAPHICS_SUPPORTED) */

#if !defined (NO_LATIN1_CHARSET)
#define LATIN1_START 0xa0
char latin1_to_ibm[] =
{
	0x20, 0xad, 0x9b, 0x9c,  ' ', 0x9d,  ' ', 0x15,	/* a0 - a7 */
	 ' ',  'c', 0xa6, 0xae, 0xaa, 0xc4,  'R',  ' ',	/* a8 - af */
	0xf8, 0xf1, 0xfd, 0xfc, 0x27, 0xe6,  ' ', 0xfa,	/* b0 - b7 */
	 ' ',  ' ', 0xa7, 0xaf, 0xac, 0xab,  ' ', 0xa8,	/* b8 - bf */
	 'A',  'A',  'A',  'A', 0x8e, 0x8f, 0x92, 0x80,	/* c0 - c7 */
	 'E', 0x90,  'E',  'E',  'I',  'I',  'I',  'I',	/* c8 - cf */
	 'D', 0xa5,  'O',  'O',  'O',  'O', 0x99,  'x',	/* d0 - d7 */
	 'O',  'U',  'U',  'U', 0x9a,  'Y',  ' ', 0xe1,	/* d8 - df */
	0x85, 0xa0, 0x83,  'a', 0x84, 0x86, 0x91, 0x87,	/* e0 - e7 */
	0x8a, 0x82, 0x88, 0x89, 0x8d, 0xa1, 0x8c, 0x8b,	/* e8 - ef */
	 'd', 0xa4, 0x95, 0xa2, 0x93,  'o', 0x94, 0xf6,	/* f0 - f7 */
	 'o', 0x97, 0xa3, 0x96, 0x81,  'y',  ' ', 0x98	/* f8 - ff */
};
#endif	/* !NO_LATIN1_CHARSET */


#if defined (FRONT_END)
/* main

	Replaces the old main() in HE.C (when FRONT_END is #defined),
	since we need to process MS-DOS-specific switches for graphics
	mode setting.
*/

int main(int argc, char *argv[])
{
	unsigned int i, j;
	int arg;

#if defined (GRAPHICS_SUPPORTED)
	unsigned char c;
	struct videoconfig vc;
	union REGS inregs, outregs;
	struct SREGS segregs;

	display_type = VGAHIGH_DT;

	color_space = JCS_RGB;          /* from jpeglib.h */

	for (arg=1; arg<argc; arg++)
	{
		if (argv[arg][0]=='-')
		{
			for (i=1; i<strlen(argv[arg]); i++)
			{
				switch (tolower(argv[arg][i]))
				{
					case 'h':
						display_type = VGAHIGH_DT;
						break;
					case 'l':
						display_type = VGALOW_DT;
						break;
					case 't':
						display_type = NONE_DT;
						break;
					case 'b':
						color_space = JCS_GRAYSCALE;
						break;
					case 's':
						small_font = true;
						break;

					case '?':
						PrintPortInformation();
						exit(0);

					default:
						goto PrintSwitches;
				}
			}
		}
	}

	/* Now skim through and remove any switches, since
	   ParseCommandLine() in HEMISC.C doesn't know a switch
	   from a hole in the ground
	*/
	for (i=1; i<(unsigned)argc; i++)
	{
		if (argv[i][0]=='-')
		{
			for (j=i; j<(unsigned)argc-1; j++)
				argv[j] = argv[j+1];
			arg--, argc--, i--;
		}
	}

	if (argc==1)
	{
PrintSwitches:
		Banner();
		printf("SWITCHES:");
		printf("\t-h  High resolution\n");
		printf("\t\t-l  Low resolution\n");
		printf("\t\t-t  Text-only\n");
		printf("\t\t-b  Black-and-white graphics mode\n");
		printf("\t\t-s  Small font (high-res only)\n");
		printf("\t\t-?  Additional information\n");
		exit(0);
	}

CalculateFontDimensions:

	/* "Large" font only for 640x480 */
	if (display_type!=VGAHIGH_DT) small_font = true;

#if defined (USE_BIOS_FONT)
	/* Calculate dimensions for small font... */
	if (small_font)
	{
		FONT_HEIGHT = 8;
		c = 3;          /* font code for 8x8  */
	}

	/* ...or regular-sized font */
	else
	{
		FONT_HEIGHT = 16;
		c = 8;          /* font code for 8x16  */
	}
	
	_asm
	{
		push bp
		mov ah, 11h
		mov al, 30h
		mov bh, c
		int 10h
		mov i, es
		mov j, bp
		pop bp
	}
	text_font_addr = (unsigned long)i*16L + (unsigned long)j;

#endif	/* if defined (USE_BIOS_FONT) */

	/* Make sure we have hardware that'll display some sort of
	   graphics:
	*/
	if (!_setvideomode(display_type))
	{
		_setvideomoderows(_DEFAULTMODE, DEFAULT_TEXT_ROWS);
		printf("FATAL ERROR:  Unsupported display mode\n");
		exit(-1);
	}

	if (display_type!=NONE_DT)
		_settextrows((small_font && display_type==VGAHIGH_DT)?60:30);
	else
		/* Force color for text display */
		color_space = JCS_RGB;


	/* Just because this takes so damned long */
	if (display_type!=VGAHIGH_DT && display_type!=NONE_DT)
		printf("Initializing palette...");

	hugo_initpalette();

#endif  /* if defined (GRAPHICS_SUPPORTED) */

RunMain:
	return he_main(argc, argv);     /* start the engine */
}

#if defined (GRAPHICS_SUPPORTED)
void PrintPortInformation(void)
{
	Banner();
	printf("\nThis version of the Hugo Engine has VGA-enabled graphics support.\n");
	printf("Its functioning modes are high-resolution (640x480, 16 colors) and\n");
	printf("low-resolution (320x200, 256 colors).  SVGA-enabled graphics are\n");
	printf("available in the 32-bit build of the Engine.\n");

	printf("\nJPEG graphics display is based in part on the work of the\n");
	printf("Independent JPEG Group.\n");
}
#endif  /* if defined (GRAPHICS_SUPPORTED) */
#endif  /* if defined (FRONT_END) */


/*
    MEMORY ALLOCATION:

    hugo_blockalloc(), and hugo_blockfree() are necessary because of
    the way MS-DOS handles memory allocation across more than 64K.
    For most systems, these will simply be normal ANSI function calls.
*/

void *hugo_blockalloc(long num)
{
	return halloc(num, sizeof(char));       /* i.e. malloc() */
}

void hugo_blockfree(void *block)
{
	hfree(block);                           /* i.e. free() */
}


/*
    FILENAME MANAGEMENT:

    Different operating systems have different ways of naming files.
    The following routines are simply required to know and be able to
    dissect/build the components of a particular filename,
    storing/restoring the components via the specified char arrays.

    For example, in MS-DOS:

	hugo_splitpath("C:\HUGO\FILES\HUGOLIB.H", ...)
		becomes:  C:, HUGO\FILES, HUGOLIB, H

    and

	hugo_makepath(..., "C:", "HUGO\FILES", "HUGOLIB", "H")
		becomes:  C:\HUGO\FILES\HUGOLIB.H

    The appropriate equivalent nomenclature should be used for the
    operating system in question.
*/

void hugo_splitpath(char *path, char *drive, char *dir, char *fname, char *ext)
{
	_splitpath(path, drive, dir, fname, ext);
}

void hugo_makepath(char *path, char *drive, char *dir, char *fname, char *ext)
{
	_makepath(path, drive, dir, fname, ext);
}


/* hugo_getfilename

    Loads the name of the filename to save or restore (as specified by
    the argument <a>) into the line[] array.
*/

void hugo_getfilename(char *a, char *b)
{
	unsigned int i, p;

	sprintf(line, "Enter path and filename %s.", a);

	AP(line);

	sprintf(line,"%c(Default is %s): \\;", NO_CONTROLCHAR, b);
	AP(line);

	p = var[prompt];
	var[prompt] = 0;        /* null string */

	RunInput();

	var[prompt] = p;

	remaining = 0;

	strcpy(line, "");
	if (words==0)
		strcpy(line, b);
	else
	{
		for (i=1; i<=(unsigned int)words; i++)
			strcat(line, word[i]);
	}
}


/* hugo_overwrite

    Checks to see if the given filename already exists, and prompts to
    replace it.  Returns true if file may be overwritten.
*/

int hugo_overwrite(char *f)
{
	FILE *tempfile;

	tempfile = fopen(f, "rb");
	if (tempfile==NULL)                     /* if file doesn't exist */
		return true;

	fclose(tempfile);

	sprintf(pbuffer, "Overwrite existing \"%s\" (Y or N)? ", f);
	RunInput();

	if (words==1 && (!strcmp(strupr(word[1]), "Y") || !strcmp(strupr(word[1]), "YES")))
		return true;

	return false;
}


/* hugo_closefiles

    Closes all open files.  NOTE:  If the operating system automatically
    closes any open streams upon exit from the program, this function may
    be left empty.
*/

void hugo_closefiles()
{
	fcloseall();                    /* non-ANSI function */
/*
	fclose(game);
	if (script) fclose(script);
	if (io) fclose(io);
	if (record) fclose(record);
*/
}


/* hugo_getkey

    Returns the next keystroke waiting in the keyboard buffer.  It is
    expected that hugo_getkey() will return the following modified
    keystrokes:

	up-arrow        11 (CTRL-K)
	down-arrow      10 (CTRL-J)
	left-arrow       8 (CTRL-H)
	right-arrow     21 (CTRL-U)
*/

int hugo_getkey(void)
{
	int b;

	b = getch();

	/* A minor hack because we don't want the left-arrow and backspace
	   to have the same effect:
	*/
	if (b==8) return BACKSPACE_KEY;

	/* Since in QuickC, extended-keyboard keys return a 0, prompting
	   another call to getch() for the second part of the key value
	   waiting in the buffer; these extended keystrokes are returned
	   as a value greater than 255. */

	if (b==0)
	{
		switch (b=getch())
		{
			case 71:		/* Home */
				{b = HOME_KEY; break;}
			case 72:		/* up-arrow */
				{b = 11; break;}
			case 75:		/* left-arrow */
				{b = 8; break;}
			case 77:		/* right-arrow */
				{b = 21; break;}
			case 79:		/* End */
				{b = END_KEY; break;}
			case 80:        	/* down-arrow */
				{b = 10; break;}
			case 82:		/* Insert */
				{b = INSERT_KEY; break;}
			case 83:		/* DEL */
				{b = DELETE_KEY; break;}
			case 115:
				{b = CTRL_LEFT_KEY; break;}
			case 116:
				{b = CTRL_RIGHT_KEY; break;}

			default:                /* anything else */
				b += 256;
		}
		return b;
	}
#if !defined (NO_LATIN1_CHARSET)
	/* Convert DOS extended-ASCII codes to Latin1 */
        else if ((b&0xff) > 128)
	{
		int i;
		
		for (i=LATIN1_START; i<=0xff; i++)
		{
			if ((unsigned char)latin1_to_ibm[i-LATIN1_START]==(unsigned char)(b&0xff))
			{
				b = i;
				break;
			}
		}
		if (i>0xff) b = ' ';	/* invalid character */
	}
#endif

	return b;
}


/* hugo_getline

    Gets a line of input from the keyboard, storing it in <buffer>.

    (NOTE:  The function keys used here are QuickC/MS-DOS specific.
    They have been chosen to somewhat mirror the command-line editing
    keys of MS-DOS.  For other systems, the "switch (b)" lines
    will likely need to be changed.)

    The advantage of using this instead of an available library input
    function is when building the Debugger--pressing the Tab key switches
    to the debug screen; if a command such as Restart, Restore, etc. is
    given, the engine must know to abandon the input.

    Note also that this input function assumes a non-proportional font.
*/

#define CURRENT_CHAR(c)	((c<(signed)strlen(buffer))?buffer[c]:' ')

void hugo_getline(char *p)
{
	char ch[2];
	int a, b, thiscommand;
	int c;				/* c is the character being added */
	int x, y, oldx, oldy;

NewPrompt:
	hugo_settextcolor(fcolor);
	hugo_setbackcolor(bgcolor);

	strcpy(buffer, "");
	c = 0;
	thiscommand = hcount;

	oldx = (currentpos+FIXEDCHARWIDTH)/FIXEDCHARWIDTH;
	oldy = currentline;
	y = oldy;
	x = oldx + hugo_strlen(p);

	hugo_print(p);

GetKey:

	hugo_settextpos(x, y);
	if (print_cursor)
	{
		override_latin1_conversion = true;
		sprintf(ch, "%c", 219);
		hugo_print(ch);
		override_latin1_conversion = false;
	}
	else
	{
		if (insert_mode)
			_settextcursor(0x0607);
		else
			_settextcursor(0x0007);
	}


	b = hugo_getkey();

	hugo_settextcolor(icolor);
	hugo_setbackcolor(bgcolor);

	/* Now, start key checking */
	switch (b)
	{
#if defined (DEBUGGER)
		case (9):                       /* Tab */
		{
			during_input = true;
			Debugger();
			during_input = false;

			/* If the debugger is stopping execution, quitting,
			   etc., a blank string must be returned in the
			   input buffer.
			*/
			if (debugger_collapsing)
			{
				strcpy(buffer, "");
				if (active_screen==GAME)
					hugo_scrollwindowup();
				return;
			}

			goto GetKey;
		}
#endif

		case (13):                      /* Enter */
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			full = 0;

			if (script) fprintf(script, "%s%s\n", p, buffer);

			while (kbhit())
				getch();
			hugo_settextpos(1, y + 2);
			if (y >= physical_windowheight/lineheight - 1)
				hugo_scrollwindowup();

			strcpy(buffer, Rtrim(buffer));
			hugo_addcommand();
#if defined (NO_LATIN1_CHARSET)
			hugo_stripaccents(buffer);
#endif

			_settextcursor(0x0607);		/* normal cursor */

			return;
		}
		case INSERT_KEY:
		{
			insert_mode = !insert_mode;
			goto GetKey;
		}
		case BACKSPACE_KEY:
		case DELETE_KEY:
		{
			if (strlen(buffer)>0)
			{
				if (b==BACKSPACE_KEY)
				{
					if (c==0) goto GetKey;
					c--;
				}

				for (a=c; a<=(signed)strlen(buffer); a++)
					buffer[a] = buffer[a+1];

				if (b==BACKSPACE_KEY) x--;

				hugo_settextpos(oldx+hugo_strlen(p), y);
				hugo_print(buffer);
				hugo_settextpos(oldx+hugo_strlen(p)+strlen(buffer), y);
				hugo_print("  ");
			}
			goto GetKey;
		}
		case (8):			/* left-arrow */
		{
			if (c > 0)
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				x--, c--;
			}
			goto GetKey;
		}
		case (21):			/* right-arrow */
		{
			if (c<(signed)strlen(buffer))
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				x++, c++;
			}
			goto GetKey;
		}
		case CTRL_LEFT_KEY:
		{
			if (c)
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				do
				{
					do
						c--, x--;
					while (c && buffer[c-1]!=' ');
				}
				while (c && buffer[c]==' ');
			}
			goto GetKey;
		}
		case CTRL_RIGHT_KEY:
		{
			if (c<(signed)strlen(buffer))
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				do
				{
					do
						c++, x++;
					while (c<(signed)strlen(buffer) &&
						buffer[c-1]!=' ');
				}
				while (c<(signed)strlen(buffer) && buffer[c]==' ');
			}
			goto GetKey;
		}
		case HOME_KEY:
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			c = 0;
			x = oldx + hugo_strlen(p);
			goto GetKey;
		}
		case END_KEY:
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			c = strlen(buffer);
			x = oldx + hugo_strlen(p) + strlen(buffer);
			goto GetKey;
		}
		case (27):                      /* Escape */
		case (24):                      /* CTRL-X */
		{
BlankLine:
			hugo_settextpos(oldx+hugo_strlen(p), y);

			memset(buffer, ' ', text_windowwidth);

			buffer[text_windowwidth-oldx-hugo_strlen(p)] = '\0';
			hugo_print(buffer);
			hugo_settextpos(oldx, y);

			goto NewPrompt;
		}
		case (256+61):                  /* F3 */
		case (11):                      /* up-arrow */
		{
			if (--thiscommand<0)
			{
				thiscommand = 0;
				goto GetKey;
			}
			a = strlen(buffer);
RestoreCommand:
			hugo_restorecommand(thiscommand);
			x = oldx + strlen(buffer) + hugo_strlen(p);
			hugo_settextpos(oldx+hugo_strlen(p), y);
			hugo_print(buffer);
			while (a >= (int)strlen(buffer))
				{hugo_print(" ");
				a--;}
			hugo_settextpos(x-1, y);
			c = strlen(buffer);
			goto GetKey;
		}
		case (10):                      /* down-arrow */
		{
			a = strlen(buffer);
			if (++thiscommand>=hcount) goto BlankLine;
			goto RestoreCommand;
		}
	}

	/* Disallow invalid keystrokes */
	if (b < 32 || b>=256) goto GetKey;

	/* Hugo circa v2.2 allowed '^' and '~' for '\n' and '\"',
	   respectively
	*/
	if (game_version<=22 && (b=='^' || b=='~')) goto GetKey;

	if (oldx+strlen(buffer) >= text_windowwidth-2)
		goto GetKey;

	hugo_settextpos(x++, y);
	sprintf(ch, "%c", b);                   /* add the new character */
	hugo_print(ch);
	buffer[strlen(buffer)+1] = '\0';
	if (c<(signed)strlen(buffer) && insert_mode)
	{
		hugo_settextpos(x, y);
		hugo_print(buffer+c);
		for (a=strlen(buffer); a>c; a--)
			buffer[a] = buffer[a-1];
	}
	buffer[c] = (char)b;

	c++;

	goto GetKey;
}


/* hugo_waitforkey

    Provided to be replaced by multitasking systems where cycling while
    waiting for a keystroke may not be such a hot idea.
*/

int hugo_waitforkey(void)
{
	while (kbhit())                 /* clear key buffer */
		hugo_getkey();
	while (!kbhit());

	return hugo_getkey();
}


/* hugo_iskeywaiting

    Returns true if a keypress is waiting to be retrieved.
*/

int hugo_iskeywaiting(void)
{
	return kbhit()?1:0;
}


/* hugo_timewait

    Waits for 1/n seconds.  Returns false if waiting is unsupported.
*/

int hugo_timewait(int n)
{
	clock_t goal;
	
	goal = CLOCKS_PER_SEC/n + clock();
	while (goal > clock());

	return true;
}


/*
    COMMAND HISTORY:

    To store/retrieve player inputs for editing.
*/

void hugo_addcommand(void)
{
	int i;

	if (!strcmp(buffer, "")) return;

	if (hcount>=HISTORY_SIZE)
	{
		hugo_blockfree(history[0]);
		for (i=0; i<HISTORY_SIZE-1; i++)
			history[i] = history[i+1];
		hcount = HISTORY_SIZE-1;
	}

	/* Because the debugger might use (up to) all available memory for
	   code line storage, a different means of memory allocation is
	   needed (at least in MS-DOS due to limited available memory to
	   begin with).
	*/
#if !defined (DEBUGGER)
	if ((history[hcount] = hugo_blockalloc((long)((strlen(buffer)+1)*sizeof(char))))==NULL)
#else
	if ((history[hcount] = AllocMemory((size_t)((strlen(buffer)+1)*sizeof(char))))==NULL)
#endif
	{
		hugo_blockfree(history[0]);
		if (hcount)
		{
			for (i=0; i<hcount; i++)
				history[i] = history[i+1];
			hcount--;
		}
		return;
	}

	for (i=0; i<=(int)strlen(buffer); i++)
		history[hcount][i] = buffer[i];
	hcount++;
}

void hugo_restorecommand(int n)
{
	int i;

	if (n < 0 || (n>=hcount && hcount!=HISTORY_SIZE-1)) return;

	i = 0;
	do
		buffer[i] = history[n][i];
	while (history[n][i++]!='\0');
}


/* hugo_stripaccents */

#if defined (NO_LATIN1_CHARSET)

void hugo_stripaccents(char *a)
{
	/* If there is no Latin1 character set, accented characters
	   should be converted to ASCII equivalents at input time.
	   (NOTE:  These >127 extended ASCII codes are not 
	   machine-independent.) */

	int i;

	for (i=0; i<(int)strlen(a); i++)
	{
		if ((unsigned char)a[i]>=0x80)
		{
			switch ((unsigned char)a[i])
			{
				case 131:               /* 'A' accents */
				case 132:
				case 133:
				case 134:
				case 160:
					{a[i] = 'a';
					break;}
				case 142:
				case 143:
					{a[i] = 'A';
					break;}

				case 135:               /* 'C' cedilla */
					{a[i] = 'c';
					break;}
				case 128:
					{a[i] = 'C';
					break;}

				case 130:               /* 'E' accents */
				case 136:
				case 137:
				case 138:
					{a[i] = 'e';
					break;}
				case 144:
					{a[i] = 'E';
					break;}

				case 139:               /* 'I' accents */
				case 140:
				case 141:
				case 161:
					{a[i] = 'i';
					break;}

				case 164:               /* 'N' tilde */
					{a[i] = 'n';
					break;}
				case 165:
					{a[i] = 'N';
					break;}

				case 147:               /* 'O' accents */
				case 148:
				case 149:
				case 162:
					{a[i] = 'o';
					break;}
				case 153:
					{a[i] = 'O';
					break;}

				case 129:               /* 'U' accents */
				case 150:
				case 151:
				case 163:
					{a[i] = 'u';
					break;}
				case 154:
					{a[i] = 'U';
					break;}

				case 152:               /* 'Y' accents */
					{a[i] = 'y';
					break;}

				case 174:               /* '<<' */
				case 175:               /* '>>' */
					{a[i] = '\"';
					break;}

				default:
				{
					strcpy(a+i, a+i+1);
				}
			}
		}
	}
}

#endif /* if defined (NO_LATIN1_CHARSET) */


/*
    DISPLAY CONTROL:

    These functions are specific library calls to QuickC/MS-DOS
    for screen control.  Each _function() should be replaced by
    the equivalent call for the operating system in question.  If
    a particular operation is unavailable, in some cases it may be
    left empty.  In other cases, it may be necessary to write a
    brief function to achieve the desired effect.
*/

#if !defined (DEBUGGER)         /* If the Debugger is being built, these
				   functions are included in HDQUICKC.C */
void hugo_init_screen(void)
{
/* Does whatever has to be done to initially set up the display. */

#if defined (GRAPHICS_SUPPORTED)
/* Shouldn't need to do this again
	_setvideomoderows(display_type, small_font?60:30);
*/
#else
	_setvideomoderows(_TEXTC80, text_rows); /* simple 80-column text */
#endif
}

void hugo_cleanup_screen(void)
{
/* Does whatever has to be done to clean up the display pre-termination. */

	_setvideomoderows(_DEFAULTMODE, DEFAULT_TEXT_ROWS);

	hugo_settextcolor(DEF_FCOLOR);
	hugo_setbackcolor(DEF_BGCOLOR);
	hugo_settextwindow(1, 1, SCREENWIDTH/FIXEDCHARWIDTH, SCREENHEIGHT/FIXEDLINEHEIGHT);
	hugo_clearfullscreen();
	hugo_settextpos(1, 1);
}

#endif  /* !defined (DEBUGGER) */


int hugo_hasgraphics(void)
{
/* Returns true if the current display is capable of graphics display */

#if defined (GRAPHICS_SUPPORTED)
	if (display_type!=NONE_DT)
		return true;
#elif defined (DEBUGGER)
	return fake_graphics_mode;
#else
	return false;
#endif
}


void hugo_setgametitle(char *t)
{}


void hugo_clearfullscreen(void)
{
/* Clears everything on the screen, moving the cursor to the top-left
   corner of the screen */

#if defined (GRAPHICS_SUPPORTED)
	if (display_type!=NONE_DT)
	{
		ClearGRwindow(1, 1, SCREENWIDTH, SCREENHEIGHT);
	}
	else
#else
		_clearscreen(_GCLEARSCREEN);
#endif

	/* Must be set: */
	currentpos = 0;
	currentline = 1;
}


void hugo_clearwindow(void)
{
/* Clears the currently defined window, moving the cursor to the top-left
   corner of the window */

#if defined (GRAPHICS_SUPPORTED)
	if (display_type!=NONE_DT)
		ClearGRwindow(text_windowleft, text_windowtop, text_windowright, text_windowbottom);
	else
#endif
		_clearscreen(_GWINDOW);

	/* Must be set: */
	currentpos = 0;
	currentline = 1;
}


#if defined (GRAPHICS_SUPPORTED)
void ClearGRwindow(int left, int top, int right, int bottom)
{
	/* Only necessary in graphics mode, since in text mode QuickC's
	   library will clear to the current text background color
	*/
	int i;

	_setcolor(HUGO_PALETTE[current_back_color]);
	_rectangle(_GFILLINTERIOR, (left-1)*FIXEDCHARWIDTH, (top-1)*FIXEDLINEHEIGHT,
		right*FIXEDCHARWIDTH-1, bottom*FIXEDLINEHEIGHT-1);
}
#endif


void hugo_settextmode(void)
{
/* This function does whatever is necessary to set the system up for
   a standard text display */

	struct videoconfig vc;

#if defined (GRAPHICS_SUPPORTED)
	if (display_type!=NONE_DT)
	{
		print_cursor = true;
/* Shouldn't need to do this again:
		_setvideomoderows(display_type,
			(small_font and display_type==VGAHIGH_DT)?60:30);
	}
	else
	{
		_setvideomoderows(_TEXTC80, text_rows);
*/
	}
#endif

	_getvideoconfig(&vc);

	/* Screen parameters/measurements must be set (as character
	   or pixel coordinates, as applicable):
	*/
#if defined (GRAPHICS_SUPPORTED)
	if (display_type!=NONE_DT)
	{
		/* fixed-width/non-proportional */
		charwidth = FIXEDCHARWIDTH = FONT_WIDTH;
		lineheight = FIXEDLINEHEIGHT = FONT_HEIGHT;

#if !defined (USE_BIOS_FONT)
		/* Because text_font is only 8x8 */
		if (!small_font) lineheight = FIXEDLINEHEIGHT*=2;
#endif

		/* Graphics/text display, so pixel coordinates: */
		SCREENWIDTH = vc.numxpixels;
		SCREENHEIGHT = vc.numypixels;
	}
	else
	{
#endif  /* if defined (GRAPHICS_SUPPORTED) */

		/* again, for non-proportional */
		charwidth = FIXEDCHARWIDTH = 1;
		lineheight = FIXEDLINEHEIGHT = 1;

		/* Text-only display, so character coordinates: */
		SCREENWIDTH = vc.numtextcols;
		SCREENHEIGHT = vc.numtextrows;

#if defined (GRAPHICS_SUPPORTED)
	}
#endif

	/* It's necessary to make sure the video BIOS is set to use the
	   high-bit of the background color as intensity, not blinking
	*/
	_asm
	{
		mov ah, 10h		/* function */
		mov al, 03h		/* sub-function */
		mov bl, 0h		/* enable intensity */
		INT 10h
	}

	/* Must be set: */
	hugo_settextwindow(1, 1, SCREENWIDTH/FIXEDCHARWIDTH, SCREENHEIGHT/FIXEDLINEHEIGHT);
}


void hugo_settextwindow(int left, int top, int right, int bottom)
{
/* Once again, the arguments for the window are passed using character
   coordinates--a system setting a window using pixel coordinates will
   have to make the necessary conversions using FIXEDCHARWIDTH and 
   FIXEDLINEHEIGHT.

   The text window, once set, represents the scrolling/bottom part of the
   screen.  It is also assumed that the text window will constrain the
   cursor location--see hugo_settextposition(), below.

   This function must also set physical_window... parameters using
   the transformations given.
*/
	_settextwindow(top, left, bottom, right);

	/* Character, not pixel, coordinates: */
	/* (Note that these are QuickC-specific parameters) */
	text_windowtop = top;        text_windowbottom = bottom;
	text_windowleft = left;      text_windowright = right;

	text_windowwidth = text_windowright-text_windowleft+1;


	/* Must be set: */
	/* (Engine-required parameters) */
	physical_windowleft = (left-1)*FIXEDCHARWIDTH;
	physical_windowtop = (top-1)*FIXEDLINEHEIGHT;
	physical_windowright = right*FIXEDCHARWIDTH-1;
	physical_windowbottom = bottom*FIXEDLINEHEIGHT-1;
	physical_windowwidth = (right-left+1)*FIXEDCHARWIDTH;
	physical_windowheight = (bottom-top+1)*FIXEDLINEHEIGHT;

#if defined (GRAPHICS_SUPPORTED)
	ConstrainCursor();
#endif
}


void hugo_settextpos(int x, int y)
{
/* The top-left corner of the current active window is (1, 1).

   (In other words, if the screen is being windowed so that the top row
   of the window is row 4 on the screen, the (1, 1) refers to the 4th
   row on the screen, and (2, 1) refers to the 5th.)

   All cursor-location is based on FIXEDCHARWIDTH and FIXEDLINEHEIGHT.

   This function must also properly set currentline and currentpos (where
   currentline is a the current character line, and currentpos may be
   either in pixels or characters, depending on the measure being used).

   Note that the Hugo function call uses x and y directly as text-
   screen coordinates; pixel-based systems will likely have to make a
   calculation to pixel coordinates.  In other words, pixel-based
   screen location will need to calculate pixel coordinates to
   simulate text-screen coordinates.
*/
	_settextposition(y, x);         /* row, column */

	/* Must be set: */
	currentline = y;
	currentpos = (x-1)*FIXEDCHARWIDTH;   /* Note:  zero-based */

#if defined (GRAPHICS_SUPPORTED)
	current_text_col = text_windowleft-1+x;
	current_text_row = text_windowtop-1+y;
	ConstrainCursor();
#endif
}


#if defined (GRAPHICS_SUPPORTED)
void ConstrainCursor(void)
{
	/* ConstrainCursor() is needed only in graphics mode, since
	   HEQUICKC.C's cursor-positioning doesn't automatically pay
	   attention to any currently-defined window.
	*/
	if (current_text_col > text_windowright)
			current_text_col = text_windowright;
	if (current_text_col < text_windowleft)
			current_text_col = text_windowleft;
	if (current_text_row > text_windowbottom)
			current_text_row = text_windowbottom;
	if (current_text_row < text_windowtop)
			current_text_row = text_windowtop;
}
#endif


void hugo_print(char *a)
{
/* Essentially the same as printf() without formatting, since printf()--at
   least in QuickC--doesn't take into account color setting, font changes,
   windowing, etc.  The newline character '\n' must be explicitly included
   at the end of a line in order to produce a linefeed.  The new cursor
   position is set to the end of this printed text.  Upon hitting the
   right edge of the screen, the printing position wraps to the start of
   the next line. */

#if defined (GRAPHICS_SUPPORTED)
	int i, len;

	len = strlen(a);
	
	if (display_type!=NONE_DT)
	{
		for (i=0; i<len; i++)
		{
			switch (a[i])
			{
				case '\n':
					++current_text_row;
					break;
				case '\r':
					current_text_col = text_windowleft;
					break;
				default:
				{
					char c;
					
					c = a[i];
#if !defined (NO_LATIN1_CHARSET)
					if (!override_latin1_conversion)
					{
					    if ((unsigned char)a[i] >= LATIN1_START)
						c = latin1_to_ibm[(unsigned char)a[i]-LATIN1_START];
					    else if ((unsigned char)c==0x97)	/* DOS fonts don't have em dash? */
					    	c = 0xc4;
					}
#endif
					PrintGRchar((unsigned char)c);

					if (++current_text_col > text_windowright)
					{
						current_text_col = text_windowleft;
						++current_text_row;
					}
				}
			}

			if (current_text_row > text_windowbottom)
			{
				current_text_row = text_windowbottom;
				hugo_scrollwindowup();
			}
		}
	}
	else
#endif  /* if defined (GRAPHICS_SUPPORTED) */
	{
#if !defined (NO_LATIN1_CHARSET)
		char l[MAXBUFFER];
		int i, len;

		strcpy(l, a);
		len = strlen(l);
	
		for (i=0; i<len; i++)
		{
			if ((unsigned char)l[i] >= LATIN1_START)
				l[i] = latin1_to_ibm[(unsigned char)l[i]-LATIN1_START];
		}
#endif
		if (currentfont==NORMAL_FONT || currentfont==PROP_FONT)
			_settextcolor(current_text_color);
		else
		{
			hugo_fakefontcolors();
			_settextcolor(current_font_color);
		}
		_outtext(l);
	}
}


#if defined (GRAPHICS_SUPPORTED)
/* PrintGRchar

	Prints a text character to the graphics screen using the
	the default VGA font.
*/

void PrintGRchar(int c)
{
	int color, col, row, font_row;
	union REGS inregs, outregs;

	for (row=0; row<FONT_HEIGHT; row++)
	{
#if !defined (USE_BIOS_FONT)
		font_row = text_font[c-32][row];
#else
		font_row = *((int _far *)(text_font_addr+c*FONT_HEIGHT+row)) & 0xff;
#endif
		/* Double each pixel for boldface */
		if (current_text_font & BOLD_FONT)
			font_row |= font_row>>1;

		/* Add a bottom line for underline */
		if ((current_text_font & UNDERLINE_FONT) && row==FONT_HEIGHT-1)
			font_row = 0xffff;

		/* Shift the top half of the character for italic */
		if ((current_text_font & ITALIC_FONT) && row<FONT_HEIGHT/2)
			font_row = font_row>>1;

		for (col=0; col<FONT_WIDTH; col++)
		{
			/* If the font says a pixel should be plotted here */
			if (font_row & 1<<(FONT_WIDTH-col-1))
				color = HUGO_PALETTE[current_text_color];
			else
				color = HUGO_PALETTE[current_back_color];

			/* Use BIOS interrupt 10h, function 0Ch to
			   plot a pixel:
			*/
			inregs.h.ah = 0x0c;             /* function */
#if defined (DEBUGGER)
			/* video page */
			inregs.h.bh = (unsigned char)active_screen;
#else
			inregs.h.bh = 0x00;
#endif
			inregs.h.al = (unsigned char)color;
			inregs.x.cx = (current_text_col-1)*FONT_WIDTH+col;
			inregs.x.dx = (current_text_row-1)*FONT_HEIGHT+row;

#if !defined (USE_BIOS_FONT)
			if (!small_font) inregs.x.dx*=2;
#endif
			int86(0x10, &inregs, &outregs); /* plot the point */

#if !defined (USE_BIOS_FONT)
			if (!small_font)
			{
				inregs.x.dx++;
				int86(0x10, &inregs, &outregs);
			}
#endif
		}
	}
}
#endif  /* if defined (GRAPHICS_SUPPORTED) */


void hugo_scrollwindowup()
{
	_scrolltextwindow(_GSCROLLUP);  /* i.e. one line */

#if defined (GRAPHICS_SUPPORTED)
	/* The irksome QuickC graphical text functions don't fill in the
	   background color below a scroll with anything but black
	*/
	if (display_type!=NONE_DT && bgcolor!=HUGO_BLACK)
	{
		_setcolor(HUGO_PALETTE[current_back_color]);
		_rectangle(_GFILLINTERIOR, (text_windowleft-1)*FIXEDCHARWIDTH,
			(text_windowbottom-1)*FIXEDLINEHEIGHT, text_windowright*FIXEDCHARWIDTH-1,
			text_windowbottom*FIXEDLINEHEIGHT-1);
	}
#endif
}


void hugo_font(int f)
{
/* Since MS-DOS doesn't support fonts per se, boldface, italic, and
   underline print are represented by colors, defaulting to black (0) if
   the background color is the same as the typeface color.  Not very
   spectacular, but then neither is MS-DOS. */

	int current;
	int bold_color;

#if defined (GRAPHICS_SUPPORTED)
	if (display_type==NONE_DT)
	{
/*
		_settextcolor(hugo_color(fcolor));

		if (f & UNDERLINE_FONT)
			hugo_settextcolor((bgcolor!=UNDERLINE_COLOR) ? UNDERLINE_COLOR : HUGO_BLACK);

		if (f & ITALIC_FONT)
			hugo_settextcolor((bgcolor!=ITALIC_COLOR) ? ITALIC_COLOR : HUGO_BLACK);

		if (f & BOLD_FONT)
		{
			bold_color = hugo_color((int)fcolor)+8;
			if (bold_color>=16) bold_color -=8;
			hugo_settextcolor(((int)bgcolor!=bold_color) ? bold_color : HUGO_BLACK);
		}
*/
	}
	else
		current_text_font = f;
#endif
}

void hugo_fakefontcolors()
{
	if (currentfont & UNDERLINE_FONT)
		current_font_color = (bgcolor!=UNDERLINE_COLOR) ? UNDERLINE_COLOR : HUGO_BLACK;

	if (currentfont & ITALIC_FONT)
		current_font_color = (bgcolor!=ITALIC_COLOR) ? ITALIC_COLOR : HUGO_BLACK;

	if (currentfont & BOLD_FONT)
	{
		int bold_color = current_text_color+8;
		if (bold_color>=16) bold_color -=8;
		current_font_color = ((int)bgcolor!=bold_color) ? bold_color : HUGO_BLACK;
	}
}


/* Constants needed for coercion of colors in the event that we're forcing
   to grayscale (needed by hugo_settextcolor() and hugo_setbackcolor()):
*/
#define BW_DIM_BACK      8
#define BW_DIM_FORE     11
#define BW_BRIGHT_BACK	12
#define BW_BRIGHT_FORE	15

void hugo_settextcolor(int c)           /* foreground (print) color */
{
	current_text_color = hugo_color(c);
#if defined (GRAPHICS_SUPPORTED)
	/* If we're forcing a black-and-white palette, it is necessary
	   to tweak the colors--otherwise 0-15 are just varying shades
	   of black-->gray-->white
	*/
	if (color_space==JCS_GRAYSCALE && current_text_color!=0)
	{
		if (current_text_color>=8) current_text_color = BW_BRIGHT_FORE;
		else current_text_color = BW_DIM_FORE;
	}
	_settextcolor(HUGO_PALETTE[hugo_color(current_text_color)]);
#else
/* hugo_print() does this now:
	_settextcolor(current_text_color);
*/
#endif
}


void hugo_setbackcolor(int c)           /* background color */
{
	current_back_color = hugo_color(c);
#if defined (GRAPHICS_SUPPORTED)
	/* As with hugo_settextcolor(), it's necessary to tweak the
	   "color" if we're forcing to grayscale
	*/
	if (color_space==JCS_GRAYSCALE && current_back_color!=0)
	{
		if (current_back_color>=8) current_back_color = BW_BRIGHT_BACK;
		else current_back_color = BW_DIM_BACK;
	}
	_setbkcolor(HUGO_PALETTE[hugo_color(current_back_color)]);
#else
	/* Because the QuickC text-output function won't output text on 
	   a "bright" background:
	*/
	c = hugo_color(c);

	if (c > 8 && c < 16)
	{
		/* First make it non-bright */
		c-=8;

		/* Then make sure the text is still visible */
		if (c==_gettextcolor())
			c = (c==HUGO_BLACK)?HUGO_WHITE:HUGO_BLACK;
	}

	_setbkcolor(current_back_color = c);
#endif
}


int hugo_color(int c)
{
	/* Color-setting functions should always pass the color through
	   hugo_color() in order to properly set default fore/background
	   colors:
	*/

	if (c==16)      c = DEF_FCOLOR;
	else if (c==17) c = DEF_BGCOLOR;
	else if (c==18) c = DEF_SLFCOLOR;
	else if (c==19) c = DEF_SLBGCOLOR;
	else if (c==20) c = hugo_color(fcolor);

/* Uncomment this block of code and change "c = ..." values if the system
   palette differs from the Hugo palette.

   If colors are unavailable on the system in question, it may suffice
   to have black, white, and brightwhite (i.e. boldface).  It is expected
   that colored text will be visible on any other-colored background.

	switch (c)
	{
		case HUGO_BLACK:	 c = 0;  break;
		case HUGO_BLUE:		 c = 1;  break;
		case HUGO_GREEN:	 c = 2;  break;
		case HUGO_CYAN:		 c = 3;  break;
		case HUGO_RED:		 c = 4;  break;
		case HUGO_MAGENTA:	 c = 5;  break;
		case HUGO_BROWN:	 c = 6;  break;
		case HUGO_WHITE:	 c = 7;  break;
		case HUGO_DARK_GRAY:	 c = 8;  break;
		case HUGO_LIGHT_BLUE:	 c = 9;  break;
		case HUGO_LIGHT_GREEN:	 c = 10; break;
		case HUGO_LIGHT_CYAN:	 c = 11; break;
		case HUGO_LIGHT_RED:	 c = 12; break;
		case HUGO_LIGHT_MAGENTA: c = 13; break;
		case HUGO_YELLOW:	 c = 14; break;
		case HUGO_BRIGHT_WHITE:	 c = 15; break;
*/
	return c;
}


/* CHARACTER AND TEXT MEASUREMENT

	For non-proportional printing, these should be pretty much as
	described below for the QuickC functions, as screen dimensions
	will be given in characters, not pixels.

	For proportional printing, screen dimensions need to be in
	pixels, and each width routine must take into account the
	current font and style.

	The hugo_strlen() function is used to give the length of
	the string not including any non-printing control characters.
*/

int hugo_charwidth(char a)
{
	if (a==FORCED_SPACE)
		return FIXEDCHARWIDTH;         /* same as ' ' */

	else if ((unsigned char)a >= ' ') /* alphanumeric characters */

		return FIXEDCHARWIDTH;         /* because printing under MS-DOS
					     is non-proportional */
	return 0;
}


int hugo_textwidth(char *a)
{
	int i, slen, len = 0;

	slen = (int)strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else
			len += hugo_charwidth(a[i]);
	}

	return len;
}


int hugo_strlen(char *a)
{
	int i, slen, len = 0;

	slen = (int)strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else len++;
	}

	return len;
}


#if !defined (COMPILE_V25)
int hugo_hasvideo(void)
{
	return false;
}

int hugo_playvideo(HUGO_FILE infile, long reslength,
	char loop_flag, char background, int volume)
{
	fclose(infile);
	return true;
}
void hugo_stopvideo(void)
{}
#endif

#if !defined (GRAPHICS_SUPPORTED)
int hugo_displaypicture(FILE *infile, long reslength) /* from hejpeg.c */
{
	fclose(infile);
	return true;
}
#endif

#if !defined (SOUND_SUPPORTED)
int hugo_playmusic(FILE *infile, long reslength, char loop_flag)
{
	fclose(infile);
	return true;	/* not an error */
}

void hugo_musicvolume(int vol)
{}

void hugo_stopmusic(void)
{}

int hugo_playsample(FILE *infile, long reslength, char loop_flag)
{
	fclose(infile);
	return true;	/* not an error */
}

void hugo_samplevolume(int vol)
{}

void hugo_stopsample(void)
{}
#endif
