//	<><><><><><><><><><><><><>  Plot3DF.cpp  <><><><><><><><><><><><><><> */
//		
// ----------------------------------------------------
//
// Implementation of Plot3DF C++ class
//
// --------------------------------------------------------------------------
//

#include "stdafx.h"	// precompiled header
#include <stdio.h>
#include "Plot3DF.h"

// ------------------------------------------------------------
//                         Constructor
// ------------------------------------------------------------
Plot3DF::Plot3DF()
{
	m_plot_mode = PLOT_CROSS_SECTIONS;

	// use default values
	SetParameters();
	SetClipMode();

	// initialize dynamic memory
	ZeroMaskPointers();
}

// ------------------------------------------------------------
//                          Destructor
// ------------------------------------------------------------
Plot3DF::~Plot3DF()
{
	DeleteMaskArrays();
}

// ------------------------------------------------------------
//                   Hidden Line Masking Routines
// ------------------------------------------------------------
// returns: 0=ok, 1=error
int Plot3DF::AllocateMaskArrays()
{
	int rc=0;

	m_pMask1  = new MASK_POINT[m_mask_size];
	m_pMask2  = new MASK_POINT[m_mask_size];
	if (m_pMask1 == 0 || m_pMask2 == 0)
	{
		DeleteMaskArrays();
		rc = 1;
	}
	return(rc);
}

// ------------------------------------------------------------
void Plot3DF::ZeroMaskPointers()
{
	m_nXps    = 0;
	m_nYps    = 0;
	m_nZps    = 0;

	m_mask_size = 0;
	m_pMask1  = 0;
	m_pMask2  = 0;
}

// ------------------------------------------------------------
// delete mask arrays
void Plot3DF::DeleteMaskArrays()
{
	delete [] m_pMask1;
	delete [] m_pMask2;
	ZeroMaskPointers();
}

// ------------------------------------------------------------
int Plot3DF::SetParameters(unsigned nzplanes,unsigned remove_hidden_lines)
{
	if (nzplanes<2) nzplanes=2;
	m_nzplanes = nzplanes;
	m_remove_hidden_lines = remove_hidden_lines;
	return(0);
}

// ------------------------------------------------------------
void Plot3DF::Dump()
{
	DBGPLOT("------------------ Plot3DF Dump Begin -----------------------\n");
	DBGPLOT("Inputs:\n");
	DBGPLOT("  Nzplanes=%6u  Remove_HiddenLines=%u\n", m_nzplanes,m_remove_hidden_lines);

	DBGPLOT("Plotting Variables:\n");
	DBGPLOT("  nXps=%u  nYps=%u  nZps=%u\n", m_nXps, m_nYps, m_nZps);
	DBGPLOT("  dxp=%10.4f  dyp=%10.4f  dzp=%10.4f\n", m_dxp,m_dyp,m_dzp);

	DBGPLOT("Mask Array: nsize=%u\n", m_mask_size);
	DBGPLOT("  pMask1:  %p\n", m_pMask1);
	DBGPLOT("  pMask2:  %p\n", m_pMask2);

	DBGPLOT("------------------- Plot3DF Dump End ------------------------\n");
}

// ------------------------------------------------------------
int Plot3DF::InitPlot()
{
	PlotOutputOn();
	CalcTM();	// calculate transformation matrix
	CalcCorners();	// calculate corners of x,y plate
	CalcDeltas();
	AllocateMaskArrays();
	return(0);
}

// ------------------------------------------------------------
void Plot3DF::CalcDeltas()
{
	double   dxp, dyp, dxv, dyv;
	unsigned uxv, uyv;

	PLOT_ASSERT(m_xvmax > m_xvmin);
	PLOT_ASSERT(m_yvmax > m_yvmin);

	// range check nplanes
	if (m_nzplanes < 2) m_nzplanes = 2;

	// video ranges
	uxv = m_xvmax - m_xvmin;
	uyv = m_yvmax - m_yvmin;
	if (uxv < 2) uxv = 2;
	if (uyv < 2) uyv = 2;
	dxv = uxv;
	dyv = uyv;

	// calc number of increments
	m_nXps = uxv;
	m_nYps = uyv;
	m_nZps = m_nzplanes;
	m_mask_size = m_nXps;

	// calculate for spherical range
	if (m_use_sphere_range)
	{
		m_xpmid = xpf(m_sphere_xcenter,m_sphere_ycenter,m_sphere_zcenter);
		m_ypmid = ypf(m_sphere_xcenter,m_sphere_ycenter,m_sphere_zcenter);
		m_zpmid = zpf(m_sphere_xcenter,m_sphere_ycenter,m_sphere_zcenter);
		m_xpmax = m_xpmid + m_sphere_radius;
		m_xpmin = m_xpmid - m_sphere_radius;
		m_ypmax = m_ypmid + m_sphere_radius;
		m_ypmin = m_ypmid - m_sphere_radius;
		m_zpmax = m_zpmid + m_sphere_radius;
		m_zpmin = m_zpmid - m_sphere_radius;
	}

	// ratio so image fits in the video window
	// keep 1:1 aspect ratio with x,y video axis
	dxp = m_xpmax-m_xpmin;
	dyp = m_ypmax-m_ypmin;
	if (dxp*dyv > dyp*dxv)
	{
		dyp = (dyp*dyv)/(dxv*2.);
		m_ypmin = m_ypmid - dyp;
		m_ypmax = m_ypmid + dyp;
	}
	else
	{
		dxp = (dxp*dxv)/(dyv*2.);
		m_xpmin = m_xpmid - dxp;
		m_xpmax = m_xpmid + dxp;
	}

	// apply scale factor
	if (m_scale==0.) m_scale = 1.e-6;	// avoid division by zero
	dxp = (m_xpmax-m_xpmin)/(m_scale*2.);
	dyp = (m_ypmax-m_ypmin)/(m_scale*2.);
	m_xpmax = m_xpmid + dxp;
	m_xpmin = m_xpmid - dxp;
	m_ypmax = m_ypmid + dyp;
	m_ypmin = m_ypmid - dyp;

	// calc increments deltas
	m_dxp = (m_xpmax-m_xpmin)/double(m_nXps-1);
	m_dyp = m_dxp;	// maintain proper aspect ratio
	m_dzp = (m_zpmax-m_zpmin)/double(m_nZps-1);

	// calc plot to video scaling
	m_p2v = (m_scale * (double)(m_xvmax - m_xvmin)) / (double)(m_xpmax - m_xpmin);

	// plotting must start from outmost of the screen inward for proper hidden line removal
	if (m_zpcorner[1]>=m_zpcorner[0])
	{
		m_xstart[0] =  m_xmax;
		m_xstart[1] =  m_xmin;
	}
	else
	{
		m_xstart[0] =  m_xmin;
		m_xstart[1] =  m_xmax;
	}
	if (m_zpcorner[2]>m_zpcorner[0])
	{
		m_ystart[0] =  m_ymax;
		m_ystart[1] =  m_ymin;
	}
	else
	{
		m_ystart[0] =  m_ymin;
		m_ystart[1] =  m_ymax;
	}
	if (m_ypcorner[0]<m_ypcorner[4])
	{
		m_zstart[0] = m_zmin;
		m_zstart[1] = m_zmax;
	}
	else
	{
		m_zstart[0] = m_zmax;
		m_zstart[1] = m_zmin;
	}
}

// ------------------------------------------------------------
// returns: 0=not visible, 1=visible
int Plot3DF::IsVisible(double xp,double yp,double zp)
{
	double x,y,z,f1,f2,zpsave;
	int w1,w2;

	if (m_remove_hidden_lines==0) return(1);  // visible if no hidden line removal

	// zearch for a zero crossing outward from screen
	zpsave = zp;
	zp = zpsave - m_dzp;
	x  = xof(xp,yp,zp);
	y  = yof(xp,yp,zp);
	z  = zof(xp,yp,zp);
	f1 = Func(x,y,z);
	if (m_clip_mode == CLIP_USER)
	{
		w1 = Within(x,y,z);
		if (m_is_clip_inverted) w1 = !w1;
	}
	else
		w1 = Plot3DF::Within(x,y,z);
	for (zp-=m_dzp; zp>=m_zpmin; zp-=m_dzp)
	{
		x  = xof(xp,yp,zp);
		y  = yof(xp,yp,zp);
		z  = zof(xp,yp,zp);
		f2 = Func(x,y,z);
		if (m_clip_mode == CLIP_USER)
		{
			w2 = Within(x,y,z);
			if (m_is_clip_inverted) w2 = !w2;
		}
		else
			w2 = Plot3DF::Within(x,y,z);

		if (f1*f2<=0. && w1 && w2)
		{
			return(0);
		}
		else
		{
			f1 = f2;
			w1 = w2;
		}
	}
	return(1);
}

// ------------------------------------------------------------
// does the actual work of plotting
int Plot3DF::Plot()
{
	int rc;
	if (m_plot_mode == PLOT_PARAMETRIC)
		rc = PlotParametric();
	else
		rc = PlotSectional();
	return(rc);
}


// --------------------
//   POINT3  POINT4
//
//   POINT1  POINT2
// --------------------

#define POINT0	 0,  0,  0	// no movement indicator
#define POINT1	 1,  0,  0
#define POINT2	 2,  1,  0
#define POINT3	 3,  0,  1
#define POINT4	 4,  1,  1

// Draw Look-up-table
typedef struct DrawLUT_s
{
	int	from;	 // 0=none 1,2,3,4
	int xv_from; // 0 or 1
	int yv_from; // 0 or 1
	int	to;		 // 0,1,2,3,4
	int xv_to;	 // 0 or 1
	int yv_to;	 // 0 or 1
} DrawLUT;

static DrawLUT drawLUT[16] =
{
   { POINT0, POINT0 },	 //  0
   { POINT1, POINT4 },	 //  1
   { POINT2, POINT3 },	 //  2
   { POINT1, POINT2 },	 //  3
   { POINT2, POINT3 },	 //  4
   { POINT2, POINT4 },	 //  5
   { POINT1, POINT4 },	 //  6
   { POINT1, POINT4 },	 //  7
   { POINT1, POINT4 },	 //  8
   { POINT2, POINT3 },	 //  9
   { POINT1, POINT3 },	 // 10
   { POINT2, POINT3 },	 // 11
   { POINT3, POINT4 },	 // 12
   { POINT2, POINT3 },	 // 13
   { POINT1, POINT4 },	 // 14
   { POINT0, POINT0 }	 // 15
};

void Plot3DF::PlotMask(unsigned iyp)
{
	unsigned ixp,mask;
	int xv,yv,xvm1,yvm1,w1,w2;
	long color;
	MASK_POINT* pMaskPlot[2][2];
	MASK_POINT* pFrom;
	MASK_POINT* pTo;

	pMaskPlot[0][0] = m_pMask1  ;
	pMaskPlot[0][1] = m_pMask1+1;
	pMaskPlot[1][0] = m_pMask2  ;
	pMaskPlot[1][1] = m_pMask2+1;
	xvm1 = m_xvmin - 1;
	yvm1 = m_yvmin - 1;
	for (ixp=1;ixp<m_nXps;ixp++)
	{
		mask = (pMaskPlot[1][0]->sf & 8) | 
			   (pMaskPlot[1][1]->sf & 4) |
			   (pMaskPlot[0][0]->sf & 2) |
			   (pMaskPlot[0][1]->sf & 1) ;

		if (drawLUT[mask].from!=0)
		{
			pFrom = pMaskPlot[drawLUT[mask].xv_from][drawLUT[mask].yv_from];
			pTo   = pMaskPlot[drawLUT[mask].xv_to  ][drawLUT[mask].yv_to  ];

			if (m_clip_mode == CLIP_USER)
			{
				w1 = Within(pFrom->x,pFrom->y,pFrom->z);
				w2 = Within(  pTo->x,  pTo->y,  pTo->z);
				if (m_is_clip_inverted) w1 = !w1;
				if (m_is_clip_inverted) w2 = !w2;
			}
			else
			{
				w1 = Plot3DF::Within(pFrom->x,pFrom->y,pFrom->z);
				w2 = Plot3DF::Within(  pTo->x,  pTo->y,  pTo->z);
			}

			if (w1 && w2 &&	// both points must be within drawing area
			    IsVisible(pFrom->xp,pFrom->yp,pFrom->zp) &&	// both points must be visible
				IsVisible(pFrom->xp,pFrom->yp,pFrom->zp) )
			{
				xv = drawLUT[mask].xv_from + ixp + xvm1;
				yv = drawLUT[mask].yv_from + iyp + yvm1;
				FilterMoveTo(xv,yv);
				xv = drawLUT[mask].xv_to   + ixp + xvm1;
				yv = drawLUT[mask].yv_to   + iyp + yvm1;
				color = Color(pFrom->x,pFrom->y,pFrom->z);
				FilterColorTo(color);
				FilterDrawTo(xv,yv);
			}
		}

		// advance to next check point
		pMaskPlot[1][0] = pMaskPlot[1][1]++;
		pMaskPlot[0][0] = pMaskPlot[0][1]++;
	} // for ixp
}

// ------------------------------------------------------------
// plot cross sections
int Plot3DF::PlotSectional()
{
	MASK_POINT* pMask;
	unsigned ix,iy,iz;
	double   xp,yp,zp;

	InitPlot();
	cbBegPlot();

	// Note: incrementing xp,yp,zp is faster than table lookups
	for (iz=0,zp=m_zpmin; iz<m_nZps; iz++,zp+=m_dzp)
	{
		cbCheckUserAbort();
		if (m_abort_plotting)
		{
			cbEndPlot();
			return(1);
		}
		cbBegZPlane(zp);
		pMask = m_pMask2;
		for (iy=0,yp=m_ypmin; iy<m_nYps; iy++,yp+=m_dyp)
		{
			for (ix=0,xp=m_xpmin; ix<m_nXps; ix++,xp+=m_dxp)
			{
				pMask->xp = xp;
				pMask->yp = yp;
				pMask->zp = zp;
				pMask->x  = xof(xp,yp,zp);
				pMask->y  = yof(xp,yp,zp);
				pMask->z  = zof(xp,yp,zp);
				pMask->f  = Func( pMask->x, pMask->y, pMask->z );
				pMask->sf = (pMask->f >= 0.) ? 0xffff : 0;
				pMask++;
			} // for ix
			if (iy != 0)
				PlotMask(iy);
			// swap mask arrays
			pMask    = m_pMask1;
			m_pMask1 = m_pMask2;
			m_pMask2 = pMask;
			// Note: pMask is now set for next iteration
		} // for iy
		cbEndZPlane(zp);
	} // for iz

	DrawAxis();

	cbEndPlot();
	return(0);
}

// ------------------------------------------------------------
// plot parametrically
int Plot3DF::PlotParametric()
{
	double	u,v,du,dv,uinc,vinc,x,y,z,xp,yp,zp;
	unsigned iu,iv,mask;
	int		xv,yv,v1=0,v2=0,within;
	long color;

	InitPlot();
	cbBegPlot();

	// calculate du, dv, uinc, vinc
	if (m_nulines<2) m_nulines = 2;
	if (m_nvlines<2) m_nvlines = 2;
	du = (m_umax-m_umin)/(double)(m_nulines-1);
	dv = (m_vmax-m_vmin)/(double)(m_nvlines-1);
	if (m_nuincs<2) m_nuincs = 2;
	if (m_nvincs<2) m_nvincs = 2;
	uinc = (m_umax-m_umin)/(double)(m_nuincs-1);
	vinc = (m_vmax-m_vmin)/(double)(m_nvincs-1);

	// draw constant u lines
	if (m_draw_u)
	{
		for (v=m_vmin,iv=0; iv<m_nvlines; v+=dv,iv++)
		{
			cbCheckUserAbort();
			if (m_abort_plotting)
			{
				cbEndPlot();
				return(1);
			}
			cbBegULine(v);
			for (u=m_umin,iu=0; iu<m_nuincs; u+=uinc,iu++)
			{
				x = XParametric(u,v);
				y = YParametric(u,v);
				z = ZParametric(u,v);
				xp = xpf(x,y,z);
				yp = ypf(x,y,z);
				zp = zpf(x,y,z);
				xv = xvf(xp);
				yv = yvf(yp);
				v2 = IsVisible(xp,yp,zp) ? 1 : 0;
				if (iu==0)
				{
					FilterMoveTo(xv,yv);
				}
				else
				{
					mask = (v2<<1) | v1;
					switch(mask)
					{
					case 0:	// hidden to hidden
						break;
					case 1:	// visible to hidden
						break;
					case 2:	// hidden to visible
						FilterMoveTo(xv,yv);
						break;
					case 3:	// visible to visible
						if (m_clip_mode == CLIP_USER)
						{
							within = Within(x,y,z);
							if (m_is_clip_inverted) within = !within;
						}
						else
							within = Plot3DF::Within(x,y,z);
						if (!within)
						{
							FilterMoveTo(xv,yv);
							break;
						}
						color = Color(x,y,z);
						FilterColorTo(color);
						FilterDrawTo(xv,yv);
						break;
					}
				}
				// setup for next time
				v1 = v2;
			} // for u
			cbEndULine(v);
		} // for v
	} // if draw u

	// draw constant v lines
	if (m_draw_v)
	{
		for (u=m_umin,iu=0; iu<m_nulines; u+=du,iu++)
		{
			cbCheckUserAbort();
			if (m_abort_plotting)
			{
				cbEndPlot();
				return(1);
			}
			cbBegVLine(u);
			for (v=m_vmin,iv=0; iv<m_nvincs; v+=vinc,iv++)
			{
				x = XParametric(u,v);
				y = YParametric(u,v);
				z = ZParametric(u,v);
				xp = xpf(x,y,z);
				yp = ypf(x,y,z);
				zp = zpf(x,y,z);
				xv = xvf(xp);
				yv = yvf(yp);
				v2 = IsVisible(xp,yp,zp) ? 1 : 0;
				if (iv==0)
				{
					FilterMoveTo(xv,yv);
				}
				else
				{
					mask = (v2<<1) | v1;
					switch(mask)
					{
					case 0:	// hidden to hidden
						break;
					case 1:	// visible to hidden
						break;
					case 2:	// hidden to visible
						FilterMoveTo(xv,yv);
						break;
					case 3:	// visible to visible
						if (m_clip_mode == CLIP_USER)
						{
							within = Within(x,y,z);
							if (m_is_clip_inverted) within = !within;
						}
						else
							within = Plot3DF::Within(x,y,z);
						if (!within)
						{
							FilterMoveTo(xv,yv);
							break;
						}
						FilterDrawTo(xv,yv);
						break;
					}
				}
				// setup for next time
				v1 = v2;
			} // for v
			cbEndVLine(u);
		} // for u
	} // if draw v

	cbEndPlot();
	return(0);
}

// ------------------------------------------------------------
void Plot3DF::DrawAxis(void)
{
	CString str;
	double dx,dy,dz,x1,y1,z1,x2,y2,xp,yp;
	double degrees,gap_length,tick_length;
	int xv1,yv1,xv2,yv2;
	unsigned n;

	if (!m_show_axis) return;

	FilterColorTo(m_axis.axis_color);
	cbBegDrawAxis();

	// Constant Y Front Edge
	MapAndDrawLine(m_xstart[0],m_ystart[0],m_zstart[0],
				   m_xstart[1],m_ystart[0],m_zstart[0]);

	// Constant X Front Edge
	MapAndDrawLine(m_xstart[0],m_ystart[0],m_zstart[0],
				   m_xstart[0],m_ystart[1],m_zstart[0]);

	// Vertical Front Edges
	MapAndDrawLine(m_xstart[0],m_ystart[0],m_zstart[0],
				   m_xstart[0],m_ystart[0],m_zstart[1]);

	MapAndDrawLine(m_xstart[1],m_ystart[0],m_zstart[0],
				   m_xstart[1],m_ystart[0],m_zstart[1]);

	MapAndDrawLine(m_xstart[0],m_ystart[1],m_zstart[0],
				   m_xstart[0],m_ystart[1],m_zstart[1]);

	// -------------
	// X tick marks
	// -------------
	if (m_axis.nxticks > 0)
	{
		n = m_axis.nxticks-1;
		if (n==0) n=1; // avoid divions by zero
		dx =  m_xstart[0] - m_xstart[1];
		dy = (m_ystart[1] - m_ystart[0])/((double)n);
		gap_length  = dx * (double)m_axis.tick_spacing / 500.;
		tick_length = dx * (double)m_axis.tick_length  / 500.;
	
		x1 = m_xstart[0]+gap_length;  y1 = m_ystart[0];	z1 = m_zstart[0];
		x2 = x1 + tick_length;
		MapPoint(m_xstart[1],m_ystart[0],m_zstart[0],&xv1,&yv1);
		MapPoint(m_xstart[0],m_ystart[0],m_zstart[0],&xv2,&yv2);
		degrees = Rad2Deg( atan2( -(double)(yv2-yv1), (double)(xv2-xv1) ) );
		for (n=0;n<m_axis.nxticks;n++,y1+=dy)
		{
			MapPoint(x1,y1,z1,&xv1,&yv1);
			MapPoint(x2,y1,z1,&xv2,&yv2);
			FilterMoveTo(xv1,yv1);
			FilterDrawTo(xv2,yv2);
			str.Format("y=%-8.4lf",y1);
			PlotText(&m_axis,degrees,xv2,yv2,str);
		}
	}

	// -------------
	// Y tick marks
	// -------------
	if (m_axis.nyticks > 0)
	{
		n = m_axis.nyticks-1;
		if (n==0) n=1; // avoid divions by zero
		dy =  m_ystart[0] - m_ystart[1];
		dx = (m_xstart[1] - m_xstart[0])/((double)n);
		gap_length  = dy * (double)m_axis.tick_spacing / 500.;
		tick_length = dy * (double)m_axis.tick_length  / 500.;
	
		y1 = m_ystart[0]+gap_length;  x1 = m_xstart[0];	z1 = m_zstart[0];
		y2 = y1 + tick_length;
		MapPoint(m_xstart[0],m_ystart[1],m_zstart[0],&xv1,&yv1);
		MapPoint(m_xstart[0],m_ystart[0],m_zstart[0],&xv2,&yv2);
		degrees = Rad2Deg( atan2( -(double)(yv2-yv1), (double)(xv2-xv1) ) );
		for (n=0;n<m_axis.nyticks;n++,x1+=dx)
		{
			MapPoint(x1,y1,z1,&xv1,&yv1);
			MapPoint(x1,y2,z1,&xv2,&yv2);
			FilterMoveTo(xv1,yv1);
			FilterDrawTo(xv2,yv2);
			str.Format("x=%-8.4lf",x1);
			PlotText(&m_axis,degrees,xv2,yv2,str);
		}
	}

	// -------------
	// Z tick marks
	// -------------
	if (m_axis.nzticks > 0)
	{
		n = m_axis.nzticks-1;
		if (n==0) n=1; // avoid divions by zero
		dz =  m_zstart[1] - m_zstart[0];
		gap_length  = dz * (double)m_axis.tick_spacing / 500.;
		tick_length = dz * (double)m_axis.tick_length  / 500.;
		dz = dz/(double)n;

		x1 = m_xcorner[0];  y1 = m_ycorner[0];  z1=m_xpcorner[0];
		if (m_xpcorner[1]>z1)
		{	x1 = m_xcorner[1];  y1 = m_ycorner[1];  z1=m_xpcorner[1];	}
		if (m_xpcorner[2]>z1)
		{	x1 = m_xcorner[2];  y1 = m_ycorner[2];  z1=m_xpcorner[2];	}
		if (m_xpcorner[3]>z1)
		{	x1 = m_xcorner[3];  y1 = m_ycorner[3];  z1=m_xpcorner[3];	}
		MapAndDrawLine(x1,y1,m_zstart[0],
					   x1,y1,m_zstart[1]);
		z1 = m_zstart[0];
		for (n=0;n<m_axis.nzticks;n++,z1+=dz)
		{
			xp = xpf(x1,y1,z1)+gap_length;
			yp = ypf(x1,y1,z1);
			xv1 = xvf(xp);
			yv1 = yvf(yp);
			xp += tick_length;
			xv2 = xvf(xp);
			yv2 = yvf(yp);
			FilterMoveTo(xv1,yv1);
			FilterDrawTo(xv2,yv2);
			str.Format("z=%-8.4lf",z1);
			PlotText(&m_axis,0.,xv2,yv2+m_axis.font_height,str);
		}
	}

	cbEndDrawAxis();
}

/*	<><><><><><><><><><><><><>  Plot3DF.cpp  <><><><><><><><><><><><><><> */














