(*  :Title:	Supporting Routines  *)

(*  :Authors:	Brian Evans, James McClellan  *)

(*
    :Summary:	To provide routines that Mathematica should have.
		Many are borrowed from Lisp.
 *)

(*  :Context:	SignalProcessing`Support`SupCode`  *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1989-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*  :History:	*)

(*  :Keywords:	list processing, sequences, number theory, set theory	*)

(*  :Source:	*)

(*  :Warning:	*)

(*  :Mathematica Version:  1.2 or 2.0  *)

(*  :Limitation:  *)

(*
    :Functions:	AllSubsets
		Arrow2D
		AssociateItem
		CirclePS
		Combine
		ComplexQ
		ComplexTo2DCoord
		ComplexTo2DCoordList
		ConstantQ
		ConstantTerm
		Element
		EmptyQ
		GenerateCode
		GeneratePattern
		GenerateSymbol
		GetAllExponents
		GetAllFactors
		GetRoot
		GetRootList
		GetStateField
		GetVariables
		GetValue
		HasAttributes
		ImaginaryQ
		InRange
		IncList
		InfinityQ
		ListQ
		MixedPolynomialQ
		MyApart
		MyCollectAll
		MyFreeQ
		MyMessage
		MyTogether
		NegExponent
		NormalizedQ
		PatternQ
		PointwisePlot
		PrintIt
		ProtectIt
		RationalQ
		RationalFunctionQ
		RationalPolynomialQ
		RealQ
		RealValuedQ
		ReplaceWith
		SameFormQ
		Second
		SetExclusion
		SetStateField
		StripPackage
		SubsetQ
		TableLookup
		Third
		ToCollection
		ToList
		UnprotectIt
		VariableQ
		ZeroQ
 *)


If [ TrueQ[ $VersionNumber >= 2.0 ],
     Unprotect[ ListQ ];
     Clear[ ListQ ];
     General::spell = "Possible spelling error: new symbol name \"`1`\" is similiar to existing symbols `2`.";
     General::spell1 = "Possible spelling error: new symbol name \"`1`\" is similiar to existing symbol \"`2`\".";
     Off[ General::spell ];
     Off[ General::spell1 ] ]


(*  B E G I N     P A C K A G E  *)

BeginPackage[ "SignalProcessing`Support`SupCode`" ]


(*  U S A G E     I N F O R M A T I O N  *)

AllSubsets::usage =
	"AllSubsets[set] returns a list of all subsets of set. \
	AllSubsets[set, function] applies function to every subset
	with more than one element. \
	For example, AllSubsets[ {v1,v2,v3}, Plus ] returns \
	{v1, v2, v3, v1 + v2, v1 + v2, v1 + v3, v1 + v2 + v3}. \
	Note that the subset does not contain the null set."

Arrow2D::usage =
	"Arrow2D[tail, plotwidth, plotheight] returns a graphics object \
	which has the shape of an arrow."

AssociateItem::usage =
	"AssociateItem[item, lookuplist, newlist] finds the location of \
	item in the lookuplist and returns the element of newlist in \
	that position. \
	If item is not is lookup list, Null is returned. \
	If item is a list, then a list of associations is returned."

CirclePS::usage =
	"CirclePS[r] and CirclePS[r, p] will return a graphics object, \
	a circle of radius r with plotstyle p. \
	In Mathematica 2.0, CirclePS makes use of the Circle primitive."

Combine::usage =
	"Combine[object, joindata] sets the value of object to the \
	joining of object with joindata; however, if object has no \
	value, then object is set to joindata."

ComplexQ::usage =
	"ComplexQ[z] returns True if z is a complex number, False otherwise."

ComplexTo2DCoord::usage =
	"ComplexTo2DCoord[x] returns the two-dimensional coordinate \
	corresponding to the complex form of x. \
	That is, a pair of values in the form { Re[x], Im[x] } is returned."

ComplexTo2DCoordList::usage =
	"ComplexTo2DCoordList[zlist] returns a list of coordinates \
	corresponding to the complex form of each element in the \
	zlist. \
	That is, zlist is mapped through ComplexTo2DCoord."

ConstantQ::usage =
	"ConstantQ[x] returns True if x is always constant. \
	If x is an atom, then x is constant it is a number or it x has \
	a Constant attribute. \
	If x is a function of the form f[arg1, arg2, ...], then x is \
	considered to be constant if arg1, arg2, ..., are constant."

ConstantTerm::usage =
	"ConstantTerm[expr, x] returns the constant term of the \
	expression expr."

Element::usage =
	"Element[list, n] returns the nth element of list. \
	Also, Element[l1, l2, ..., n] returns ln."

EmptyQ::usage =
	"EmptyQ[packet] returns True the packet of data is empty."

GenerateCode::usage =
	"GenerateCode[object] converts object to a string (if necessary) \
	and then evaluates it (translates it to Mathematica code)."

GeneratePattern::usage =
	"GeneratePattern[namestring] generates a symbol with the name \
	equal to namestring followed by an underscore, which makes the \
	object be a pattern which can be used for pattern matching. \
	For example, GeneratePattern[\"a\"] yields the pattern (a_)."

GenerateSymbol::usage =
	"GenerateSymbol[namestring], GenerateSymbol[namestring, trailer], \
	and GenerateSymbol[namestring, trailer, header], generate a \
	symbol by concatenating header, namestring, and trailer."

GetAllExponents::usage =
	"GetAllExponents[expr, x] returns all exponents of the \
	term x in the expression expr. \
	GetAllExponents[z^3 + 2 z^6, z] returns {6, 3}."

GetAllFactors::usage =
	"GetAllFactors[expr, x] returns all factors of the \
	term x in the expression expr. \
	GetAllFactors[z^3 + 2 z^6, z] returns {1, 2}. \
	GetAllFactors[z^-3 + 2 z^-6, z] returns {1/2, 1}."

GetOperatorVariables::usage =
	"GetOperatorVariables[op] returns the variable(s) in the \
	parameterized operator op. \
	By default, GetOperatorVariables[ op[par1, par2, ...] ] \
	returns the first parameter, par1."

GetRoot::usage =
	"GetRoot[rule] extracts the value from an expression like \
	{z -> 0.}, which is 0. in this case."

GetRootList::usage =
	"GetRootList[p, x] returns a list of the approximate numerical roots \
	of expression p, a function of x, with infinite roots removed. \ 
	GetRootList[p, x, filter] applies filter to the list of roots \
	returned by the Solve function (defaults to N)."

GetStateField::usage =
	"GetStateField[state, field] returns the value of the slot \
	field n the list state."

GetVariables::usage =
	"GetVariables[expr] returns a list of all of the variables in \
	the expression expr. \
	See VariableQ for the definition of a variable."

GetValue::usage =
	"GetValue[f[n], n, n0] finds the numeric value of f[n] at n = n0 \
	and GetValue[f[n1,n2], {n1,n2}, {n01, n02}] finds the numeric \
	value of f[n1,n2] at n1 = n01 and n2 = n02. \
	When the first argument has the variables embedded in it, \
	two arguments are sometimes enough:  GetValue[ object, n0 ]. \
	This is true when the object is an abstract signal."

HasAttributes::usage =
	"HasAttributes[symbol, attribute1, attribute2, ...] returns True \
	if the evaluation of symbol is another symbol and the attributes \
	to be checked are a subset of the attributes of this other symbol. \
	HasAttributes[Plus, {Listable, Orderless}] would return True."

ImaginaryQ::usage =
	"ImaginaryQ[z] returns True if z is a number whose real part is zero."

InRange::usage =
	"InRange[a, b, c, amin, cmax, leftcompare, rightcompare] returns \
	True if b in between a and c. \
	The inclusiveness of the interval a to c is determined by the \
	arguments leftcompare and rightcompare, each of which defaults to \
	LessEqual. \
	So, InRange[a, b, c] returns True if a <= b <= c. \
	Non-numeric values, like Infinity, can be used for amin and cmax, \
	which default to -Infinity and +Infinity, respectively."

IncList::usage =
	"IncList[list, start, end] will increment the first element of list. \
	If this becomes greater than the first element of end, \
	then the next element of list will be incremented, and so forth. \
	For example, IncList[{9,0,0}, {0,0,0}, {10,10,10}] would return \
	{0,1,0} and IncList[{9,9,9}, {0,0,0}, {10,10,10}] would return \
	{10,10,10}. \
	This is useful when enumerating values over an \
	n-dimensional rectangular prism."

InfinityQ::usage =
	"InfinityQ[a] will return True if a is Infinity, -Infinity, \
	ComplexInfinity, DirectedInfinity[], or DirectedInfinity[r]."

ListQ::usage =
	"ListQ[expr] gives True if expr is a List, and False otherwise."

MixedPolynomialQ::usage =
	"MixedPolynomialQ[p] and MixedPolynomialQ[p,x] return \
	return True if p is a polynomial in negative and positive \
	(mixed) powers of x.  \
	Note that rational numbers like 5/6 and 1 are polynomials. \
	MixedPolynomialQ[x + x^-1, x] is True."

MyApart::usage =
	"MyApart[ rational_polynomial, x ] decomposes the rational \
	polynomial into a sum of fractions whose numerators are of \
	the form (x + b)^n where b is a constant and n is an integer. \
	MyApart[rational_polynomial, x, filter] specifies a filter to be \
	placed on the output of the Solve command used to root the \
	denominator:  Identity for rational and N for real-valued roots. \
	The default is Identity. \
	In Mathematica 1.2, MyApart is about 25 times slower than Apart."

MyCollectAll::usage =
	"MyCollectAll[ expression, var ] attempts to collect all \
	subexpressions of expression in terms of var."

MyFreeQ::usage =
	"MyFreeQ[expr, form], when form is not a list, yields True if no \
	subexpression in expr matches form. \
	If form is a list, then True is returned if expr is free of \
	each element of form. \
	This is similar to MyFreeQ[expr, form1, form2, ...] which expands to \
	MyFreeQ[expr, form1] and MyFreeQ[expr, form2] and ...."

MyMessage::usage =
	"MyMessage[message-label, return-value, arg1, arg2, ...] first calls \
	Message[message-label, arg1, arg2, ...] and then returns return-value."

MyTogether::usage =
	"MyTogether[x] puts the expression x over a common denominator. \
	This is the same as Together[x] without the effect of Cancel. \
	This function is compatible with DFT expressions."

NegExponent::usage =
	"NegExponent[poly, x] returns the maximum exponent of x^-1."

NormalizedQ::usage =
	"NormalizedQ[e,x] gives True if the constant term is one \
	or zero, and gives False otherwise."

NullPlot::usage =
	"NullPlot is a 2-d graphics object which only contains the origin."

PatternQ::usage =
	"PatternQ[expr] returns True if the head of expr is Pattern."

PointwisePlot::usage =
	"PointwisePlot[coordlist, text] and \
	PointwisePlot[coordlist, text, multiplicitytext] \
	will plot the coordinates in coordlist as text \
	for 2-D and 3-D graphics. \
	An optional fourth argument specifies the size of the font to use. \
	For multiple occurrences of the same coordinate,\
	the object multiplicitytext is displayed. \
	The last two arguments are usually symbols, numbers, or \
	FontForm objects."

PrintIt::usage =
	"PrintIt[graphics, printer] will print out graphics on a printer. \
	If the printer is not specified, the default printer is used."

ProtectIt::usage =
	"ProtectIt[expr] evaluates expr. \
	If it evaluates to a symbol, that symbol will be write protected. \
	Rules can be written for that symbol, \
	but values can no longer be assigned to it."

RationalQ::usage =
	"RationalQ[m] returns True if m is a rational number. \
	If m is an integer, then this function also return True, \
	since the set of integers are a subset of rationals."

RationalFunctionQ::usage =
	"RationalFunctionQ[f,x] returns True if expression f is of the form \
	f = g(x) / h(x), where h(x) depends on x but g(x) does not have \
	to depend on x. \
	For example, 1 / ( x + 1 ) is a rational function \
	in x but x^3 + x^2 + x / ( x + 1) is not."

RationalPolynomialQ::usage =
	"RationalPolynomialQ[p] and RationalPolynomialQ[p,x] return \
	True if p is a rational polynomial in x. \
	Note that rational numbers like 5/6 and 1 are also \
	rational polynomials."

RealQ::usage =
	"RealQ[z] returns True if z is a floating-point number \
	(has a head of Real), False otherwise.  See RealValuedQ."

RealValuedQ::usage =
	"RealValuedQ[z] gives True if z is a number whose imaginary \
	component is 0, and gives False otherwise.  See RealQ."

ReplaceWith::usage =
	"ReplaceWith[oldexpr, newexpr] is a generalized way to specify \
	a substitution when the substitution may be either atomic \
	and a list of substitutions."

SameFormQ::usage =
	"SameFormQ[pattern, expr1, expr2, ...] returns True if every \
	expression matches pattern via MatchQ. \
	Once an expression does not match, \
	this function immediately returns False."

Second::usage =
	"Second[list] returns the second element of list."

SetExclusion::usage =
	"SetExclusion[set1, set2, ...] returns a set equal to the union \
	of the sets minus the intersection of the sets."

SetStateField::usage =
	"SetStateField[state, field, value] will return a new state, \
	which is a copy of the list state except that the value of \
	the slot field will be equal to value."

SPfunctions::usage =
	"SPfunctions maintains a current list of those new routines \
	that have been loaded from the signal processing packages."

SPLessGreaterRules::usage =
	"SPLessGreaterRules are a collection of rules for simplifying \
	expressions involving inequalities."

SPoperators::usage =
	"SPoperators maintains a current list of the new mathematical \
	operators that have been loaded from the signal processing packages."

SPsignals::usage =
	"SPsignals maintains a current list of those new signals \
	(mathematical functions) that have been loaded from the \
	signal processing packages."

SPSimplificationRules::usage =
	"SPSimplificationRules are a collection of rules common in \
	simplifying formulas encountered in signal processing. \
	These rules require too much overhead to encode them directly \
	into Mathematica."

StripPackage::usage =
	"StripPackage[symbol] returns the symbol (as a string) after its \
	context has been removed."

SubsetQ::usage =
	"SubsetQ[set1, set2, set3,  ...] returns True if set1 is a subset \
	of set2 and set2 is a subset of set3, etc."

TableLookup::usage =
	"TableLookup[index, hlist, len, val] returns hlist[[index]] \
	if index is between 1 and len, inclusive; otherwise, val is \
	returned."

Third::usage =
	"Third[list] returns the third element of list."

ToCollection::usage =
	"ToCollection[expr] strips the head off of arg and returns \
	the argument of expr as a collection. \
	ToCollection returns an object that is a sequence, \
	which is represented in Mathematica 1.2 as (a1, a2, ...) \
	and in Mathematica 2.0 as Sequence[a1, a2, ...]. \
	So, it provides a unified way of generating collections (sequences)."

ToList::usage =
	"ToList[arg] returns arg if arg is a list. \
	Otherwise, List[arg] is returned. \
	ToList[arg1, arg2, ...] returns List[arg1, arg2, ...]."

UnprotectIt::usage =
	"UnprotectIt[expr] evaluates expr. \
	If it evaluates to a symbol, \
	write protection will be removed for that symbol."

VariableQ::usage =
	"VariableQ[x] returns True if x is a symbol that \
	(1) does not have a numerical value associated with it and
	(2) does not have its Constant attribute enabled. \
	Pi fails the first test, so it is not considered a variable. \
	A variable can also have the form of C[n] where n is an integer \
	and C is a symbol whose Constant attribute is enabled."

ZeroQ::usage =
	"ZeroQ[x] returns True if x is 0 or 0.0"

(*  E N D     U S A G E     I N F O R M A T I O N  *)


Begin["`Private`"]


(*  A L T E R     E X I S T I N G    F U N C T I O N S  *)

(*  And     *)
Unprotect[And]
And/: Simplify[And[a1_, args__]] := Apply[And, Union[{a1, args}]]
Protect[And]

(*  ClearAttributes  *)
Unprotect[ClearAttributes]
SetAttributes[ClearAttributes, HoldFirst]
Protect[ClearAttributes]

(*  Det     *)
Unprotect[Det]
Det[x_?NumberQ] := x
Protect[Det]

(*  Dot     *)
Unprotect[Dot]
Dot[x_?NumberQ, y_?NumberQ] := x y
Protect[Dot]

(*  SetAttributes  *)
Unprotect[SetAttributes]
SetAttributes[SetAttributes, HoldFirst]
Protect[SetAttributes]

(*  TeXForm  *)
Unprotect[Re, Im]
Im/: Format[ Im[x_], TeXForm ] := StringForm["\\Im{m}(``)", x]
Re/: Format[ Re[x_], TeXForm ] := StringForm["\\Re{e}(``)", x]
Protect[Re, Im]



(*  S I M P L I F I C A T I O N     R U L E S  *)

numericq[x_] := NumberQ[ N[x] ]

MinMaxRules = {
	Min[a_, a_] :> a,
	Min[t1_?numericq, t2_?numericq] :> If [ N[t1] < N[t2], t1, t2 ],
	Max[a_, a_] :> a,
	Max[t1_?numericq, t2_?numericq] :> If [ N[t1] > N[t2], t1, t2 ]
}

LessRules = {
	Less[Max[a_, b__], a_] :> Less[Max[b], a],
	LessEqual[Max[a_, b__], a_] :> LessEqual[Max[b], a],
	Less[Times[-1, b_], 0] :> Greater[b, 0],
	LessEqual[Times[-1, b_], 0] :> GreaterEqual[b, 0]
}

GreaterRules = {
	Greater[Min[a_, b__], a_] :> Greater[Min[b], a],
	GreaterEqual[Min[a_, b__], a_] :> GreaterEqual[Min[b], a],
	Greater[Times[-1, b_], 0] :> Less[b, 0],
	GreaterEqual[Times[-1, b_], 0] :> LessEqual[b, 0]
}

AbsRules = {
	Abs[- a_] :> Abs[a],
	Abs[x_?NumberQ y_] :> Abs[x] Abs[y],
	Re[Abs[a_]] :> Abs[a],
	Im[Abs[a_]] :> 0,
	Abs[Abs[a_]] :> Abs[a]
}

ReImRules = {
	Re[- a_] :> - Re[a],
	Im[- a_] :> - Im[a],

	Re[Im[a_]] :> Im[a],
	Im[Re[a_]] :> 0,

	Conjugate[Conjugate[x_]] :> x,
	Conjugate[x_?Positive] :> x,
	Conjugate[x_?Negative] :> x
}

TimesRules = {
	a_^k_. b_^k_. :> 1 /; ( a == 1/b )
}

ExpLogRules = {
	Exp[c_. Log[b_]] :> b^c,
	base_^(c_. Log[base_, b_]) :> b^c,
	Log[c_. Exp[b_]] :> Log[c] + b,
	Log[base_, c_. base_^b_] :> Log[base, c] + b,
	Erf[-a_] :> -Erf[a],
	Exp[ a_. Complex[0, b_] Pi ] :> Exp[ Mod[a b, 2] I Pi ] /;
		RationalQ[a b] && ( (a b < 0) || (a b >= 2) )
}

SPSimplificationRules =
	Join[MinMaxRules, AbsRules, TimesRules, ReImRules, ExpLogRules]

SPLessGreaterRules = Join[LessRules, GreaterRules]


(*  M E S S A G E S  *)

PointwisePlot::invalid = "Null coordinate list passed."


(*  G L O B A L S  *)

NullPlot := Graphics [ Point[{0, 0}], DisplayFunction -> Identity ]


(*  F U N C T I O N S  *)

(*  AllSubsets  *)
subset[ list_, n_, len_, h_] :=
	Block [ {i, left = 1, leftseq, right = n - 1, result = {}},
		While [ right < len,
		        leftseq = Take[list, {left, right}];
		        For [ i = right + 1, i <= len, i++,
		              AppendTo[ result,
	              			Apply[ h, Append[ leftseq,
						          list[[i]] ] ] ] ];
		        ++left;
		        ++right ];
		result ]	              

AllSubsets[ list_ ] := AllSubsets[list, List]

AllSubsets[ {v1_, v2_}, h_ ] := { v1, v2, h[v1, v2] }

AllSubsets[ {v1_, v2_, v3_}, h_ ] :=
	{ v1, v2, v3, h[v1, v2], h[v1, v3], h[v2, v3], h[v1, v2, v3] }

AllSubsets[ list_List, h_ ] :=
	Block [	{j, len = Length[list], result = list},
		For [ j = 2, j < len, j++,
		      result = result ~Join~ subset[list, j, len, h] ];
		AppendTo[result, Apply[h, list] ] ]

(*  Arrow2D  *)
Arrow2D[tail_, plotwidth_, plotheight_:1] :=
	Block [	{arrowleft, arrowright, head, xoffset, yoffset},
		xoffset = 0.1 plotheight;
		yoffset = 0.3 plotheight;
		head = tail + { 0, plotheight };
		arrowleft = head - { xoffset, yoffset };
		arrowright = head + { xoffset, - yoffset };
		Graphics[ Line[{tail, head, arrowleft, arrowright, head}] ] ]

(*  AssociateItem  *)
AssociateItem[item_, lookuptable_, assoctable_] :=
	Map[Function[var, AssociateItem[var, lookuptable, assoctable]], item] /;
	ListQ[item]
AssociateItem[item_, lookuptable_, assoctable_] :=
	If [ MemberQ[lookuptable, item],
	     assoctable [[ ToCollection[ToCollection[Position[lookuptable, item] ]] ]] ] /;
	! ListQ[item]

(*  CirclePS  *)
If [ TrueQ[$VersionNumber >= 2.0],
     CirclePS[r_] := Graphics[ Circle[{0, 0}, r] ];
     CirclePS[r_,p_] := Graphics[ { p, Circle[{0, 0}, r] } ],

     CirclePS[r_] := ParametricPlot[ { r Cos[theta], r Sin[theta] },
			             { theta, 0, 2 Pi},
			             DisplayFunction -> Identity ];
     CirclePS[r_,p_] := ParametricPlot[ { r Cos[theta], r Sin[theta] },
					{ theta, 0, 2 Pi},
					DisplayFunction -> Identity,
					PlotStyle -> p ] ]

(*  Combine  *)
SetAttributes[Combine, {HoldFirst}]
Combine[object_, joindata_] :=
	If [ ValueQ[object],
	     object = Sort[object ~Join~ joindata],
	     object = joindata ]

(*  ComplexQ  *)
ComplexQ[z_] := NumberQ[z] && SameQ[Head[z], Complex]

(*  ComplexTo2DCoord and ComplexTo2DCoordList  *)
ComplexTo2DCoord[z_] := { Re[z], Im[z] }
  
ComplexTo2DCoordList[zlist_] := Map[ ComplexTo2DCoord, zlist ]

(*  ConstantQ  *)
ConstantQ[x_?AtomQ] := NumberQ[x] || HasAttributes[x, Constant]
ConstantQ[f_[x__]] := Apply[And, Map[ConstantQ, List[x]]]

(*  ConstantTerm  *)
ConstantTerm[expr_, z_:Global`x] :=
	Block [ {nonpropterms},
		keepconstants[e_] := If [ MyFreeQ[e,z], e, 0 ];
		nonpropterms = Coefficient[expr, z, 0];
		If [ MyFreeQ[nonpropterms, z],
		     nonpropterms,
		     Map[keepconstants, nonpropterms] ] ]

(*  Element  *)
Element[h_[args__], i_?IntegerQ] := h[args] [[i]]
Element[x_?AtomQ, i_] := x
Element[x__, i_?IntegerQ] := ToList[x] [[i]]

(*  EmptyQ  *)
EmptyQ[x_?AtomQ] := False
EmptyQ[h_[]] := True
EmptyQ[h_[values__]] := False

(*  GenerateCode  *)
GenerateCode[code_] := ToExpression[ToString[code]]

(*  GeneratePattern  *)
GeneratePattern[name_] := GenerateSymbol[name, "_"]

(*  GenerateSymbol  *)
GenerateSymbol[name_] := GenerateCode[name]
GenerateSymbol[name_, trailer_] :=
	GenerateCode[StringForm["````", name, trailer]]
GenerateSymbol[name_, trailer_, header_] :=
	GenerateCode[StringForm["``````", header, name, trailer]]

(*  GetAllExponents and GetAllFactors  *)
depthfirstsearch[expr_, lhs_, rule_] :=
	Block [ {cur, i, len},

		If [ AtomQ[expr],
		     If [ MatchQ[expr, lhs],
			  PrependTo[list, Replace[expr, rule]] ],

		     len = Length[expr];
		     For [ i = 1, i <= len, i++,
			   cur = expr[[i]];
			   If [ MatchQ[cur, lhs],
				PrependTo[list, Replace[cur, rule]],
				depthfirstsearch[cur, lhs, rule] ] ] ];

		Null ]

depthdriver[expr_, lhs_, rule_] :=
	Block [	{},
		list = {};
		depthfirstsearch[expr, lhs, rule];
		list ]

GetAllExponents[ expr_, z_ ] :=
	depthdriver[expr, (c_. z^n_.), (c_. z^n_. :> n) ]

GetAllFactors[ expr_, z_ ] :=
	depthdriver[expr, (c_. z^n_.), (c_. z^n_. :> c^Sign[n]) ] /.
	( Sign[x_] :> 1 )

(*  GetOperatorVariables  *)
GetOperatorVariables[ h_[var_, rest___] ] := var

(*  GetRoot  *)
GetRoot[{}] := {}				(* no roots *)
GetRoot[rule_] := Second[First[rule]]

(*  GetRootList  *)
GetRootList[p_, x_, filter_:N] :=
	Select[ Map[ GetRoot, filter[ Solve[ p == 0, x ] ] ],
		(! MatchQ[#1, DirectedInfinity[___]])& ]

(*  GetStateField  *)
GetStateField[state_List, field_] := state[[field]]

(*  GetVariables  *)
extractrules = { f_[x__][y__][z__] :> bogus[x, y, z],
		 f_[x__][y__] :> bogus[x, y],
		 (x_ -> y_) :> {},
		 (x_ :> y_) :> {} }

GetVariables[x_] :=
	Union[ Select[ Level[x /. extractrules, Infinity], VariableQ ] ]

(*  GetValue  *)
GetValue[f_, n_Symbol, n0_] :=
	Block [	{value},
		value = N [ f /. n -> n0 ];
		If [ NumberQ[value],
		     value,
		     N [ Limit[f, n -> n0] ] ] ]

GetValue[f_, {n1_Symbol, n2_Symbol}, {n01_, n02_}] :=
	Block [	{value},
		value = N [ f /. { n1 -> n01, n2 -> n02 } ];
		If [ NumberQ[value],
		     value,
		     N[ Limit[ Limit[f, n1 -> n01], n2 -> n02] ] ] ]

(*  HasAttributes  *)
HasAttributes[symbol_, attrib1_, attribs__] :=
	HasAttributes[symbol, {attrib1, attribs}]

HasAttributes[symbol_Symbol, attrib_] :=
	Block [	{attributes, protected},
		attributes = Attributes[Attributes];
		Unprotect[Attributes];
		ClearAttributes[Attributes, {HoldFirst, HoldAll, HoldRest}];
		protected = If [ AtomQ[attrib],
				 MemberQ[Attributes[symbol], attrib],
				 SubsetQ[attrib, Attributes[symbol]] ];
		SetAttributes[Attributes, attributes];
		protected ]

(*  ImaginaryQ  *)
ImaginaryQ[z_] := NumberQ[z] && ZeroQ[Re[z]]

(*  InRange, function will be automatically threaded if a,b,c are not atoms  *)
SetAttributes[MyInRange, Listable]

InRange[a_, b_, c_, amin_:-Infinity, cmax_:Infinity, leftcompare_:LessEqual, rightcompare_:LessEqual ] :=
	Apply[And,
	      ToList[MyInRange[a, b, c, amin, cmax, leftcompare, rightcompare]]]

MyInRange[a_, b_, c_, amin_, cmax_, leftcompare_, rightcompare_] :=
	Which [ SameQ[a, amin] && SameQ[c, cmax],
		  True,
		SameQ[a, amin],
		  SameQ[b, amin] || rightcompare[b, c],
		SameQ[c, cmax],
		  SameQ[b, cmax] || leftcompare[a, b],
		True,
		  leftcompare[a, b] && rightcompare[b, c] ]


(*  IncList  *)
IncList[list_, start_, end_] :=
	Block [	{incflag, newlist},
		newdigit[d_, s_, e_] :=
			Which [ incflag && SameQ[d + 1, e],
				  s,
				incflag,
				  incflag = False;
				  d + 1,
				True,
				  d ];
		incflag = True;
		newlist = newdigit[list, start, end];
		If [ SameQ[newlist, start], end, newlist ] ]

SetAttributes[newdigit, Listable]

(*  InfinityQ  *)
InfinityQ[e_List] := Apply[And, Map[InfinityQ, e]]
InfinityQ[DirectedInfinity[]] := True 
InfinityQ[DirectedInfinity[r_]] := True
InfinityQ[a_] := False

(*  ListQ--  it is an undocumented primitive in Mma 2.0+	*)
(*           in 2.0, it does not always return True or False	*)
ListQ[object_] := SameQ[Head[object], List]

(*  MixedPolynomialQ  *)
twosided[ c_. z_^r_., z_ ] := FreeQ[c, z] && IntegerQ[r]
twosided[ c_, z_ ] := FreeQ[c, z]

MixedPolynomialQ[c_] := MixedPolynomialQ[c, Global`x]

MixedPolynomialQ[x_?AtomQ, z_] := True
MixedPolynomialQ[Plus[a_, b__], z_] := Apply[And, Map[twosided[#1, z]&, {a, b}]]
MixedPolynomialQ[x_, z_] := twosided[x, z]

(*  MyApart --  kludge around the way Apart does partial fractions *)
(*		Root denominator and replace roots with symbols	   *)
MyApart[ratpoly_, x_, filter_:Identity] :=
	Block [	{apart, denom, denomfactored, normfact, numer,
		 rootlist, rootmult, rules},

		numer = Numerator[ratpoly];
		denom = Denominator[ratpoly];
		normfact = Last[ CoefficientList[denom, x] ];
		numer /= normfact;
		denom /= normfact;
		rootlist = Sort[ GetRootList[denom, x, filter] ];
		{ denomfactored, rules } = multiplicityform[rootlist, x];
		apart = Apart[numer / denomfactored, x];

		apart /. rules ]

multiplicityform[ roots_, x_ ] :=
	Block [	{count = 1, cur, denom = 1, i, last,
		 length, sublist = {}, sym = 1},
		Clear[localvar];	(* localvar is global to package *)
		length = Length[roots];
		last = First[roots];
		For [ i = 2, i <= length, i++,
		      cur = roots[[i]];
		      If [ SameQ[ cur, last ],
			   count++,
			   denom *= (x - localvar[sym])^count;
			     PrependTo[ sublist, localvar[sym] -> last ];
			     sym++;
			     count = 1 ];
		      last = cur ];

		denom *= (x - localvar[sym])^count;
		PrependTo[ sublist, localvar[sym] -> last ];

		{ denom, sublist } ]

(*  MyCollectAll  *)
MyCollectAll[ Plus[term1_, terms__], x_ ] := Collect[ Plus[term1, terms], x ]
MyCollectAll[ h_[a_], x_ ] := h[ MyCollectAll[a,x] ]
MyCollectAll[ h_[a_, b__], x_ ] := Apply[h, Map[MyCollectAll[#1, x]&, {a, b}]]
MyCollectAll[ a_, x_ ] := a

(*  MyFreeQ  *)
MyFreeQ[expr_, {form_}] := FreeQ[expr, form]
MyFreeQ[expr_, {form1_, forms__}] := FreeQ[expr, form1] && MyFreeQ[expr, forms]
MyFreeQ[expr_, form_] := FreeQ[expr, form]
MyFreeQ[expr_, form1_, forms__] := FreeQ[expr, form1] && MyFreeQ[expr, forms]

(*  MyMessage  *)
SetAttributes[MyMessage, HoldFirst]
MyMessage[message_, return_] :=
	Block [	{},
		Message[message];
		return ]
MyMessage[message_, return_, args__] :=
	Block [	{},
		Message[message, args];
		return ]

(*  MyTogether  *)
MyTogether[ Plus[a_, b_, c__] ] := MyTogether[ Plus[MyTogether[Plus[a, b]], c] ]
MyTogether[ Plus[a_, b_] ] := MyTogether[ Numerator[a], Denominator[a], Numerator[b], Denominator[b] ]
MyTogether[ a_ ] := a

MyTogether[ a_, b_, c_, b_ ] := ( a + c ) / b
MyTogether[ a_, b_, c_, b_ d_ ] := ( a d + c ) / ( b d )
MyTogether[ a_, b_ d_, c_, d_ ] := ( a + b c ) / ( b d )
MyTogether[ a_, b_, c_, d_ ] := ( a d + b c ) / ( b d )

(*  NegExponent  *)
NegExponent[p_, x_:Global`x] := Exponent[p /. x -> x^-1, x]

(*  NormalizedQ  *)
NormalizedQ[e_, z_] :=
	Block [	{leadingcoef},
		leadingcoef = ConstantTerm[Expand[e], z];
		TrueQ[( leadingcoef == 0 ) || ( leadingcoef == 1 )] ]

(*  PatternQ  *)
PatternQ[expr_] := SameQ[Head[expr], Pattern]

(*  PointwisePlot  *)

PointwisePlot[coordlist_, singtext_] :=
	PointwisePlot[coordlist, singtext, singtext]

(*  plots each unique set of coordinates.  multiple occurrences of the   *)
(*    same coordinate are plotted as <text>(n), where n is the number of *)
(*    occurrences.  First, the coordinate list is sorted.  A Null is     *)
(*    appended because the scanning function compares the current        *)
(*    coordinate with the last, so that Null forces the last coordinate  *)
(*    to be processed.  After the pointwiseplot graphics commands are    *)
(*    built up, the resulting plot is returned as a graphics object.     *)
(*    The point size of the text defaults to 18.  Supported font sizes   *)
(*    are 10, 12, 14, 18, 20, 24, ...					 *)

PointwisePlot[coordlist_, singtext_, multtext_, fontsize_:18] :=
	Block [	{counter = 1, text, lastcoord = Null,
		 pointwiseplot = {}, ptsize, str},
		ptsize = Round[fontsize];
		Scan [ Function[ coord,
			 Which [ SameQ[lastcoord, Null],   (* initial cond.   *)
				   counter = 1;
				   lastcoord = coord,
				 SameQ[coord, lastcoord],  (* multiple occur. *)
				   ++counter,
				 True,				   (* plot it *)
				   str = If [ SameQ[counter, 1],
				   	      singtext,
					      multtext ];
				   text = If [ TrueQ[$VersionNumber >= 2.0],
					       FontForm[str, {"Bold", ptsize}],
					       FontForm[str, "Bold", ptsize] ];
				   AppendTo[ pointwiseplot,
					     Text[text, lastcoord] ];
				   counter = 1;
				   lastcoord = coord ] ],
		       Append[Sort[coordlist], Null] ];
		Graphics[pointwiseplot] ] /;
	! EmptyQ[coordlist]

PointwisePlot[coordlist_, singtext_, multtext_] :=
	MyMessage[PointwisePlot::invalid, NullPlot] /;
	EmptyQ[coordlist] 

(*  PrintIt  *)
PrintIt[graphics_] :=
	Display["!psfix | lpr", graphics]

PrintIt[graphics_, printer_] :=
	Display[ToString[StringForm["!psfix | lpr -P``", printer]], graphics]

(*  ProtectIt  *)
ProtectIt[symbol_Symbol] := Apply[Protect, {symbol}]

(*  RationalQ  *)
RationalQ[z_Integer] := True
RationalQ[z_Rational] := True
RationalQ[z_] := False

(*  RationalFunctionQ  *)
RationalFunctionQ[f_, x_:Global`x] :=
	( ! SameQ[Head[f], Plus] ) && ( ! MyFreeQ[Denominator[f], x] )

(*  RationalPolynomialQ  *)
RationalPolynomialQ[p_] :=
	PolynomialQ[Numerator[p]] && PolynomialQ[Denominator[p]]
RationalPolynomialQ[p_, x_] :=
	PolynomialQ[Numerator[p], x] && PolynomialQ[Denominator[p], x]

(*  RealQ  *)
RealQ[z_] := SameQ[Head[z], Real]

(*  RealValuedQ  *)
RealValuedQ[z_] := NumberQ[z] && ZeroQ[Im[z]]

(*  ReplaceWith  *)
SetAttributes[ReplaceWith, {Listable}]
ReplaceWith[org_, val_] := org -> val

(*  SameFormQ  *)
SameFormQ[form_, expr_] := MatchQ[expr, form]
SameFormQ[form_, expr1_, expr__] := SameFormQ[form, expr1] && SameFormQ[form, expr]

(*  Second  *)
Unprotect[Second]
Second[x_] := x[[2]]
Protect[Second]

(*  SetExclusion  *)
SetExclusion[sets__] := Complement[Union[sets], Intersection[sets]]

(*  SetStateField  *)
SetStateField[state_List, field_, value_] :=
	Block [ {newstate},
		newstate = state;
		newstate[[field]] = value;
		newstate ]

(*  StripPackage  *)
StripPackage[symbol_Symbol] := StripPackage[ ToString[symbol] ]

StripPackage[symbol_String] :=
	Block [	{expandedstring, pos},
		expandedstring = Characters[symbol];
		pos = Position[expandedstring, "`"];
		If [ SameQ[pos, {}],
		     symbol,
		     Apply[ StringJoin,
			    Drop[expandedstring, Last[Last[pos]]] ] ] ]

(*  SubsetQ  *)
SubsetQ[x1_] := True
SubsetQ[x1_, x2_] :=
	Block [	{x1sorted},
		x1sorted = Sort[x1];
		SameQ[x1sorted, Intersection[x1sorted, x2]] ]
SubsetQ[x1_, x2_, x__] := SubsetQ[x1, x2] && SubsetQ[x2, x]

(*  TableLookup  *)
TableLookup[index_, table_, len_, val_] :=		(* multidimensional *)
	Which [ TrueQ[ Apply[Or, Map[InfinityQ, index]] ],
		  val,
		TrueQ[ InRange[1, index, len] ],
		  Apply[Part, {table} ~Join~ index],
		True,
		  val ] /;
	ListQ[index]

TableLookup[index_, table_, len_, val_] :=		(* one-dimensional  *)
	Which [ InfinityQ[index],
		  val,
		TrueQ[ 1 <= index <= len ],
		  table[[index]],
		True,
		  val ] /;
	( InfinityQ[index] || IntegerQ[index] ) && IntegerQ[len]

(*  Third  *)
Third[x_] := x[[3]]

(*  ToCollection  *)
ToCollection[x_?AtomQ] := x
ToCollection[h_[args___]] := args
ToCollection[a__] := a

(*  ToList  *)
ToList[] := {}
ToList[arg_List] := arg
ToList[arg_] := List[arg] /; ! SameQ[Head[arg], List]
ToList[arg1_, args__] := List[arg1, args]

(*  UnprotectIt  *)
UnprotectIt[symbol_Symbol] :=
	Block [	{attributes},
		attributes = Attributes[Unprotect];
		Unprotect[Unprotect];
		ClearAttributes[Unprotect, {HoldFirst, HoldAll, HoldRest}];
		Unprotect[symbol];
		SetAttributes[Unprotect, attributes] ]

(*  VariableQ  *)
VariableQ[x_Symbol] := ! ConstantQ[x]
VariableQ[x_[n_Integer]] := HasAttributes[x, Constant]
VariableQ[x_] := False

(*  ZeroQ  *)
ZeroQ[x_] := SameQ[x, 0] || SameQ[x, 0.0]


(*  E N D     P A C K A G E *)

End[]
EndPackage[]


If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ]


(*  H E L P     I N F O R M A T I O N  *)

Block [	{newfuns},
	newfuns =
	{ AllSubsets,		CirclePS,		Combine,
	  ComplexQ,		ComplexTo2DCoord,	ComplexTo2DCoordList,
	  ConstantQ,		ConstantTerm,		Element,
	  GenerateCode,		GenerateSymbol,		GetAllExponents,
	  GetAllFactors,	GetRoot,		GetRootList,
	  GetStateField,	GetValue,		GetVariables,
	  ImaginaryQ,		InRange,		IncList,
	  InfinityQ,		ListQ,			MixedPolynomialQ,
	  MyApart,		MyCollectAll,		MyFreeQ,
	  MyMessage,		MyTogether,		NegExponent,
	  NormalizedQ,		PointwisePlot,		PrintIt,
	  RationalFunctionQ,	RationalPolynomialQ,	RationalQ,
	  RealQ,		RealValuedQ,		ReplaceWith,
	  SameFormQ,		Second,			SetExclusion,
	  SetStateField,	StripPackage,		SubsetQ,
	  TableLookup,		Third,			ToCollection,
	  ToList,		VariableQ,		ZeroQ };
	Combine[ SPfunctions, newfuns ];
	Apply[ Protect, newfuns ] ]


(*  E N D I N G     M E S S A G E  *)

Print["Support module has been loaded."]

