(*  :Title:	Continuous Time Fourier Transform  *)

(*  :Authors:	Wally McClure, Brian Evans, James McClellan  *)

(*  :Summary:	*)

(*  :Context:	Signal Processing`Digital`Fourier`  *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1990-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*
    :History:	start -- February 22, 1990
		made into package -- April 18, 1990
		added Dialogue ability -- October, 1990
		added many more transform pairs -- February, 1991
 *)

(*  :Keywords:	Fourier transform, frequency response  *)

(*
    :Source:	{Signals and Systems} by A. V. Oppenheim and A. S. Willsky.
		  Prentice Hall.
		{The Fourier Transform and Its Applications} by R. Bracewell
 *)

(*  :Warning:	*)

(*  :Mathematica Version:  1.2 or 2.0  *)

(*
    :Limitation:   If the function to  be transformed does not match either
		   a specific Fourier transform or the LaPlace transform, a 
		   nasty expression is returned.
 *)

(*
    :Discussion: When loaded into Mathematica,  this package will allow the 
		 computation of  continuous  Fourier  and  inverse  Fourier
		 transforms.   It  turns  out  that in order to show a user
		 each step  of  transformation process, the transform rules
		 needed to be placed in a list of deferred (delayed) rules.

	MyFourier[f, t, w, s] computes  the  one-dimensional Fourier trans-
	form  of  f  using  t  as  the time variable and w as the frequency
	variable.

	MyInverseCTFT[f, w, t, s] computes  the   inverse   one-dimensional
	continuous-time Fourier transform using w as the frequency variable
	and t as the time variable.
 *)

(*
   :Functions:	CTFTransform
		InvCTFTransform
		MultiDCTFT
		MultiDInvCTFT
		MyFourier
		MyInverseCTFT
 *)


If [ TrueQ[ $VersionNumber >= 2.0 ],
     Off[ General::spell ];
     Off[ General::spell1 ] ];


(*  B E G I N     P A C K A G E  *)

BeginPackage[ "SignalProcessing`Analog`Fourier`",
	      "SignalProcessing`Analog`LaPlace`",
	      "SignalProcessing`Analog`InvLaPlace`",
	      "SignalProcessing`Analog`LSupport`",
	      "SignalProcessing`Digital`DSupport`",
	      "SignalProcessing`Support`TransSupport`",
	      "SignalProcessing`Support`ROC`",
	      "SignalProcessing`Support`SigProc`",
	      "SignalProcessing`Support`SupCode`" ]


(*  U S A G E     I N F O R M A T I O N  *)

CTFTData::usage =
	"CTFTData is the data-tag for the CTFT."

CTFTransform::usage =
	"CTFTransform[f, t] or CTFTransform[f, t, w] computes the \
	continuous-time Fourier transform of f.  It has time argument(s) t \
	and frequency argument(s) w."

InvCTFTransform::usage =
	"InvCTFTransform[f, w, t] computes the inverse continuous time \
	Fourier transform of f with frequency arguments w and time \
	arguments t."

(*  E N D     U S A G E     I N F O R M A T I O N  *)


Begin[ "`Private`" ]


(*  Printed forms of local objects so that Dialogue -> All works neatly  *)

Format[ myinvFourier[ x_, w_, t_, s_, oplist_, state_ ] ] :=
	SequenceForm[ ColumnForm[{"F" Superscript[-1],
				  "  " ~StringJoin~ ToString[w]}],
		      { x } ]

Format[ myFourier[ x_, t_, w_, s_, oplist_, state_ ] ] :=
	SequenceForm[ ColumnForm[{"F",
				  "  " ~StringJoin~ ToString[t]}],
		      { x } ]

Format[ derivative[x_, w_Symbol, k_] ] := 
	SequenceForm[ ColumnForm[{"D" Superscript[k],
				  "  " ~StringJoin~ ToString[w]}],
		      { x } ]

Format[	substitute[x_, w_, wnew_] ] :=
	SequenceForm[ {x}, Subscript[w -> wnew] ]

Format[	CTFTtoL[x_, w_, s_] ] :=
	SequenceForm[ {x}, Subscript[w -> s / I] ]


(*  Extension of operators  *)
Unprotect[FT, InvFT]
FT/: TheFunction[ FT[t_, w_] [f_] ] := CTFTransform[f, t, w]
InvFT/: TheFunction[ InvFT[w_, t_] [f_] ] := InvCTFTransform[f, w, t]
Protect[FT, InvFT]


(*  Options for the CTFT  *)
CTFTransform/: Options[ CTFTransform ] :=
	{ Dialogue -> False } ~Join~ Options[ LaPlace ]

(*  Extension of TheFunction operator  *)
CTFTData/: TheFunction[ CTFTData[x_, var_] ] := x

(*  Magnitude/Phase plot of a CTFT object *)
mppoptions = {	Domain -> Continuous, DomainScale -> Linear,
		MagRangeScale -> Log, PhaseRangeScale -> Degree,
		PlotRange -> All }

CTFTData/: MagPhasePlot[ CTFTData[trans_, vars_], rest___ ] :=
	Apply [	MagPhasePlot, { trans, rest } ~Join~ mppoptions ]


(*  State definition for both rule bases *)

expandfield = 1			(* apply expand to the expression	*)
laplacefield = 2		(* all else has failed so try Laplace	*)


(*  B E G I N     F O R W A R D     F O U R I E R     T R A N S F O R M  *)


(*  CTFTransform  *)
CTFTransform[ f_ ] :=
	Message[ Transform::novariables, "t (time)", GetVariables[f] ]
CTFTransform[ f_, t_ ] :=
	CTFTransform[ f, t, DummyVariables[Length[t], Global`w] ]
CTFTransform[ f_, t_, w_, options___ ] :=
	Block[ {op, svars, trans, vars},
		vars = Which [ Length[t] > Length[w],
				 DummyVariables[Length[t], Global`w],
			       Length[t] < Length[w],
				 Take[w, Length[t]],
			       True,
				 w ];
		svars = DummyVariables[Length[t], s];

		op = ToList[options] ~Join~ Options[CTFTransform];
		trans = If [ Length[t] > 1, 
			     MultiDCTFT[f, t, vars, svars, op],
			     MyFourier[f, t, vars, svars, op] ]; 	

		If [ InformUserQ[Replace[Dialogue, op]],
		     Scan[explain, trans, Infinity] ];

		CTFTData[ FourierSimplify[trans], FVariables[vars] ] ]

(*  Multidimensional continuous-time Fourier transform  *)
MultiDCTFT[f1_, t_, w_, s_, op_] :=
	Block [	{i, func = f1, length},
		length = Length[t];
		For [ i = 1, i <= length, i++,
		      func = MyFourier[ func, t[[i]], w[[i]], s[[i]], op ] ];
		func ]

(*  One-dimensional continuous-time Fourier transform  *)
MyFourier[ f_, t_, w_, s_, oplist_ ] :=
	Block [	{newexpr = myFourier[f, t, w, s, FixUp[oplist], {True, True}],
		 oldexpr = Null, trace },
		trace = SameQ[ Replace[Dialogue, oplist], All];
		Off[ Power::infy ];
		While [ ! SameQ[newexpr, oldexpr],
			If [ trace, Print[newexpr]; Print["which becomes"] ];
			oldexpr = newexpr;
			newexpr = MapAll[transform, oldexpr] ];

		oldexpr = TransformFixUp[ oldexpr, t, w, oplist, myFourier,
					  False, CTFTransform, Null, Null ];

		newexpr = posttransform[oldexpr];
		While [ ! SameQ[newexpr, oldexpr],
			If [ trace, Print[newexpr]; Print["which becomes"] ];
			oldexpr = newexpr;
			newexpr = posttransform[oldexpr] ];
		If [ ! FreeQ[{newexpr}, s],
		     newexpr = ( newexpr /. s -> I w ) ];
		On[ Power::infy ];
		If [ trace, Print[newexpr] ];
		If [ SameQ[Head[newexpr], LTransData],
		     TheFunction[newexpr],
		     newexpr ] ]

(*  Supporting Functions  *)

explain[ myFourier[x_, t_, rest__]] :=
	Message[ Transform::incomplete, "forward Fourier transform", x, t ]

(*  FixUp  *)
FixUp[ op_ ] := { Dialogue -> Replace[Dialogue, op],
		  Simplify -> Replace[Simplify, op] }

LtoCTFT[ SignalProcessing`Analog`LaPlace`Private`mylaplace[x_, lrest__], t_, w_, s_, rest__ ] :=
	CTFTtoL[ myFourier[x, t, w, s, rest], w, s ]
LtoCTFT[ x_, t_, w_, s_, trest__ ] := x

posttransform[x_] := ( x /. postrules )

postrules = {
	derivative[x_, w_, 1] :>
		D[ posttransform[x], w ],
	derivative[x_, w_, k_Integer] :>
		D[ posttransform[x], {w, k} ],
	CTFTtoL[x_] :>
		Transform[ x, -Infinity, Infinity ],
	CTFTtoL[x_, w_, s_] :>
		Transform[ substitute[x, w, s / I], -Infinity, Infinity ],
	myFourier[ LTransData[x_, lrest__], rest__ ] :>
		x,
	substitute[x_, w_, wnew_] :>
		( posttransform[x] /. w -> wnew ),
	x_ :> x
}

transform[myFourier[x_, rest__]] := Replace[myFourier[x, rest], CTFTRules]
transform[x_] := x



(*  B E G I N     C T F T     R U L E     B A S E  *)

CTFTRules = {

(*  Clean up any calls to the Laplace transform rule base  *)
myFourier[ LTransData[x_, rest__], t_, w_, s_, oplist_, state_ ] :> x,


(*  A.  Transform pairs			*)
(*	 1.  zero function		*)
myFourier[ 0, t_, w_, s_, oplist_, state_ ] :> 0,

(*	 2.  step function		*)
myFourier[ CStep[t_], t_, w_, s_, oplist_, state_ ] :>
	-I / w  +  Pi Delta[w],

(*	 3.  impulse function		*)
myFourier[ Delta[t_ + b_.], t_, w_, s_, oplist_, state_ ] :>
	Exp[I b w] /;
	FreeQ[b, t],

(*	 4.  constant function		*)
myFourier[ c_, t_, w_, s_, oplist_, state_ ] :> 2 Pi c Delta[w] /;
	FreeQ[c, t],

(*	 5.  two-sided cosine		*)
myFourier[ Cos[w0_. t_ + b_.], t_, w_, s_, oplist_, state_ ] :>
	Exp[I w b / w0] Pi ( Delta[w - w0] + Delta[w + w0] ) /;
	FreeQ[{b, w0}, t],

(*	 6.  two-sided sine		*)
myFourier[ Sin[w0_. t_ + b_.], t_, w_, s_, oplist_, state_ ] :>
	Exp[I w b / w0] Pi / I ( Delta[w - w0] - Delta[w + w0] ) /;
	FreeQ[{b, w0}, t],

(*	 7.  two-sided sinc		*)
myFourier[ Sin[a_. t_] / t_, t_, w_, s_, oplist_, state_ ] :> 
	Pi ( CStep[ w + a ] - CStep[ w - a ] ) /;
	FreeQ[a, t],

myFourier[ Sinc[a_. t_ + b_.], t_, w_, s_, oplist_, state_ ] :> 
	Exp[I w b / a] Pi / a ( CStep[ w + a ] - CStep[ w - a ] ) /;
	FreeQ[{a,b}, t],

myFourier[ (Sinc[a_. t_ + b_.])^2, t_, w_, s_, oplist_, state_ ] :> 
	Exp[I w b / a] (Pi / a)^2 ( Unit[-2][w + 2 a] - 2 Unit[-2][w] + Unit[-2][w - 2 a] ) /;
	FreeQ[{a,b}, t],

(*	 8.  pulse forms		*)
myFourier[ CStep[t_ + a_] - CStep[t_ - a_] + x_., t_, w_,
		s_, oplist_, state_ ] :>
	2 a Sinc[a w] + myFourier[x, t, w, s, oplist, state] /;
	FreeQ[a, t],

myFourier[ x_. + CStep[t_ + A_.] - CStep[t_ + B_.], t_, w_,
		s_, oplist_, state_ ] :>
	myFourier[CPulse[Abs[B - A], t - Min[-A, -B]], t, w, s, oplist, state] +
	  myFourier[x, t, w, s, oplist, state] /;
	FreeQ[{A,B}, t],

myFourier[ CPulse[T_, t_ + a_.], t_, w_, s_, oplist_, state_ ] :>
	T Exp[-I w T / 2] Exp[I a w] Sinc[w T / 2] /;
	FreeQ[{a, T}, t],

(*	 9.  two-sided Gaussian		*)
myFourier[ Exp[- a_. t_^2], t_, w_, s_, oplist_, state_ ] :>
	Sqrt[Pi] Exp[- w^2 / (4 a)] / Sqrt[a] /;
	FreeQ[a, t],

(*	10.  cosine x over x		*)
myFourier[ Cos[a_. t_] / t_, t_, w_, s_, oplist_, state_ ] :>
	Abs[a] I ( CStep[-Abs[a] - w] - CStep[w - Abs[a]] ) /;
	FreeQ[a, t],

(*	11.  other two-sided forms	*)
myFourier[ Exp[-Abs[a_. t_]] Sinc[a_. t_], t_, w_, s_, oplist_, state_ ] :>
	ArcTan[ 2 a^2 / w^2 ] / a /;
	FreeQ[a, t],

myFourier[ Sign[t_], t_, w_, s_, oplist_, state_ ] :> - I 2 / w,

myFourier[ 1 / t_, t_, w_, s_, oplist_, state_ ] :> -I Sign[w] / 2,

myFourier[ BesselJ[0, a_. t_], t_, w_, s_, oplist_, state_ ] :>
	CPulse[2 a, w + a] / ( Pi Sqrt[a^2 - w^2] ) /;
	FreeQ[a, t],


(*  B.  Properties  *)
(*	1.  homogeneity		*)
myFourier[ a_ b_, t_, w_, s_, oplist_, state_ ] :> 
	a myFourier[ b, t, w, s, oplist, state ] /; FreeQ[a, t],

(*	2.  additivity		*)
myFourier[ a_ + b_, t_, w_, s_, oplist_, state_ ] :> 
	myFourier[ a, t, w, s, oplist, state ] +
	myFourier[ b, t, w, s, oplist, state ],

(*	3.  one-sided transform	*)
myFourier[ x_. CStep[t_ + t0_], t_, w_, s_, oplist_, state_ ] :>
	Exp[- I w t0] myFourier[ ( x /. t -> (t - t0)) CStep[t],
				 t, w, s, oplist, state ] /;
	FreeQ[t0, t],

(*	4.  derivative		*)
myFourier[ Derivative[k_][x_][t_], t_, w_, s_, oplist_, state_ ] :>
	(I w)^k myFourier[ x[t], t, w, s, oplist, state ] /;
	FreeQ[k, t],

(*	5.  integration		*)
myFourier[ Integrate[x_, {tau_, -Infinity, t_}], t_, w_,
		s_, oplist_, state_ ] :>
	Block [	{trans},
		trans = myFourier[x, tau, w, s, oplist, state];
		trans ( 1/(I w) + Pi Delta[w] ) ],

(*	6.  multiply by t	*)
myFourier[ t_^a_. x_, t_, w_, s_, oplist_, state_ ] :> 
	I^a derivative[ myFourier[x, t, w, s, oplist, state], w, a ] /;
	FreeQ[a, t] && Implies[ NumberQ[a], IntegerQ[a] && a > 0 ],

(*	7.  exponential raised to an imaginary power times a function  *)
myFourier[ x_. Exp[ Complex[0, w0_] w1_. t_ + rest_. ], t_, w_,
		s_, oplist_, state_ ] :>
	substitute[myFourier[x Exp[rest], t, w, s, oplist, state], w, w0 w1] /;
	FreeQ[{w0,w1}, t],

(*	8.  pulse train		*)
myFourier[ Summation[n_, -Infinity, Infinity, 1][Delta[t_ + a_. + n_ T_.]],
		t_, w_, s_, oplist_, state_ ] :> 
	2 Pi Exp[I a w] *
	  Summation[n, -Infinity, Infinity, 1][Delta[w + 2 Pi n / T]] / T /;
	FreeQ[{a, n, T}, t],

(* 	9.  convolution		*)
myFourier[ CConvolve[t_][x_, rest__], t_, w_, s_, oplist_, state_ ] :> 
	Apply[ Times,
	       Map[ myFourier[#1, t, w, s, oplist, state]&,
		    { x, rest } ] ],


(*  C.	Strategies		*)
(*	1.  expand expression	*)
myFourier[ x_, t_, w_, s_, oplist_, state_ ] :>
	myFourier [ Expand[x], t, w, s, oplist,
		    SetStateField[state, expandfield, False] ] /;
	GetStateField[state, expandfield],

(*	2.  function times a window	*)
myFourier[ x_ CPulse[l_, t_ + a_.], t_, w_, s_, oplist_, state_ ] :>
	Integrate[ x Exp[- I w t], {t, -a, -a + l} ] /;
	FreeQ[{a,l}, t],

(*	3.  take the Laplace transform and adjust the result  *)
myFourier[ x_, t_, w_, s_, oplist_, state_ ] :> 
	Block [	{ltrans, newstate},
		ltrans = LaPlace[x, t, s, { Dialogue -> False } ~Join~ oplist];
		If [ InformUserQ[Replace[Dialogue, oplist]],
		     Print[ "( after taking the Laplace transform of " ];
		     Print[ "  ", x ];
		     Print[ "  which yields" ];
		     Print[ "  ", ltrans ];
		     Print[ "  and adjusting the result )" ] ];
		newstate = SetStateField[state, laplacefield, False];
		If [ SameQ[Head[ltrans], LTransData],
		     ( TheFunction[ltrans] /. s -> I w ),
		     MapAll[ LtoCTFT[#1, t, w, s, oplist, newstate]&,
			     ltrans ] ] ] /;
	GetStateField[state, laplacefield]
}


(*  E N D     F O R W A R D     F O U R I E R     T R A N S F O R M  *)




(*  B E G I N     I N V E R S E     F O U R I E R     T R A N S F O R M  *)

(*  Options  *)
InvCTFTransform/: Options[InvCTFTransform] :=
	{ Dialogue -> False, Terms -> False } ~Join~ Options[InvLaPlace]

(*  InvCTFTransform  *)
InvCTFTransform[ CTFTData[x_, FVariables[w_]] ] := InvCTFTransform[x, w]

InvCTFTransform[ CTFTData[x_, FVariables[w_]], rest__ ] :=
	InvCTFTransform[x, rest]

InvCTFTransform[ x_, w_ ] :=
	InvCTFTransform[ x, w, DummyVariables[Length[w], Global`t] ]

InvCTFTransform[ f_, w_, t_, options___ ] :=
	Block[ {op, s, svars, trans, vars},
		op = ToList[options] ~Join~ Options[InvCTFTransform];

		vars = Which [ Length[w] > Length[t],
				 DummyVariables[Length[w], Global`t],
			       Length[w] < Length[t],
				 Take[t, Length[w]],
			       True,
				 t ];

		svars = DummyVariables[Length[w], s];

		trans = If [ Length[w] > 1, 
			     MultiDInvCTFT[f, w, vars, svars, op],
			     MyInverseCTFT[f, w, vars, svars, op] ];

		If [ InformUserQ[Replace[Dialogue, op]],
		     Scan[invexplain, trans, Infinity] ];

		FourierSimplify[trans] ]

(*  MultiDInvCTFT  *)
MultiDInvCTFT[ f1_, w_, t_, s_, op_ ]:= 
	Block [	{i, func = f1, length},
		length = Length[t];
		For [ i = 1, i <= length, i++,
		      func = MyInverseCTFT[func, w[[i]], t[[i]], s[[i]], op] ];
		func ]

invexplain[ myinvFourier[x_, w_, rest__]] :=
	Message[ Transform::incomplete, "inverse Fourier transform", x, w ]

(*  driver for 1D inverse CTFT rules  *)
MyInverseCTFT[ f_, w_, t_, s_, op_ ] :=
	MyInverseCTFT[ f, w, t, s, op, { True, True } ]

MyInverseCTFT[ f_, w_, t_, s_, oplist_, state_ ] :=
	Block [	{newexpr = myinvFourier[f, w, t, s, InvFixUp[oplist], state],
		 oldexpr = Null, trace },
		trace = SameQ[ Replace[Dialogue, oplist], All];
		While [ ! SameQ[newexpr, oldexpr],
			If [ trace, Print[newexpr]; Print["which becomes"] ];
			oldexpr = newexpr;
			newexpr = MapAll[invtransform, oldexpr] ];

		oldexpr = TransformFixUp[ oldexpr, w, t, oplist, myinvFourier,
					  False, InvCTFTransform, Null, Null ];

		newexpr = postinvtransform[oldexpr];
		While [ ! SameQ[newexpr, oldexpr],
			If [ trace, Print[newexpr]; Print["which becomes"] ];
			oldexpr = newexpr;
			newexpr = postinvtransform[oldexpr] ];
		If [ trace, Print[newexpr] ];
		newexpr ]


postinvtransform[x_] := ( x /. postinvrules )

postinvrules = {
	derivative[x_, t_, 1] :>
		D[ postinvtransform[x], t ],
	derivative[x_, t_, k_] :>
		D[ postinvtransform[x], {t, k} ],
	substitute[x_, t_, tnew_] :>
		( postinvtransform[x] /. t -> tnew ),
	x_ :> x
}


invtransform[myinvFourier[x_, rest__]] :=
	Replace[myinvFourier[x, rest], InvCTFTRules]
invtransform[x_] := x


InvFixUp[ op_ ] := { Apart -> Replace[Apart, op],
		     Dialogue -> Replace[Dialogue, op],
		     Simplify -> Replace[Simplify, op],
		     Terms -> Replace[Terms, op] }


(*  The inverse CTFT rules *)


InvCTFTRules = {

(*  Clean up expression returned by inverse Laplace transform	*)
myinvFourier[ SignalProcessing`Analog`InvLaPlace`Private`myinvlaplace[ x_, s_, rest__], w_, t_, s_, op_, state_ ] :>
	myinvFourier[ x /. s -> I w, w, t, s, op, state ],


(*  A.	Lookup rules		*)
(*	 1.  constant functions	*)
myinvFourier[ 0, w_, t_, s_, op_, state_ ] :> 0,
myinvFourier[ 1, w_, t_, s_, op_, state_ ] :> Delta[t],

(*	 2.  two delta functions	*)
myinvFourier[ c_. Delta[w_ + a_] + d_. Delta[w_ + b_], w_, t_,
		s_, op_, state_ ] :> 
	Limit[d, w -> -b] I Sin[b t] / Pi /;
	( a == - b ) && ( Abs[c] == 1 ) && ( c == - d ) && FreeQ[a, w],

myinvFourier[ c_. Delta[w_ + a_] + c_. Delta[w_ + b_], w_, t_,
		s_, op_, state_ ] :>
	c Cos[b t] / Pi /;
	( a == - b ) && FreeQ[{a,c}, w],

(*	 3.  one delta function	*)
myinvFourier[ c_. Delta[w_ + a_.], w_, t_, s_, op_, state_ ] :>
	Limit[c, w -> -a] Exp[- I a t] / ( 2 Pi )  /;
	FreeQ[a, w],

(*	 4.  pulse train		*)
myinvFourier[ Summation[k_Symbol, -Infinity, Infinity, 1]
			[Delta[w_ + k_ T_.]], w_, t_, s_, op_, state_ ] :> 
	Summation[k, -Infinity, Infinity, 1]
		 [Delta[t + 2 Pi k / T]] / T /;
	FreeQ[{k,T}, w],

(*	 5.  step in time	*)
myinvFourier[ Pi Delta[w_] - I / w_, w_, t_, s_, op_, state_ ] :>
	CStep[t],

(*	 6.  pulse functions	*)
myinvFourier[ ( CStep[w_ + a_] - CStep[w_ + b_] ) c_., w_, t_,
		s_, op_, state_ ] :>
	c Sinc[a t] / Pi /;
	FreeQ[{a,b,c}, w] && a == -b,

myinvFourier[ x_. + CStep[w_ + a_] + b_ CStep[w_ + c_], w_, t_,
		s_, op_, state_ ] :>
	Sin[a t] / ( Pi t ) + myinvFourier[x, w, t, s, op, state] /;
	( a == - c ) && ( b == - 1 ) && FreeQ[a, w],

(*	 7.  two-sided Gaussian		*)
myinvFourier[ Exp[- b_. w_^2], w_, t_, s_, op_, state_ ] :>
	Exp[- t^2 / (2 b)] / Sqrt[2 Pi b] /;
	FreeQ[b, w],

(*	 8.  cosine x over x		*)
myinvFourier[ CStep[-w_ - a_] - CStep[w_ - a_], w_, t_, s_, op_, state_ ] :>
	Cos[a t] / (a I t) /;
	FreeQ[a, w],

(*	 9.  cosine x over x		*)
myinvFourier[ Sinc[a_. w_ + b_.], w_, t_, s_, op_, state_ ] :>
	Exp[- b t / a] CPulse[2 a, t + a] / (2 a) /;
	FreeQ[{a, b}, w],

(*	10.  other two-sided forms	*)
myinvFourier[ ArcTan[b_ / w_^2], w_, t_, s_, op_, state_ ] :>
	Sqrt[b/2] Exp[-Abs[Sqrt[b/2] t]] Sinc[Sqrt[b/2] t] /;
	FreeQ[b, w],

myinvFourier[ Sign[w_], w_, t_, s_, oplist_, state_ ] :> I 2 / t,

myinvFourier[ 1 / w_, w_, t_, s_, oplist_, state_ ] :> I Sign[t] / 2,

myinvFourier[ CPulse[2 a_, w_ + a_.] / Sqrt[b_ - w_^2], w_, t_,
		s_, oplist_, state_ ] :>
	Pi BesselJ[0, a t] /;
	FreeQ[a, t] && ( b == a^2 ),


(*  B.	Properties				*)
(*	1.  Homogeneity (pick off constants)	*)
myinvFourier[ a_ x_, w_, t_, s_, op_, state_ ] :> 
	a myinvFourier[x, w, t, s, op, state] /;
	FreeQ[a, w],

(*	2.  Additivity				*)
myinvFourier[ a_ + b_, w_, t_, s_, op_, state_ ] :>
	myinvFourier[ a, w, t, s, op, state ] +
	myinvFourier[ b, w, t, s, op, state ],

(*	3.  Multiplication by I w F(w) is d/dt of f(t).	*)
myinvFourier[ w_^k_. x_., w_, t_, s_, op_, state_ ] :>
	(-I)^k derivative[ myinvFourier[x, w, t, s, op, state], t, k ] /;
	FreeQ[k, w] && Implies[ NumberQ[k], IntegerQ[k] && k > 0 ],

(*	4.  Multiplication by Exp[I a w] is shift by a in time.	*)
myinvFourier[ x_. Exp[Complex[0, t0_] t1_. w_ + rest_.], w_, t_,
		s_, op_, state_ ] :>
	substitute[myinvFourier[x Exp[rest], w, t, s, op, state], t, t + t0 t1] /;
	FreeQ[{t0,t1}, w],

(*	5.  Integration in time			*)
myinvFourier[ ( 1 / ( I w_ ) + Pi Delta[w_] ) x_, w_, t_, s_, op_, state_ ] :> 
	Block [	{tau = Unique["tau"]},
		integrate[ myinvFourier[x, w, tau, s, op, state],
			   {tau, -Infinity, t} ] ],

(* 	6.  Convolution				*)
myinvFourier[ CConvolve[w_][x_, rest__], w_, t_, s_, op_, state_ ] :> 
	Apply[ Times,
	       Map[ myinvFourier[#1, w, t, s, oplist, state]&,
		    { x, rest } ] ],


(*  C.	Strategies				*)
(*	1.  Expand expression			*)
myinvFourier[ x_, w_, t_, s_, op_, state_ ] :> 
	myinvFourier [	Expand[x], w, t, s, op,
			SetStateField[state, expandfield, False] ] /;
	GetStateField[ state, expandfield ],

(*	2.  function times a window	*)
myinvFourier[ x_ CPulse[l_, w_ + a_.], w_, t_, s_, oplist_, state_ ] :>
	Integrate[ x Exp[I w t], {w, -a, -a + l} ] /;
	FreeQ[{a,l}, w],

(*	3.  Try an inverse Laplace transform if all else fails	*)
myinvFourier[ x_, w_, t_, s_, op_, state_ ] :>
	Block [	{invltrans, newstate},
		invltrans = InvLaPlace[ x /. w -> s / I, s, t,
					{ Dialogue -> False } ~Join~ op ];
		If [ InformUserQ[Replace[Dialogue, op]],
		     Print[ "( after taking the inverse Laplace transform of" ];
		     Print[ "  ", x ];
		     Print[ "  which yields" ];
		     Print[ "  ", invltrans ];
		     Print[ "  and adjusting the result )" ] ];
		newstate = SetStateField[state, laplacefield, False];
		If [ FreeQ[invltrans, s],
		     invltrans,
		     myinvFourier[ x, w, t, s, op, newstate ] ] ] /;
	GetStateField[state, laplacefield]

}

(*  E N D     I N V E R S E     F O U R I E R     T R A N S F O R M  *)


(*  E N D     P A C K A G E  *)

End[]
EndPackage[]

If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ];


(*  H E L P     I N F O R M A T I O N  *)

Combine[ SPfunctions, { CTFTransform, InvCTFTransform } ]
Protect[ CTFTransform, InvCTFTransform ]


(*  E N D I N G     M E S S A G E  *)

Print[ "The continuous Fourier transform (CTFT) functions are loaded." ]
Null
