(*  :Title:	One Dimensional Filters*)

(*  :Authors:	Wally McClure, Brian Evans, James McClellan  *)

(*  :Summary:	Design of one-dimensional analog filters  *)

(*  :Context:	SignalProcessing`Analog`FilterDesign`  *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1989-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*
    :History:	date begun -- December 24, 1989
		made into package -- May 2, 1990
		cleaned up for release -- October 29, 1990
		lowpass filters keep tract of root locations -- March, 1991
*)

(*  :Keywords:	filter design, prototype *)

(*
    :Source:	{Theory and Application of Digital Signal Processing}
		  by Lawrence Rabiner and Bernard Gold, 1975
		{Fundamentals of Digital Signal Processing}
		  by Lonnie Ludeman, 1986
		{Approximation Methods for Electronic Filter Design}
		  by Richard Daniels, 1974
		{Digital Filter Design} by T. W. Parks and C. S. Burrus
*)

(*
    :Warning:	The EllipticK function in Mathematica is slightly different
		from the function K used in the filter design literature:

		K(k) = EllipticK[k^2]

		In general, the modulus in all calls to Mathematica elliptic
		functions need to be squared to agree with the form of
		the same functions for filter design.
 *)

(*  :Mathematica Version:  1.2 or 2.0  *)

(*
    :Limitation:  Only works with Butterworth, Chebyshev I, Chebyshev II,
		  and elliptic filters.
*)

(*  :Functions:	AnalogFilter DesignFilter FilterTransform RationalChebyshev  *)

(*
    :Discussion:  The usage information for DesignFilter is in the
		  "FilterSupport.m", context SignalProcessing`Support`.

		  In analog filter design, it is common to design an
		  analog lowpass filter prototype and then transform
		  it into a highpass, bandpass, or bandstop filter
		  (or even a lowpass filter with a different cutoff
		  frequency).  This is the approach of this file.
 *)


If [ TrueQ[ $VersionNumber >= 2.0 ],
     Off[ General::spell ];
     Off[ General::spell1 ] ];


(*  B E G I N     P A C K A G E  *)

BeginPackage[ "SignalProcessing`Analog`FilterDesign`",
	      "SignalProcessing`Support`FilterSupport`",
	      "SignalProcessing`Support`SupCode`" ]


If [ ! TrueQ[$VersionNumber >= 2.0],
     Elliptic::notloaded = "Elliptic functions are not properly loaded.";
     If [ SameQ[Head[EllipticF::usage], MessageName],
	  Message[Elliptic::notloaded] ] ]


(*  U S A G E     I N F O R M A T I O N  *)

AnalogFilter::usage =
	"AnalogFilter[class, time-variable, order, epsilon, A, \
	transition-ratio, cutoff-frequency] will return an object with \
	head CIIR that represents a one-dimensional lowpass analog filter. \
	Here, class is one of Bessel, Butterworth, ChebyshevI, ChebyshevII, \
	Elliptic. \
	The last four arguments are filter parameters. \
	The value of the cutoff-frequency defaults to 1. \
	See also FilterParameters and FilterOrder."

BesselPolynomial::usage =
	"BesselPolynomial[order, s] results from truncating a continued \
	fraction expansion representation of the unit delay exp(-s). \
	Bessel polynomials satisfy the recursion \
	B[n, s] = (2n - 1) B[n-1, s] + s^2 B[n-2, s], with \
	B[0, s] = 1 and B[1, s] = s + 1."

FilterTransform::usage =
	"FilterTransform[filter-domain, lowpassfilter, s, new-filter-type, \
	wc, wcnew, wcnewupper] transforms a lowpass filter with \
	cutoff frequency wc into either a lowpass or highpass filter \
	with a cutoff of wcnew or a bandpass or bandstop filter with \
	lower/upper cutoff frequencies of wcnew and wcnewupper. \
	Here, the new-filter-type can be Lowpass, Highpass, Bandpass, or \
	Bandstop."

RationalChebyshev::usage =
	"RationalChebyshev[n, x, xL, L] returns a rational polynomial which \
	is the nth order rational Chebyshev function evaluated at x \
	for parameters xL and L."

(*  E N D     U S A G E     I N F O R M A T I O N  *)


Begin[ "`Private`" ]


(*  M E S S A G E S  *)

ellipticzero::infzero = "Zero at infinity encountered."
findmodulus::notconv = "Algorithm to adjust modulus did not converge."

DesignFilter::narrower =
	"One of the transition bandwidths will be narrower than specified."


(*  A N A L O G     F I L T E R     D E S I G N  *)

(*  BesselPolynomial, basis of Bessel filter design  *)

BesselVar = t
BesselList = { 1, 1 + BesselVar }
BesselListLength = 2

newbesselpoly[n_Integer] :=
	Block [ {newpoly},
		newpoly = Simplify[ (2 n - 1) BesselList[[n]] +
				    BesselVar^2 BesselList[[n-1]] ];
		AppendTo[BesselList, newpoly];
		++BesselListLength;
		newpoly ]

besselpoly[n_Integer] :=
	Block [	{i},
		For [ i = BesselListLength, i <= n, i++, newbesselpoly[i] ];
		BesselList[[n + 1]] ]

BesselPolynomial[n_Integer, s_] :=
	( besselpoly[n] /. BesselVar -> s ) /; n >= 0


(*  RationalChebyshev function Rn(x, L), basis of Elliptic filter response  *)

rationalchebyshevpole[K_, i_, n_?OddQ, omegar_, modulus_] :=
	omegar / JacobiSN[2 i K / n, modulus^2]
rationalchebyshevpole[K_, i_, n_?EvenQ, omegar_, modulus_] :=
	omegar / JacobiSN[(2 i - 1) K / n, modulus^2]

rationalchebyshevzero[K_, i_, n_?OddQ, modulus_] :=
	JacobiSN[2 i K / n, modulus^2]
rationalchebyshevzero[K_, i_, n_?EvenQ, modulus_] :=
	JacobiSN[(2 i - 1) K / n, modulus^2]

Options[RationalChebyshev] := { Factor -> False }

RationalChebyshev[n_, x_, xL_, L_, options___] :=
	Block [	{chebyA, chebyB, expchebyA, expchebyB, i, K, op, polelist,
		 polelistsquared, zerolist, zerolistsquared},

		halfn = If [ OddQ[n], (n - 1)/2, n/2 ];
		K = N[EllipticK[1/xL^2]];

		zerolist = N[ Table[ rationalchebyshevzero[K, i, n, 1/xL],
				     {i, 1, halfn} ] ];
		polelist = N[ xL / zerolist ];
		zerolistsquared = zerolist^2;
		polelistsquared = polelist^2;

		chebyA = Product[ (x^2 - zerolistsquared[[i]]), {i, 1, halfn} ];
		chebyA *= If [ OddQ[n],
			     x N[Sqrt[L/xL] Apply[Times, polelistsquared/xL^2]],
			     N[Sqrt[L] Apply[Times, polelistsquared / xL]] ];
		expchebyA = Chop[N[Expand[chebyA]]];

		chebyB = Product[ (x^2 - polelistsquared[[i]]), {i, 1, halfn} ];
		expchebyB = Chop[N[Expand[chebyB]]];

		op = ToList[options] ~Join~ Options[RationalChebyshev];
		If [ TrueQ[Replace[Factor, op]],
		     chebyA / chebyB,
		     expchebyA / expchebyB ] ]

(*  findmodulus  *)

findmodulus[u_] :=
	Block [	{a, b, c, d, j, q, tolerance},
		d = q = N[Exp[- Pi u]];
		a = b = c = 1.0;
		tolerance = N[ 10^-7 ];
		For [ j = 1, j <= 15, j++,
		      a += 2 c d;
		      c *= d d;
		      b += c;
		      d = d q;
		      If [ c < tolerance, Break[] ] ];
		4 Sqrt[q] (b/a)^2 ]

(*  FilterTransferFunction  *)

FilterTransferFunction[a_ x_, t_, s_] :=
	a FilterTransferFunction[x, t, s] /;
	FreeQ[a, t]

FilterTransferFunction[CFIR[t_, h_, rest___][x_], t_, s_] :=
	FilterTransferFunction[CFIR[t, h, rest], t, s] *
	FilterTransferFunction[x, t, s]
FilterTransferFunction[CFIR[t_, h_, rest___], t_, s_] :=
	Block [ {k},
		h . Table[s^k, {k, 0, Length[h] - 1}] ]

FilterTransferFunction[CIIR[t_, a_, rest___], t_, s_] :=
	Block [ {k},
		1 / ( a . Table[s^k, {k, 0, Length[a] - 1}] ) ]
FilterTransferFunction[CIIR[t_, a_, rest___][x_], t_, s_] :=
	FilterTransferFunction[CIIR[t, a, rest], t, s] *
	FilterTransferFunction[x, t, s]

(*  GetPoles  *)

GetPoles[a_ x_, t_, s_] := a GetPoles[x, t, s] /; FreeQ[a, t]
GetPoles[CFIR[t_, h_, rest___][x_], t_, s_] := GetPoles[x, t, s]
GetPoles[CIIR[t_, a_, Roots -> r_][x_], t_, s_] := r
GetPoles[CIIR[t_, a_, Roots -> r_], t_, s_] := r

(*  GetZeroes  *)

GetZeroes[a_ x_, t_, s_] := a GetZeroes[x, t, s] /; FreeQ[a, t]
GetZeroes[CFIR[t_, h_, Roots -> r_][x_], t_, s_] := r
GetZeroes[CFIR[t_, h_, Roots -> r_], t_, s_] := r
GetZeroes[CIIR[t_, a_, rest___][x_], t_, s_] := GetZeroes[x, t, s]

(*  MyChop  *)

MyChop[x_] := Re[x]


(*  ToFilter  *)

ToFilter[expnumer_, expdenom_, s_, t_] :=
	CFIR[ t, MyChop[CoefficientList[expnumer, s]] ] / expdenom /;
	FreeQ[expdenom, s]

ToFilter[expnumer_, expdenom_, s_, t_] :=
	expnumer CIIR[ t, MyChop[CoefficientList[expdenom, s]] ] /;
	FreeQ[expnumer, s]

ToFilter[expnumer_, expdenom_, s_, t_] :=
	Block [	{fir, iir},
		fir = CFIR[ t, MyChop[CoefficientList[expnumer, s]] ];
		iir = CIIR[ t, MyChop[CoefficientList[expdenom, s]] ];
		fir [ iir ] ]

(*  Supporting Routines for lowpass analog IIR filter prototype design  *)
(*  For these routines, index j goes from 1 to n (filter order).	*)

butterworthpole[j_, n_] := Exp[I Pi (1/2 + (2 j - 1)/(2 n))]

chebyshevIpole[gamma_, j_, n_] :=
	(1/gamma - gamma) Sin[(2 j - 1) Pi / (2 n)] / 2 +
	I (gamma + 1/gamma) Cos[(2 j - 1) Pi / (2 n)] / 2

chebyshevIIpole[gamma_, j_, n_, omegar_] :=
	Block [	{alpha, beta, denom},
		alpha = (1/gamma - gamma) Sin[(2 j - 1) Pi/ (2 n)] / 2;
		beta = (1/gamma + gamma) Cos[(2 j - 1) Pi/ (2 n)] / 2;
		denom = alpha^2 + beta^2;
		( omegar alpha / denom + I omegar beta / denom ) ]

chebyshevIIzero[j_, n_, omegar_] :=		(* Infinity for j = (n+1)/2 *)
	I omegar / Cos[(2 j - 1) Pi / (2 n)]

	(* poles/zero of elliptic filter [Parks and Burrus, 193-194] *)
ellipticpole[K_, jv0_, j_, n_?EvenQ, k_] :=
	Block [	{i, pole},
		i = If [ OddQ[j], j, j-1 ];		(* i = 1, 3, 5, ... *)
		pole = I JacobiSN[jv0 + i K / n, k^2];
		If [ OddQ[j], pole, Conjugate[pole] ] ]
ellipticpole[K_, jv0_, 1, n_?OddQ, k_] :=
	I JacobiSN[jv0, k^2]
ellipticpole[K_, jv0_, j_, n_?OddQ, k_] :=
	Block [	{i, pole},
		i = If [ EvenQ[j], j, j-1 ];		(* i = 0, 2, 4, ... *)
		pole = I JacobiSN[jv0 + i K / n, k^2];
		If [ OddQ[j], pole, Conjugate[pole] ] ]

ellipticzero[K_, j_, n_?EvenQ, k_] :=
	Block [	{i},
		i = If [ OddQ[j], j, j-1 ];		(* i = 1, 3, 5, ... *)
		(-1)^j I / ( k JacobiSN[i K / n, k^2] ) ]
ellipticzero[K_, 1, n_?OddQ, k_] :=
	MyMessage[ ellipticzero::infzero, Infinity ]
ellipticzero[K_, j_, n_?OddQ, k_] :=
	Block [	{i},
		i = If [ EvenQ[j], j, j-1 ];		(* i = 0, 2, 4, ... *)
		(-1)^j I / ( k JacobiSN[i K / n, k^2] ) ]


(*  Actual Filter Design for Lowpass Prototypes  *)

(*  Butterworth filter [Rabiner & Gold, 227]  *)
Butterworth/: AnalogFilter[Butterworth, t_, n_, e_, A_, trans_, wc_:1] :=
	Block [	{denom, expdenom, i, iircoeffs, poles, s},
		poles = Table[wc butterworthpole[i, n], {i, 1, n}];
		denom = Product[ (s - poles[[i]]), {i, 1, n} ];
		expdenom = N[Expand[denom]];
		iircoeffs = CoefficientList[expdenom, s];
		wc^n CIIR[ t, MyChop[iircoeffs], Roots -> poles ] ]

(*  Bessel filter [Rabiner & Gold, 228-230] *)
Bessel/: AnalogFilter[Bessel, t_, n_, e_, A_, trans_, wc_:1] :=
	Block [	{d0, denom, expdenom, i, iircoeffs, omegac, poles, s},
		denom = Expand[BesselPolynomial[n, s]];
		d0 = Coefficient[denom, s, 0];		(* normalizing factor *)
		omegac = N[ d0^(1/n) ];		  (* omegac varies with order *)
		expdenom = denom /. s -> N[(omegac s / wc)];
		iircoeffs = CoefficientList[expdenom, s] / d0;
		poles = GetRootList[ expdenom, s ];
		CIIR[ t, MyChop[ iircoeffs ], Roots -> poles ] ]

(*  Chebyshev type I filter [Rabiner & Gold, 232] [Ludeman, 139]  *)
ChebyshevI/: AnalogFilter[ChebyshevI, t_, n_, e_, A_, trans_, wc_:1] :=
	Block [	{denom, expdenom, gamma, i, iircoeffs, norm, poles, s},
		gamma = N[ ((1 + Sqrt[1 + e^2]) / e)^(1/n) ];
		poles = Table[wc chebyshevIpole[gamma, i, n], {i, 1, n}];
		denom = Product[ (s - poles[[i]]), {i, 1, n} ];
		expdenom = N[Expand[denom]];
		norm = Coefficient[expdenom, s, 0];
		If [ EvenQ[n], norm /= N[Sqrt[1 + e^2]] ];
		iircoeffs = CoefficientList[expdenom, s];
		norm CIIR[ t, MyChop[iircoeffs], Roots -> poles ] ]

(*  Chebyshev type II filter [Rabiner & Gold, 232-234]  *)
ChebyshevII/: AnalogFilter[ChebyshevII, t_, n_, e_, A_, trans_, wc_:1] :=
	Block [	{coeffs, denom, expdenom, expnumer, fir, gamma, halfn,
		 i, iir, norm, numer, poles, s, wr = wc/trans, zeroes},
		gamma = N[ (A + Sqrt[A^2 - 1])^(1/n) ];

		(*  denominator section of filter transfer function *)
		poles = Table[chebyshevIIpole[gamma, i, n, wr], {i, 1, n}];
		denom = Product[ (s - poles[[i]]), {i, 1, n} ];

		(*  all-pole section of filter  *)
		expdenom = N[Expand[denom]];
		coeffs = CoefficientList[expdenom, s];
		iir = CIIR[ t, MyChop[coeffs], Roots -> poles ];

		(*  numerator section of filter transfer function *)
		If [ OddQ[n],

		     halfn = (n + 1) / 2;
		     zeroes = Join[ Table[ chebyshevIIzero[i, n, wr],
					   {i, 1, halfn - 1} ],
				    Table[ chebyshevIIzero[i, n, wr],
					   {i, halfn + 1, n} ] ];
		     numer = Product[ (s - zeroes[[i]]), {i, 1, n - 1} ],

		     zeroes = Table[ chebyshevIIzero[i, n, wr], {i, 1, n} ];
		     numer = Product[ (s - zeroes[[i]]), {i, 1, n} ] ];

		(*  all-zero section of filter  *)
		expnumer = N[Expand[numer]];
		coeffs = CoefficientList[expnumer, s];
		fir = CFIR[ t, MyChop[coeffs], Roots -> zeroes ];

		(*  calculate the normalizing coefficient  *)
		norm = Chop[ Coefficient[expdenom, s, 0] /
		             Coefficient[expnumer, s, 0] ];

		norm iir [ fir ] ]

(*  Elliptic filter [Daniels, 56-57, 73] [Parks and Burrus, 193-194]	*)
(*  Has n poles and n zeroes; for odd n, one zero is at infinity.	*)
(*  Normally, k1 = nu = N[e / Sqrt[A^2 - 1]], but we adjust k1.		*)
Elliptic/: AnalogFilter[Elliptic, t_, n_, e_, A_, trans_, wc_:1] :=
	Block [	{coeffs, denom, expdenom, expnumer, fir, gamma, halfn,
		 i, iir, K, KC, K1, k1, jv0, jv01, jv02, norm, numer,
		 nzstart, poles, s, zeroes},

		K = N[EllipticK[trans^2]];		   (* see :Warning: *)
		KC = N[EllipticK[1 - trans^2]];

		k1 = N[e / Sqrt[A^2 - 1]];		(* org. value of nu *)
		k1 = findmodulus[ n KC / K ];	    (* adjusted value of nu *)
		K1 = N[EllipticK[k1^2]];

		jv0 = I N[ K InverseJacobiSC[1/e, 1 - k1^2] / ( n K1 ) ];

		(*  denominator section of filter transfer function *)
		poles = Table[wc ellipticpole[K, jv0, i, n, trans], {i, 1, n}];
		denom = Product[ (s - poles[[i]]), {i, 1, n} ];
		expdenom = N[Expand[denom]];
		coeffs = CoefficientList[expdenom, s];
		iir = CIIR[ t, MyChop[coeffs], Roots -> poles ];

		(*  numerator section of filter transfer function *)
		nzstart = If [ OddQ[n], 2, 1 ];
		zeroes = Table[ wc ellipticzero[K, i, n, trans],
				{i, nzstart, n} ];
		numer = Product[ (s - zeroes[[i]]), {i, 1, n - nzstart + 1} ];
		expnumer = N[Expand[numer]];
		coeffs = CoefficientList[expnumer, s];
		fir = CFIR[ t, MyChop[coeffs], Roots -> zeroes ];

		(*  calculate the normalizing coefficient  *)
		norm = Chop[ Coefficient[expdenom, s, 0] /
		             Coefficient[expnumer, s, 0] ];
		If [ EvenQ[n], norm /= N[Sqrt[1 + e^2]] ];

		norm iir [ fir ] ]


(*  DesignFilter  *)
Analog/: DesignFilter[ Analog, name_, Lowpass, parameters__ ] :=
	 DesignLPFilter[name, parameters]
Analog/: DesignFilter[ Analog, name_, Highpass, parameters__ ] :=
	 DesignHPFilter[name, parameters]
Analog/: DesignFilter[ Analog, name_, Bandpass, parameters__ ] :=
	 DesignBPFilter[name, parameters]
Analog/: DesignFilter[ Analog, name_, Bandstop, parameters__ ] :=
	 DesignBSFilter[name, parameters]

(*  findcutoff -- determine cutoff frequency for filter given	*)
(*		  omegapassband and omegastopband; this is the	*)
(*		  amount to shift a prototype with a cutoff	*)
(*		  frequency of 1 rad/sec; keep in mind that d1  *)
(*		  and d2 are in standard units not dB's		*)
findcutoff[Bessel, w1_, w2_, d1_, d2_, n_] := w1
findcutoff[Butterworth, w1_, w2_, d1_, d2_, n_] :=
	Block [	{omegacutoff1, omegacutoff2},
		omegacutoff1 = N[w1 / (-1 + 1/(1 - d1)^2)^(1/(2 n))];
		omegacutoff2 = N[w2 / (-1 + 1/(d2)^2)^(1/(2 n))];
		( omegacutoff1 + omegacutoff2 ) / 2 ]
findcutoff[ChebyshevI,  w1_, w2_, rest__] := w1
findcutoff[ChebyshevII, w1_, w2_, rest__] := w1
findcutoff[Elliptic, w1_, w2_, rest__] := w1

(* Design a lowpass filter *)
DesignLPFilter[name_, t_, dp_, ds_, w1_, w2_, rest___ ] :=
	Block [	{A, ATT, epsilon, Eripple, nu, order, trans},
		trans = w1/w2;
		FilterParameters[dp, ds, epsilon, A, Eripple, ATT, nu];
		order = FilterOrder[name, trans, nu];
		AnalogFilter[name, t, order, epsilon, A, trans,
			     findcutoff[name, w1, w2, dp, ds, order] ] ]

(* Design a highpass filter [Parks and Burrus, 201-202] *)
DesignHPFilter[ name_, t_, dp_, ds_, w1_, w2_, rest___ ] :=
	Block [	{expdenom, expnumer, fir, hptrans, lowpass, lptrans, n, s},
		lowpass = DesignLPFilter[name, t, dp, ds, 1/w2, 1/w1];
		lptrans = FilterTransferFunction[lowpass, t, s];
		n = Exponent[ Denominator[lptrans], s ];
		hptrans = lptrans /. s -> 1/s;
		expnumer = Expand[s^n Numerator[hptrans]];
		expdenom = Expand[s^n Denominator[hptrans]];
		ToFilter[ expnumer, expdenom, s, t ] ]

(*  Design a bandpass filter --  twice the order of the lowpass prototype   *)
DesignBPFilter[ name_, t_, dp_, ds_, w1_, w2_, w3_, w4_ ] :=
	Block [	{bptrans, expdenom, expnumer, lowpass,
		 lptrans, n, s, w0, wp, ws},
		w0 = Sqrt[ w2 w3 ];
		wp = N[ ( w3^2 - w0^2 ) / w3 ];
		ws = N[ Min[ (w4^2 - w0^2 ) / w4, (w0^2 - w1^2 ) / w1 ] ];
		If [ SameQ[name, Elliptic] && ! SameQ[w1 w4, w2 w3],
		     Message[ DesignFilter::narrower ] ];
		lowpass = DesignLPFilter[name, t, dp, ds, wp, ws];
		lptrans = FilterTransferFunction[lowpass, t, s];
		n = Exponent[ Denominator[lptrans], s ];
		bptrans = lptrans /. (s -> (s^2 + w0^2) / s); (* double order *)
		expnumer = Expand[s^n Numerator[bptrans]];
		expdenom = Expand[s^n Denominator[bptrans]];
		ToFilter[ expnumer, expdenom, s, t ] ]

(*  Design a bandstop filter --  twice the order of lowpass prototype  *)
DesignBSFilter[ name_, t_, dp_, ds_, w1_, w2_, w3_, w4_ ] :=
	Block [	{bstrans, denom, lowpass, lptrans, n, numer, s,
		 w0, wp, ws, wsdenom1, wsdenom2},

		w0 = Sqrt[ w1 w4 ];
		wp = N[ w1 / ( w0^2 - w1^2 ) ];

		wsdenom1 = w0^2 - w2^2;			(* ws formula could *)
		wsdenom2 = w3^2 - w0^2;			(* have 0 in denom  *)
		ws = Which [ wsdenom1 == 0,
			       w3 / wsdenom2,
			     wsdenom2 == 0,
			       w2 / wsdenom1,
			     True,
			       Min[ w2 / wsdenom1, w3 / wsdenom2 ] ] // N;

		If [ SameQ[name, Elliptic] && ! SameQ[w1 w4, w2 w3],
		     Message[ DesignFilter::narrower ] ];
		lowpass = DesignLPFilter[name, t, dp, ds, wp, ws];
		lptrans = FilterTransferFunction[lowpass, t, s];
		n = Exponent[ Denominator[lptrans], s ];
		bstrans = lptrans /. (s -> (s / (s^2 + w0^2)));
		numer = Distribute[(s^2 + w0^2)^n Numerator[bstrans]];
		denom = Distribute[(s^2 + w0^2)^n Denominator[bstrans]];
		ToFilter[ Expand[numer], Expand[denom], s, t ] ]


(*  E N D     P A C K A G E  *)

End[]
EndPackage[]

If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ];


(*  H E L P     I N F O R M A T I O N  *)

Combine[ SPfunction,
	{ AnalogFilter, DesignFilter, FilterTransform, RationalChebyshev } ]
Protec[ AnalogFilter, DesignFilter, FilterTransform, RationalChebyshev ]


(*  E N D I N G     M E S S A G E  *)

Print[ "The one-dimensional analog filter design objects are loaded." ]
Null
